package com.hummer.im._internals.bridge.helper;

import com.hummer.im.BuildConfig;
import com.hummer.im.Error;
import com.hummer.im.ErrorEnum;
import com.hummer.im.HMR;
import com.hummer.im._internals.bridge.marshall.Marshallable;
import com.hummer.im._internals.log.HummerLog;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im._internals.utility.CompletionUtils;
import com.hummer.im._internals.utility.HMRCompletion;
import com.hummer.im._internals.utility.HMRCompletionArg;
import com.hummer.im._internals.utility.HMRContext;
import com.hummer.im._internals.utility.ReportFunction;
import com.hummer.im._internals.utility.RequestIdBuilder;
import com.hummer.im.model.RequestId;
import com.hummer.im.model.auth.TokenProvider;
import com.hummer.im.model.id.User;
import com.hummer.im.model.message.BaseMessage;
import com.hummer.im.model.message.P2PMessageOptions;
import com.hummer.im.model.message.TextMessage;
import com.hummer.im.model.option.HummerOptions;
import com.hummer.im.model.user.UserOnlineStatus;

import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;


public class HummerEngine implements HummerNative.NotificationListener {

    private HummerEngine() {
        HummerNative.registerNotificationListener(this);
    }

    private static HummerEngine INSTANCE = null;

    public static synchronized HummerEngine getInstance() {
        if (INSTANCE == null) {
            INSTANCE = new HummerEngine();
        }
        return INSTANCE;
    }

    public void init(long appId) {
        init(appId, null);
    }

    public void init(long appId, Boolean isDelegateMode) {
        synchronized (this) {
            if (!mIsInited) {
                Log.setLogger(HummerLog.instance());
                if (BuildConfig.DEBUG) {
                    HummerLog.instance().enableConsoleLogger(true);
                } else {
                    HummerLog.instance().enableConsoleLogger(false);
                }

                long ret = HummerNative.init(appId, "", isDelegateMode == null ? hummerOptions.getIsDelegateMode() : isDelegateMode);
                Log.i(TAG, "createRtcEngine | init end");

                if (ret < 0) {
                    Log.e(TAG, "createRtcEngine | init failed");
                    return;
                }
                mIsInited = true;
            }
        }
    }

    public void reportReturnCode(String function, long rtt, int code) {
        HummerNative.reportReturnCode(function, rtt, code);
    }

    /********************************  Notify Begin *****************************************/

    private class NotifyBase extends Marshallable {
        Object notification;

        public Object get() {
            return notification;
        }
    }


    private class NotifyHummerStateChanged extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.HummerStateChanged(popInt(), popInt(), popString16UTF8());
        }
    }

    private class NotifyHummerKicked extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.HummerKicked(popInt(), popString16UTF8());
        }
    }

    private class NotifyHummerLogin extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.HummerLogin(popInt64(), popInt64(), popInt(), popString16UTF8());
        }
    }

    private class NotifyHummerRefreshToken extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.HummerRefreshToken(popInt64(), popInt(), popString16UTF8());
        }
    }

    private class NotifyHummerLogout extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.HummerLogout(popInt64(), popInt(), popString16UTF8());
        }
    }

    private class NotifyHummerLogCallback extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.HummerLogCallback(popInt(), popString16UTF8());
        }
    }

    private class NotifyReportInit extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.ReportInit(popString16UTF8(), popString16UTF8(), popString16UTF8());
        }
    }

    private class NotifyReportArea extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.ReportArea(popInt());
        }
    }

    private class NotifyReportMetricsReturnCode extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.ReportMetricsReturnCode(popInt(), popString16UTF8(), popInt64(), popString16UTF8());
        }
    }

    private class NotifyReport extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.Report(popString16UTF8(), popMap(String.class, String.class));
        }
    }

    @Override
    public void handleNotify(int type, byte[] data) {
        Log.i(TAG, "handleNotify | type: " + type);

        try {
            NotifyBase base;
            switch (type) {
                case HummerNotification.NOTIFY_HUMMER_STATE_CHANGED: {
                    handleHummerStateChanged(data);
                    break;
                }
                case HummerNotification.NOTIFY_HUMMER_KICKED: {
                    base = new NotifyHummerKicked();
                    base.unmarshall(data);
                    final HummerNotification.HummerKicked notify = (HummerNotification.HummerKicked) base.get();
                    handleHummerKicked(notify.getCode(), notify.getDescription());
                    break;
                }
                case HummerNotification.NOTIFY_HUMMER_TOKEN_EXPIRED: {
                    base = new NotifyHummerKicked();
                    base.unmarshall(data);
                    final HummerNotification.HummerKicked notify = (HummerNotification.HummerKicked) base.get();
                    handleTokenExpired(new Error(notify.getCode(), notify.getDescription()));
                    break;
                }
                case HummerNotification.NOTIFY_HUMMER_LOGIN: {
                    if (!mLoginCompletions.isEmpty()) {
                        base = new NotifyHummerLogin();
                        base.unmarshall(data);
                        HummerNotification.HummerLogin notify = (HummerNotification.HummerLogin) base.get();
                        RequestId requestId = new RequestId(notify.getRequestId());
                        HMRCompletion completion = mLoginCompletions.remove(requestId);
                        HummerDispatch.dispatchCompletion(completion, notify.getCode(), notify.getDesc());
                    }
                    break;
                }
                case HummerNotification.NOTIFY_HUMMER_REFRESH_TOKEN: {
                    handleRefreshTokenCallback(data);
                    break;
                }
                case HummerNotification.NOTIFY_HUMMER_LOGOUT: {
                    if (!mLogoutCompletions.isEmpty()) {
                        base = new NotifyHummerLogout();
                        base.unmarshall(data);
                        HummerNotification.HummerLogout notify = (HummerNotification.HummerLogout) base.get();
                        RequestId requestId = new RequestId(notify.getRequestId());
                        HMRCompletion completion = mLogoutCompletions.remove(requestId);
                        HummerDispatch.dispatchCompletion(completion, notify.getCode(), notify.getDesc());
                    }
                    break;
                }
                case HummerNotification.NOTIFY_HUMMER_REFRESH_TOKEN_1:
                    handleRefreshToken1Callback(data);
                    break;
                case HummerNotification.NOTIFY_HUMMER_TOKEN_WILL_EXPIRED:
                    handleTokenWillExpired(data);
                    break;
                case HummerNotification.NOTIFY_HUMMER_SEND_P2P_MESSAGE:
                    handleOnSendP2PMessage(data);
                    break;
                case HummerNotification.NOTIFY_HUMMER_FETCH_USER_ONLINE_STATUS:
                    handleFetchUserOnlineStatus(data);
                    break;
                case HummerNotification.NOTIFY_ON_P2P_MESSAGE_RECEIVED:
                    handleOnP2PMessageReceived(data);
                    break;
                default:
                    break;
            }
        } catch (Exception e) {
            Log.e(TAG, "handleNotify | type: " + type + ", err: " + e.getMessage());
        }
    }

    private void notifyPreviousTokenExpired(final Error error) {
        Log.i(TAG, "notifyPreviousTokenExpired, errCode: " + error.code);
        if (error.code == ErrorEnum.EXPIRED_TOKEN_EXCEPTION.getCode()) {
            HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                @Override
                public void visit() {
                    synchronized (mTokenInvalidListeners) {
                        for (final HMR.TokenInvalidListener l : mTokenInvalidListeners) {
                            l.onHummerTokenInvalid(HMR.TokenInvalidCode.EXPIRED, error.desc);
                            l.onHummerPreviousTokenExpired();
                        }
                    }
                }
            });
        }
    }

    private void handleHummerStateChanged(byte[] data) {
        NotifyHummerStateChanged base = new NotifyHummerStateChanged();
        base.unmarshall(data);
        final HummerNotification.HummerStateChanged notify = (HummerNotification.HummerStateChanged) base.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mHummerListeners) {
                    for (final HMR.HummerListener l : mHummerListeners) {
                        HMR.ConnectionState oldState = HMR.ConnectionState.values()[notify.getOldState()];
                        HMR.ConnectionState newState = HMR.ConnectionState.values()[notify.getNewState()];
                        l.onConnectionStateChanged(oldState, newState, notify.getReason());
                    }
                }
            }
        });
    }

    private void handleTokenExpired(Error err) {
        if (mTokenProvider != null) {
            byte[] token = mTokenProvider.getToken(HMR.getMe().getId());
            if (token != null && token.length > 0) {
                refreshToken1(new String(token), new HMR.Completion() {
                    @Override
                    public void onSuccess() {
                        Log.i(TAG, "refreshToken1 succeed by tokenProvider!");
                    }

                    @Override
                    public void onFailed(Error err) {
                        notifyPreviousTokenExpired(err);
                    }
                });
            } else {
                Log.e(TAG, "getToken is invalid by tokenProvider!");
                notifyPreviousTokenExpired(err);
            }
        } else {
            notifyPreviousTokenExpired(err);
        }
    }

    private void handleRefreshTokenCallback(byte[] data) {
        if (!mRefreshTokenCompletions.isEmpty()) {
            NotifyBase base = new NotifyHummerRefreshToken();
            base.unmarshall(data);
            HummerNotification.HummerRefreshToken notify = (HummerNotification.HummerRefreshToken) base.get();
            RequestId requestId = new RequestId(notify.getRequestId());
            HMRCompletion completion = mRefreshTokenCompletions.remove(requestId);
            HummerDispatch.dispatchCompletion(completion, notify.getCode(), notify.getDesc());
        }
    }

    /**
     * 新接口的回调
     *
     * @param data
     */
    private void handleRefreshToken1Callback(byte[] data) {
        if (!mRefreshTokenCompletions.isEmpty()) {
            NotifyBase base = new NotifyHummerRefreshToken();
            base.unmarshall(data);
            HummerNotification.HummerRefreshToken notify = (HummerNotification.HummerRefreshToken) base.get();
            RequestId requestId = new RequestId(notify.getRequestId());
            HMRCompletion completion = mRefreshTokenCompletions.remove(requestId);
            HummerDispatch.dispatchCompletion(completion, notify.getCode(), notify.getDesc());
        }
    }

    private void handleTokenWillExpired(byte[] data) {
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mHummerListeners) {
                    for (final HMR.HummerListener l : mHummerListeners) {
                        l.onHummerTokenWillExpired();
                    }
                }
            }
        });
    }

    private void handleOnSendP2PMessage(byte[] data) {
        HummerNotification.NotifyBaseCallback notify = new HummerNotification.NotifyBaseCallback();
        notify.unmarshall(data);
        final HummerNotification.BaseCallback callback = notify.get();
        RequestId requestId = new RequestId(callback.getRequestId());
        HMRCompletion completion = HUMMER_ENGINE_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callback.getCode(), callback.getDesc());
        /// report上报
        reportReturnCode(ReportFunction.SEND_P2P_MESSAGE, callback.getRequestId(), callback.getCode());
    }

    private void handleFetchUserOnlineStatus(byte[] data) {
        HummerNotification.NotifyOnFetchUserOnlineStatus notify = new HummerNotification.NotifyOnFetchUserOnlineStatus();
        notify.unmarshall(data);
        final HummerNotification.OnFetchUserOnlineStatus result = notify.get();
        RequestId requestId = new RequestId(result.getRequestId());
        HMRCompletionArg<Set<UserOnlineStatus>> completion = FETCH_USER_ONLINE_STATUS_COMPLETIONS.remove(requestId);

        HummerDispatch.dispatchCompletion(completion, result.getStatus(), result.getCode(), result.getDesc());
        reportReturnCode(ReportFunction.FETCH_USER_ONLINE_STATUS, result.getRequestId(), result.getCode());
    }

    private void handleOnP2PMessageReceived(byte[] data) {
        HummerNotification.NotifyOnP2PMessageReceived notify = new HummerNotification.NotifyOnP2PMessageReceived();
        notify.unmarshall(data);
        final HummerNotification.OnP2PMessageReceived received = notify.get();
        if (received.getBaseMessage() == null) {
            Log.e(TAG, "handleOnP2PMessageReceived | message is null");
            return;
        }

        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mHummerListeners) {
                    for (final HMR.HummerListener l : mHummerListeners) {
                        if (received.getBaseMessage() instanceof TextMessage) {
                            l.onP2PTextMessageReceived(received.getUser(), (TextMessage) received.getBaseMessage());
                        }
                    }
                }
            }
        });
    }

    public void handleLogCallback(final byte[] data) {
        HummerDispatch.runLogDirectAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                NotifyBase base = new NotifyHummerLogCallback();
                base.unmarshall(data);
                HummerNotification.HummerLogCallback notify = (HummerNotification.HummerLogCallback) base.get();
                HummerLog.instance().logCallback(notify.getLevel(), notify.getLog());
            }
        });
    }

    private void handleHummerKicked(final int code, final String reason) {
        synchronized (mKickOutHandler) {
            for (final KickOutHandler l : mKickOutHandler) {
                l.onHummerKickedResult(new Error(code, reason));
            }
        }

        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mHummerListeners) {
                    for (final HMR.HummerListener l : mHummerListeners) {
                        l.onForceoutOffline(code, reason);
                    }
                }
            }
        });
    }

    public static long login(RequestId requestId, long uid, String region, byte[] token, HMR.Completion completion) {
        HMRCompletion c = new HMRCompletion(requestId, completion);
        if (HMR.getState() == HMR.State.Closing
                || HMR.getState() == HMR.State.Closed
                || !HMRContext.isCurrentContext(requestId)) {
            Log.i(TAG, Trace.method("login2").msg("Terminal by logout")
                    .info("state", HMR.getState())
                    .info("currentContext", HMRContext.contextId == null ? 0 : HMRContext.contextId.getId())
                    .info("requestId", requestId.getId()));

            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.BAD_USER_ERROR, "Terminal by logout"));
            return 0;
        }

        mLoginCompletions.put(requestId, c);
        return HummerNative.login(requestId.getId(), uid, region, token, hummerOptions);
    }

    public static long logout(RequestId requestId, HMR.Completion completion) {
        HMRCompletion c = new HMRCompletion(requestId, completion);
        mLogoutCompletions.put(requestId, c);
        return HummerNative.logout(requestId.getId());
    }

    public static String getSdkVersion() {
        return HummerNative.getSdkVersion();
    }

    public static String makeUUID() {
        return HummerNative.makeUUID();
    }

    public static long getSyncTs() {
        return HummerNative.getSyncTs();
    }

    public static HMR.ConnectionState getConnectionState() {
        final RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        int state = HummerNative.getState(requestId.getId());
        return HMR.ConnectionState.values()[state];
    }

    public static long refreshToken(String token, HMR.Completion completion) {
        final RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        if (completion != null) {
            HMRCompletion c = new HMRCompletion(requestId, completion);
            mRefreshTokenCompletions.put(requestId, c);
        }
        return HummerNative.refreshToken(requestId.getId(), token, hummerOptions);
    }


    public static long refreshToken1(String token, HMR.Completion completion) {
        final RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        if (completion != null) {
            HMRCompletion c = new HMRCompletion(requestId, completion);
            mRefreshTokenCompletions.put(requestId, c);
        }
        return HummerNative.refreshToken1(requestId.getId(), token, hummerOptions);
    }

    public static long refreshToken1(byte[] token, HMR.Completion completion) {
        final RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        if (completion != null) {
            HMRCompletion c = new HMRCompletion(requestId, completion);
            mRefreshTokenCompletions.put(requestId, c);
        }
        return HummerNative.refreshToken1(requestId.getId(), token, hummerOptions);
    }

    public static TextMessage createTextMessage(String text) {
        return HummerNative.nativeCreateTextMessage(text);
    }

    public static void sendP2PMessage(User receiver,
                                      BaseMessage message,
                                      P2PMessageOptions options,
                                      HMR.Completion completion) {
        final RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        HMRCompletion c = new HMRCompletion(requestId, completion);

        // uid 不能小于0
        if (receiver == null || receiver.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, HMRContext.INVALID_UID));
            return;
        }

        if (completion != null) {
            HUMMER_ENGINE_COMPLETIONS.put(requestId, c);
        }

        HummerNative.sdkProcess(new HummerEvent.EventSentP2PMessage(requestId.getId(), receiver, options, message));
    }

    public static void fetchUserOnlineStatus(Set<User> users,
                                             HMR.CompletionArg<Set<UserOnlineStatus>> completion) {
        final RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        HMRCompletionArg<Set<UserOnlineStatus>> c = new HMRCompletionArg<>(requestId, completion);

        /* java 没有无符号的类型，故需要限制不能 <0 */
        for (User user : users) {
            if (user == null || user.getId() < 0) {
                CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, HMRContext.INVALID_UID));
                return;
            }
        }

        if (completion != null) {
            FETCH_USER_ONLINE_STATUS_COMPLETIONS.put(requestId, c);
        }

        HummerNative.sdkProcess(new HummerEvent.EventFetchUserOnlineStatus(requestId.getId(), users));
    }

    public static void setTokenProvider(TokenProvider provider) {
        mTokenProvider = provider;
    }

    public static void addTokenInvalidListener(final HMR.TokenInvalidListener listener) {
        if (listener != null) {
            synchronized (mTokenInvalidListeners) {
                mTokenInvalidListeners.add(listener);
            }
        }
    }

    public static void removeTokenInvalidListener(final HMR.TokenInvalidListener listener) {
        if (listener != null) {
            synchronized (mTokenInvalidListeners) {
                mTokenInvalidListeners.remove(listener);
            }
        }
    }

    public static void addHummerListener(final HMR.HummerListener listener) {
        if (listener != null) {
            synchronized (mHummerListeners) {
                mHummerListeners.add(listener);
            }
        }
    }

    public static void removeHummerListener(final HMR.HummerListener listener) {
        if (listener != null) {
            synchronized (mHummerListeners) {
                mHummerListeners.remove(listener);
            }
        }
    }

    public static void addKickOutHandler(final KickOutHandler handler) {
        if (handler != null) {
            synchronized (mKickOutHandler) {
                mKickOutHandler.add(handler);
            }
        }
    }

    public static void removeKickOutHandler(final KickOutHandler handler) {
        if (handler != null) {
            synchronized (mKickOutHandler) {
                mKickOutHandler.remove(handler);
            }
        }
    }

    public interface KickOutHandler {
        /**
         * 被踢结果通知
         *
         * @param error 错误结果
         */
        void onHummerKickedResult(Error error);
    }

    public static void setHummerOptions(HummerOptions options) {
        hummerOptions = options;
        if (options != null && options.getFetchStrategy() != null) {
            HummerNative.setFetchStrategy(options.getFetchStrategy());
        }
    }

    private static HummerOptions hummerOptions = new HummerOptions();

    private static boolean mIsInited = false;
    private static TokenProvider mTokenProvider = null;
    private static final CopyOnWriteArraySet<HMR.TokenInvalidListener> mTokenInvalidListeners = new CopyOnWriteArraySet<>();
    private static final CopyOnWriteArraySet<HMR.HummerListener> mHummerListeners = new CopyOnWriteArraySet<>();
    private static final CopyOnWriteArraySet<KickOutHandler> mKickOutHandler = new CopyOnWriteArraySet<>();

    private static Map<RequestId, HMRCompletion> mLoginCompletions = new ConcurrentHashMap<>();
    private static Map<RequestId, HMRCompletion> mLogoutCompletions = new ConcurrentHashMap<>();
    private static Map<RequestId, HMRCompletion> mRefreshTokenCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> HUMMER_ENGINE_COMPLETIONS = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<Set<UserOnlineStatus>>> FETCH_USER_ONLINE_STATUS_COMPLETIONS = new ConcurrentHashMap<>();

    private static final String TAG = "HummerEngine";
}
