package com.hummer.im._internals;

import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.ErrorEnum;
import com.hummer.im.HMR;
import com.hummer.im._internals.bridge.helper.HummerDispatch;
import com.hummer.im._internals.bridge.helper.HummerNative;
import com.hummer.im._internals.bridge.helper.HummerNotification;
import com.hummer.im._internals.bridge.marshall.Marshallable;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.utility.CompletionUtils;
import com.hummer.im._internals.utility.FileUtils;
import com.hummer.im._internals.utility.HMRCompletion;
import com.hummer.im._internals.utility.HMRContext;
import com.hummer.im._internals.utility.ReportFunction;
import com.hummer.im._internals.utility.RequestIdBuilder;
import com.hummer.im._internals.utility.ServiceProvider;
import com.hummer.im.model.RequestId;
import com.hummer.im.model.chat.AppContent;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.MessageOptions;
import com.hummer.im.model.chat.contents.Audio;
import com.hummer.im.model.chat.contents.CustomContent;
import com.hummer.im.model.chat.contents.Image;
import com.hummer.im.model.chat.contents.Text;
import com.hummer.im.model.chat.contents.Video;
import com.hummer.im.model.chat.states.Archived;
import com.hummer.im.model.chat.states.Failed;
import com.hummer.im.model.chat.states.Init;
import com.hummer.im.model.chat.states.Preparing;
import com.hummer.im.model.chat.store.MessageStoreStrategy;
import com.hummer.im.model.id.AppSession;
import com.hummer.im.model.id.Group;
import com.hummer.im.model.id.IDFactory;
import com.hummer.im.model.id.IDType;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.model.id.User;
import com.hummer.im.service.ChatService;

import java.io.File;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;

public class ChatServiceImpl implements ChatService, ServiceProvider.Service, HummerNative.NotificationListener {

    public static class MessagePacket {

        //公屏
        public static final int MSG_TYPE_TEXT = 10;
        //广播
        public static final int MSG_TYPE_BROADCAST = 20;
        //单播
        public static final int MSG_TYPE_UNICAST = 30;

        public MessagePacket() {
        }

        public MessagePacket(long userId, String uuid, long roomId, int msgType, String data, String extra,
                             Map<String, String> kvExtra) {
            this.userId = userId;
            this.uuid = uuid;
            this.roomId = roomId;
            this.data = data;
            this.msgType = msgType;
            this.extra = extra;
            this.kvExtra = kvExtra;
        }

        private long userId;
        private String uuid;
        private long roomId;
        private int msgType;
        private String data;
        private String extra;
        private Map<String, String> kvExtra;

        public long getUserId() {
            return userId;
        }

        public void setUserId(long userId) {
            this.userId = userId;
        }

        public String getUuid() {
            return uuid;
        }

        public void setUuid(String uuid) {
            this.uuid = uuid;
        }

        public long getRoomId() {
            return roomId;
        }

        public void setRoomId(long roomId) {
            this.roomId = roomId;
        }

        public int getMsgType() {
            return msgType;
        }

        public void setMsgType(int msgType) {
            this.msgType = msgType;
        }

        public String getData() {
            return data;
        }

        public void setData(String data) {
            this.data = data;
        }

        public String getExtra() {
            return extra;
        }

        public void setExtra(String extra) {
            this.extra = extra;
        }

        public Map<String, String> getKvExtra() {
            return kvExtra;
        }

        public void setKvExtra(Map<String, String> kvExtra) {
            this.kvExtra = kvExtra;
        }
    }

    public interface SendingExtension {
        MessagePacket makeMessagePacket(Message message);

        boolean parseMessage(Message message, int msgType, long userId, long roomId, String data);
    }

    private static void registerChatContentCodes() {
        Text.registerCodecs();
        Image.registerCodecs();
        Audio.registerCodecs();
        Video.registerCodecs();
        CustomContent.registerCodecs();
    }

    public static void registerSendingExtension(SendingExtension ext) {
        sendingExtensions.add(ext);
    }

    private static MessagePacket makeSendMessage(Message message) {
        for (SendingExtension ext : sendingExtensions) {
            MessagePacket messageContent = ext.makeMessagePacket(message);

            if (messageContent != null) {
                return messageContent;
            }
        }
        return null;
    }

    private static boolean completionMessage(Message message, int msgType, long userId, long roomId, String data) {
        for (SendingExtension ext : sendingExtensions) {
            if (ext.parseMessage(message, msgType, userId, roomId, data)) {
                return true;
            }
        }
        return false;
    }

    /********************************  Notify Begin *****************************************/

    public final static String CHAT_ROOM_SENDER_PREFIX = "user_";
    public final static String CHAT_ROOM_RECEIVER_PREFIX = "chatroom_";
    public final static String CHAT_ROOM_USER_RECEIVER_PREFIX = CHAT_ROOM_RECEIVER_PREFIX +
            "/user_";
    //单播或广播,如果sender是个人
    public final static String MSG_TYPE_CHAT_ROOM_SIGNAL = "hmr_chatRoom_signal/";
    public final static String MSG_TYPE_CHAT_ROOM_TEXT = "hmr_text/";

    private String makeChatRoomUser(long uid) {
        return CHAT_ROOM_USER_RECEIVER_PREFIX + uid;
    }

    private String makeChatRoom(long roomId) {
        return CHAT_ROOM_RECEIVER_PREFIX + roomId;
    }

    private long ExtractUserId(String userId) {
        return extractUserPrefix(userId, CHAT_ROOM_SENDER_PREFIX);
    }

    private long ExtractChatRoomId(String userId) {
        return extractUserPrefix(userId, CHAT_ROOM_RECEIVER_PREFIX);
    }

    private long extractChatRoomUserId(String userId) {
        return extractUserPrefix(userId, CHAT_ROOM_USER_RECEIVER_PREFIX);
    }

    private long extractUserPrefix(String userId, String prefix) {
        if (TextUtils.isEmpty(userId)) {
            return 0;
        }
        String idStr = userId.replace(prefix, "");
        long id = 0;
        try {
            id = Long.valueOf(idStr);
        } catch (NumberFormatException e) {
        }
        return id;
    }


    private class NotifyBase extends Marshallable {
        Object notification;

        public Object get() {
            return notification;
        }
    }


    private class NotifyOnMessage extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            int stateType = popInt();
            int stateCode = popInt();
            String stateDesc = popString16UTF8();
            Map<String, String> extraInfo = popMap(String.class, String.class);
            String uuid = popString16UTF8();
            long timestamp = popInt64();
            String sender = popString16UTF8();
            String receiver = popString16UTF8();
            String contentData = popString32("utf-8");
            String contentReserved = popString16();
            String extra = popString16UTF8();
            Map<String, String> kv = popMap(String.class, String.class);

            int msgType = popInt();
            long userId = popInt64();
            long roomId = popInt2Long();

            Message message = new Message();
            message.setState(Message.buildSate(stateType, stateCode, stateDesc, extraInfo));
            message.setSender(new User(ExtractUserId(sender)));
            message.setUuid(uuid);
            message.setTimestamp(timestamp);
            message.setAppExtra(extra);
            message.setKvExtra(kv);
            notification =
                    new HummerNotification.OnMessage(message, msgType, userId, roomId, contentData);

        }
    }

    private class NotifyOnSendMessage extends NotifyBase {

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            long requestId = popInt64();
            int code = popInt();
            String desc = popString16UTF8();
            Map<String, String> extraInfo = popMap(String.class, String.class);
            int stateType = popInt();
            int stateCode = popInt();
            String stateDesc = popString16UTF8();
            String uuid = popString16UTF8();
            long timestamp = popInt64();
            String sender = popString16UTF8();
            String receiver = popString16UTF8();
            String contentData = popString32("utf-8");
            String contentReserved = popString16();
            String extra = popString16();
            //回调目前还没有需要到message，所以这里可以不用对message的字段赋值
            Message message = new Message();
            notification = new HummerNotification.OnSendMessage(requestId, code, desc, extraInfo, message);
        }
    }


    @Override
    public void handleNotify(final int type, byte[] data) {
        Log.i(TAG, "handleNotify | type: " + type);
        try {
            NotifyBase base;
            if (type == HummerNotification.NOTIFY_ON_SEND_MESSAGE) {
                base = new NotifyOnSendMessage();
                base.unmarshall(data);
                final HummerNotification.OnSendMessage notify = (HummerNotification.OnSendMessage) base.get();
                RequestId requestId = new RequestId(notify.getRequestId());
                HMRCompletion completion = mSendMessageCompletions.remove(requestId);
                HummerDispatch.dispatchCompletion(completion, notify.getCode(), notify.getDesc(), notify.getExtraInfo());
            } else if (type >= HummerNotification.NOTIFY_ON_BEFORE_SEND_MESSAGE && type <= HummerNotification.NOTIFY_ON_UPDATE_MESSAGE_STATE) {
                base = new NotifyOnMessage();
                base.unmarshall(data);
                final HummerNotification.OnMessage notify = (HummerNotification.OnMessage) base.get();
                boolean result = completionMessage(notify.getMessage(), notify.getMsgType(), notify.getUserId(), notify.getRoomId()
                        , notify.getData());
                if (result) {
                    final Message message = notify.getMessage();
                    HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
                        @Override
                        public void visit() {
                            if (type == HummerNotification.NOTIFY_ON_UPDATE_MESSAGE_STATE) {
                                notifyMessageStateChange(message);

                            } else {
                                synchronized (mMessageListeners) {
                                    // 避免同一个listener被通知多次，先将它们收集到一个set内，然后在发起访问
                                    String key = String.valueOf(message.getTarget() == null ? "" : message.getTarget().getId());
                                    HashSet<MessageListener> listeners = mMessageListeners.get(key);
                                    HashSet<MessageListener> hashLis;

                                    if (listeners == null) {
                                        hashLis = new HashSet<>();
                                    } else {
                                        hashLis = (HashSet<MessageListener>) listeners.clone();
                                    }

                                    HashSet<MessageListener> fullListeners = mMessageListeners.get(DEFAULT_MESSAGE_LISTENER_KEY);

                                    if (fullListeners != null) {
                                        hashLis.addAll((HashSet<MessageListener>) fullListeners.clone());
                                    }

                                    for (MessageListener listener : hashLis) {
                                        if (type == HummerNotification.NOTIFY_ON_BEFORE_SEND_MESSAGE) {
                                            listener.beforeSendingMessage(notify.getMessage());
                                        } else if (type == HummerNotification.NOTIFY_ON_AFTER_SEND_MESSAGE) {
                                            listener.afterSendingMessage(notify.getMessage());
                                        } else if (type == HummerNotification.NOTIFY_ON_BEFORE_RECEIVE_MESSAGE) {
                                            listener.beforeReceivingMessage(notify.getMessage());
                                        } else if (type == HummerNotification.NOTIFY_ON_AFTER_RECEIVE_MESSAGE) {
                                            listener.afterReceivingMessage(notify.getMessage());
                                        }
                                    }
                                }

                                synchronized (MESSAGE_SERVICE_LISTENERS) {
                                    for (MessageServiceListener listener : MESSAGE_SERVICE_LISTENERS) {
                                       if (type == HummerNotification.NOTIFY_ON_AFTER_RECEIVE_MESSAGE) {
                                            listener.onMessageReceived(notify.getMessage());
                                        }
                                    }
                                }
                            }
                        }
                    });
                } else {
                    Log.e(TAG, "handleNotify | completionMessage failed!");
                }
            } else if (type == HummerNotification.NOTIFY_ON_MESSAGE_STATE_UPDATED) {
                handleOnMessageStateUpdated(data);
            } else if (type == HummerNotification.NOTIFY_ON_MESSAGE_PREPARE_PROGRESS) {
                handleOnMessagePrepareProgress(data);
            } else if (type == HummerNotification.NOTIFY_ON_MESSAGE_RECEIVED) {
                handleOnMessageReceived(data);
            }
        } catch (Exception e) {
            Log.e(TAG, "handleNotify | type: " + type + ", err: " + e.getMessage());
        }
    }

    private void handleOnMessageStateUpdated(byte[] data) {
        HummerNotification.NotifyOnMessageStateUpdated base = new HummerNotification.NotifyOnMessageStateUpdated();
        base.unmarshall(data);
        final HummerNotification.OnMessageStateUpdated notify = base.get();

        final Message.State state = notify.getMessage().getState();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {

                notifyMessageStateChange(notify.getMessage());

                // MessageListener 下有发送过程的通知，需要兼容
                if (state instanceof Init) {
                    synchronized (mMessageListeners){
                        for (MessageListener listener : getMessageListenerSet(notify.getMessage())) {
                            listener.beforeSendingMessage(notify.getMessage());
                        }
                    }
                } else if (state instanceof Archived || state instanceof Failed) {
                    synchronized (mMessageListeners){
                        for (MessageListener listener : getMessageListenerSet(notify.getMessage())) {
                            listener.afterSendingMessage(notify.getMessage());
                        }
                    }
                }
            }
        });

        // 回调结果最后返回
        RequestId requestId = new RequestId(notify.getRequestId());
        if (state instanceof Archived) {
            HMRCompletion completion = CHAT_SERVICE_COMPLETIONS.remove(requestId);
            HummerDispatch.dispatchCompletion(completion, ErrorEnum.OK.getCode(), ErrorEnum.OK.getDesc());
        } else if (state instanceof Failed) {
            HMRCompletion completion = CHAT_SERVICE_COMPLETIONS.remove(requestId);
            Failed failed = (Failed) state;
            HummerDispatch.dispatchCompletion(completion, failed.error.code, failed.error.desc, failed.error.extraInfo);
        }
    }

    private void notifyMessageStateChange(Message message) {
        synchronized (mStateListeners) {
            HashSet<StateListener> listeners = mStateListeners.get(message.getUuid());

            // 对listener进行深拷贝，避免多并发访问导致崩溃。
            if (listeners != null) {
                @SuppressWarnings("unchecked")
                HashSet<StateListener> hashLis = (HashSet<StateListener>) listeners.clone();
                for (final StateListener l : hashLis) {
                    l.onUpdateMessageState(message, message.getState());
                }
            }

            HashSet<StateListener> fullListeners = mStateListeners.get(DEFAULT_STATE_LISTENER_KEY);
            if (fullListeners != null) {
                @SuppressWarnings("unchecked")
                HashSet<StateListener> hashFullLis = (HashSet<StateListener>) fullListeners.clone();
                for (final StateListener l : hashFullLis) {
                    l.onUpdateMessageState(message, message.getState());
                }
            }
        }

        synchronized (MESSAGE_SERVICE_LISTENERS) {
            for (ChatService.MessageServiceListener listener : MESSAGE_SERVICE_LISTENERS) {
                listener.onMessageStateUpdated(message);
            }
        }
    }

    private void handleOnMessagePrepareProgress(byte[] data) {
        HummerNotification.NotifyOnMessagePrepareProgress base = new HummerNotification.NotifyOnMessagePrepareProgress();
        base.unmarshall(data);
        final HummerNotification.OnMessagePrepareProgress notify = base.get();

        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (MESSAGE_SERVICE_LISTENERS) {
                    for (ChatService.MessageServiceListener listener : MESSAGE_SERVICE_LISTENERS) {
                        listener.onContentProgressUpdated(notify.getMessage());
                    }
                }
            }
        });
    }

    private void handleOnMessageReceived(byte[] data) {
        HummerNotification.NotifyOnMessageReceived base = new HummerNotification.NotifyOnMessageReceived();
        base.unmarshall(data);
        final HummerNotification.OnMessageReceived notify = base.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mMessageListeners){
                    for (MessageListener listener : getMessageListenerSet(notify.getMessage())) {
                        listener.beforeReceivingMessage(notify.getMessage());
                        listener.afterReceivingMessage(notify.getMessage());
                    }
                }

                synchronized (MESSAGE_SERVICE_LISTENERS) {
                    for (ChatService.MessageServiceListener listener : MESSAGE_SERVICE_LISTENERS) {
                        listener.onMessageReceived(notify.getMessage());
                    }
                }
            }
        });
    }

    private Set<MessageListener> getMessageListenerSet(Message message){
        String key = String.valueOf(message.getTarget() == null ? "" : message.getTarget().getId());
        HashSet<MessageListener> listeners = mMessageListeners.get(key);
        HashSet<MessageListener> hashLis;

        if (listeners == null) {
            hashLis = new HashSet<>();
        } else {
            hashLis = (HashSet<MessageListener>) listeners.clone();
        }

        HashSet<MessageListener> fullListeners = mMessageListeners.get(DEFAULT_MESSAGE_LISTENER_KEY);

        if (fullListeners != null) {
            hashLis.addAll((HashSet<MessageListener>) fullListeners.clone());
        }

        return hashLis;
    }

    @Override
    public void initService() {
        HummerNative.registerNotificationListener(this);
        registerChatContentCodes();
    }

    @Override
    public void openService(@NonNull HMRCompletion completion) {
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public int serviceSort() {
        return 0;
    }

    @Override
    public void closeService() {

    }

    @Override
    public void send(@NonNull Message message, @Nullable final HMR.Completion completion) {
        final RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        HMRCompletion hmrCompletion = createHMRCompletion(requestId, completion, ReportFunction.send);
        if (message == null) {
            CompletionUtils.dispatchFailure(hmrCompletion,
                    new Error(ErrorEnum.INVALID_PARAMETER, "message cannot be null"));
            return;
        }

        sendMessage(requestId, message, null, hmrCompletion);
    }

    @Override
    public void addMessageListener(@Nullable Identifiable target, @NonNull MessageListener listener) {
        if (listener != null) {
            synchronized (mMessageListeners) {
                String lisKey = DEFAULT_MESSAGE_LISTENER_KEY;
                if (target != null) {
                    lisKey = String.valueOf(target.getId());
                }

                HashSet<MessageListener> listeners = mMessageListeners.get(lisKey);
                if (listeners == null) {
                    listeners = new HashSet<>();
                    mMessageListeners.put(lisKey, listeners);
                }
                listeners.add(listener);
            }
        }
        HMRContext.reportReturnCode(ReportFunction.addMessageListener, HMRContext.getCurrentTime());
    }

    @Override
    public void removeMessageListener(@Nullable Identifiable target, @NonNull MessageListener listener) {
        if (listener != null) {
            synchronized (mMessageListeners) {
                String lisKey = DEFAULT_MESSAGE_LISTENER_KEY;
                if (target != null) {
                    lisKey = String.valueOf(target.getId());
                }

                HashSet<MessageListener> listeners = mMessageListeners.get(lisKey);
                if (listeners != null) {
                    listeners.remove(listener);
                }
            }
        }
        HMRContext.reportReturnCode(ReportFunction.removeMessageListener, HMRContext.getCurrentTime());
    }

    @Override
    public void addStateListener(@Nullable Message message, @NonNull StateListener listener) {
        if (listener != null) {
            synchronized (mStateListeners) {
                String lisKey = DEFAULT_STATE_LISTENER_KEY;
                if (message != null) {
                    lisKey = message.getUuid();
                }

                HashSet<StateListener> listeners = mStateListeners.get(lisKey);
                if (listeners == null) {
                    listeners = new HashSet<>();
                    mStateListeners.put(lisKey, listeners);
                }
                listeners.add(listener);
            }

        }
        HMRContext.reportReturnCode(ReportFunction.addStateListener, HMRContext.getCurrentTime());
    }

    @Override
    public void removeStateListener(@Nullable Message message, @NonNull StateListener listener) {
        if (listener != null) {
            synchronized (mStateListeners) {
                String lisKey = DEFAULT_STATE_LISTENER_KEY;
                if (message != null) {
                    lisKey = message.getUuid();
                }

                HashSet<StateListener> listeners = mStateListeners.get(lisKey);
                if (listeners != null) {
                    listeners.remove(listener);
                }
            }
        }
        HMRContext.reportReturnCode(ReportFunction.removeStateListener, HMRContext.getCurrentTime());
    }

    @Override
    public void addMessageServiceListener(MessageServiceListener listener) {
        if (listener != null) {
            synchronized (MESSAGE_SERVICE_LISTENERS) {
                MESSAGE_SERVICE_LISTENERS.add(listener);
            }
        }
        HMRContext.reportReturnCode(ReportFunction.ADD_MESSAGE_SERVICE_LISTENER, HMRContext.getCurrentTime());
    }

    @Override
    public void removeMessageServiceListener(MessageServiceListener listener) {
        if (listener != null) {
            synchronized (MESSAGE_SERVICE_LISTENERS) {
                MESSAGE_SERVICE_LISTENERS.remove(listener);
            }
        }
        HMRContext.reportReturnCode(ReportFunction.REMOVE_MESSAGE_SERVICE_LISTENER, HMRContext.getCurrentTime());
    }

    @Override
    public Text createText(String text) {
        return new Text(text);
    }

    @Override
    public Image createImage(@NonNull String path, int width, int height) {
        return Image.create(path, width, height);
    }

    @Override
    public Audio createAudio(@NonNull String path, int duration) {
        return Audio.create(path, duration);
    }

    @Override
    public Video createVideo(@NonNull String videoPath,
                             long videoSize,
                             int videoWidth,
                             int videoHeight,
                             int videoDuration,
                             @NonNull String coverPath,
                             int coverWidth,
                             int coverHeight) {
        return Video.create(videoPath, videoSize, videoWidth, videoHeight, videoDuration,
                coverPath, coverWidth, coverHeight);
    }

    @Override
    public CustomContent createCustomContent(byte[] data) {
        return new CustomContent(data);
    }

    @Override
    public AppContent createAppContent(int type, byte[] data) {
        return new AppContent(type, data);
    }

    @Override
    public Message createMessage(@NonNull Identifiable target, @NonNull Content messageContent) {
        return new Message(target, messageContent);
    }

    @Override
    public Message createMessage(@NonNull Identifiable target,
                                 @NonNull Content messageContent,
                                 @NonNull MessageStoreStrategy storeStrategy) {
        return new Message(target, messageContent, storeStrategy);
    }

    @Override
    public void sendMessage(@NonNull Message message, MessageOptions options, final HMR.Completion completion) {
        final RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        HMRCompletion hmrCompletion = createHMRCompletion(requestId, completion, ReportFunction.SEND_MESSAGE_WITH_OPTIONS);

        if (message == null || message.getContent() == null) {
            CompletionUtils.dispatchFailure(hmrCompletion,
                    new Error(ErrorEnum.INVALID_PARAMETER, "Message Or Content Cannot Be Null"));
            return ;
        }
        if (message.getReceiver() == null) {
            CompletionUtils.dispatchFailure(hmrCompletion,
                    new Error(ErrorEnum.INVALID_PARAMETER, "Invalid Receiver"));
            return ;
        }

        if (message.getStoreStrategy() == null) {
            message.setStoreStrategy(new MessageStoreStrategy());
        }

        sendMessage(requestId, message, options, hmrCompletion);
    }

    private void sendMessage(RequestId requestId, Message message, MessageOptions options, HMRCompletion completion) {
        boolean flag = IDType.CHAT_ROOM.equals(IDFactory.makeType(message.getReceiver()));
        if (flag) {
            mSendMessageCompletions.put(requestId, completion);
            MessagePacket packet = makeSendMessage(message);
            HummerNative.sendMessage(requestId.getId(), packet);
        } else {
            Error error = validSendMessage(message);
            if (error != null) {
                CompletionUtils.dispatchFailure(completion, error);
                return;
            }

            CHAT_SERVICE_COMPLETIONS.put(requestId, completion);
            HummerNative.sendMessage(requestId.getId(), message, options);
        }
    }

    private HMRCompletion createHMRCompletion(@NonNull final RequestId requestId,
                                              final HMR.Completion completion,
                                              final String functionName) {
        final long now = HMRContext.getCurrentTime();
        return new HMRCompletion(requestId, new HMR.Completion() {
            @Override
            public void onSuccess() {
                if (completion != null) {
                    completion.onSuccess();
                }
                // 上报
                HMRContext.reportReturnCode(functionName, now);
            }

            @Override
            public void onFailed(Error err) {
                if (completion != null) {
                    completion.onFailed(err);
                }
                HMRContext.reportReturnCode(functionName, now, err);
            }
        });
    }

    private Error validSendMessage(Message message) {
        if (!message.getReceiver().valid()) {
            return new Error(ErrorEnum.INVALID_PARAMETER, "Invalid Receiver");
        }

        return message.getContent().validContent();
    }

    private static final String TAG = "ChatServiceImpl";

    private final Map<String, HashSet<MessageListener>> mMessageListeners = new HashMap<>();
    private final Map<String, HashSet<StateListener>> mStateListeners = new HashMap<>();
    private static final Set<MessageServiceListener> MESSAGE_SERVICE_LISTENERS = new CopyOnWriteArraySet<>();
    private final static String DEFAULT_STATE_LISTENER_KEY = "default_state_listener_key";
    private final static String DEFAULT_MESSAGE_LISTENER_KEY = "default_message_listener_key";

    private static final Map<RequestId, HMRCompletion> mSendMessageCompletions = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletion> CHAT_SERVICE_COMPLETIONS = new ConcurrentHashMap<>();

    private static List<SendingExtension> sendingExtensions = new ArrayList<>();
}


