package com.hummer.im._internals;

import androidx.annotation.NonNull;

import com.hummer.im._internals.bridge.helper.HummerDispatch;
import com.hummer.im._internals.bridge.helper.HummerNative;
import com.hummer.im._internals.bridge.helper.HummerNotification;
import com.hummer.im._internals.bridge.marshall.Marshallable;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.utility.CompletionUtils;
import com.hummer.im._internals.utility.HMRCompletion;
import com.hummer.im._internals.utility.HMRContext;
import com.hummer.im._internals.utility.ReportFunction;
import com.hummer.im._internals.utility.RequestIdBuilder;
import com.hummer.im._internals.utility.ServiceProvider;
import com.hummer.im.model.RequestId;
import com.hummer.im.service.ChannelStateService;

import java.util.HashSet;
import java.util.Set;

public class ChannelStateServiceImpl implements ChannelStateService, ServiceProvider.Service, HummerNative.NotificationListener {
    @Override
    public void handleNotify(int type, byte[] data) {
        Log.i(TAG, "handleNotify | type: " + type);
        if(type == HummerNotification.NOTIFY_HUMMER_STATE_CHANGED) {
            handleHummerStateChanged(data);
        }
    }

    @Override
    public void initService() {
        HummerNative.registerNotificationListener(this);
    }

    @Override
    public void openService(@NonNull HMRCompletion completion) {
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public int serviceSort() {
        return 0;
    }

    @Override
    public void closeService() {

    }

    @Override
    public ChannelState getState() {
        final long now = HMRContext.getCurrentTime();
        final RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        int state = HummerNative.getState(requestId.getId());
        ChannelState channelState = getConvertState(state);
        HMRContext.reportReturnCode(ReportFunction.getState, now);
        return channelState;
    }

    @Override
    public void addChannelStateListener(@NonNull ChannelStateListener listener) {
        HMRContext.reportReturnCode(ReportFunction.addChannelStateListener, HMRContext.getCurrentTime());
        if(listener != null) {
            synchronized (mChannelStateListener) {
                mChannelStateListener.add(listener);
            }
        }
    }

    @Override
    public void removeChannelStateListener(@NonNull ChannelStateListener listener) {
        HMRContext.reportReturnCode(ReportFunction.removeChannelStateListener, HMRContext.getCurrentTime());
        if(listener != null) {
            synchronized (mChannelStateListener) {
                mChannelStateListener.remove(listener);
            }
        }
    }

    private class NotifyHummerStateChanged extends Marshallable {
        private Object notification;

        public Object get() {
            return notification;
        }

        @Override
        public void unmarshall(byte[] buf) {
            super.unmarshall(buf);
            notification = new HummerNotification.HummerStateChanged(popInt(), popInt(), popString16UTF8());
        }
    }

    private void handleHummerStateChanged(byte[] data) {
        NotifyHummerStateChanged base = new NotifyHummerStateChanged();
        base.unmarshall(data);
        final HummerNotification.HummerStateChanged notify = (HummerNotification.HummerStateChanged) base.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (mChannelStateListener) {
                    for (final ChannelStateListener l : mChannelStateListener) {
                        ChannelState oldState = getConvertState(notify.getOldState());
                        ChannelState newState = getConvertState(notify.getNewState());
                        l.onUpdateChannelState(oldState, newState);
                    }
                }
            }
        });
    }

    private static ChannelState getConvertState(int state) {
        switch (state) {
            case 0:
            case 1:
                return ChannelState.Disconnected;
            case 2:
            case 3:
                return ChannelState.Connecting;
            case 4:
                return ChannelState.Connected;
            default:
                return ChannelState.Disconnected;
        }
    }

    private static final String TAG = "ChannelStateServiceImpl";

    private static final Set<ChannelStateListener> mChannelStateListener = new HashSet<>();
}
