package com.hummer.im._internals;

import androidx.annotation.NonNull;

import com.hummer.im.Error;
import com.hummer.im.ErrorEnum;
import com.hummer.im.HMR;
import com.hummer.im._internals.bridge.helper.BlacklistNative;
import com.hummer.im._internals.bridge.helper.BlacklistNotification;
import com.hummer.im._internals.bridge.helper.HummerDispatch;
import com.hummer.im._internals.bridge.helper.HummerNative;
import com.hummer.im._internals.bridge.helper.HummerNotification;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.utility.CompletionUtils;
import com.hummer.im._internals.utility.HMRCompletion;
import com.hummer.im._internals.utility.HMRCompletionArg;
import com.hummer.im._internals.utility.HMRContext;
import com.hummer.im._internals.utility.ReportFunction;
import com.hummer.im._internals.utility.ServiceProvider;
import com.hummer.im.model.RequestId;
import com.hummer.im.model.id.User;
import com.hummer.im.service.BlacklistService;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;

public class BlacklistServiceImpl implements BlacklistService, ServiceProvider.Service, HummerNative.NotificationListener {

    @Override
    public void initService() {
        HummerNative.registerNotificationListener(this);
    }

    @Override
    public void openService(@NonNull HMRCompletion completion) {
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public int serviceSort() {
        return 0;
    }

    @Override
    public void closeService() {

    }

    @Override
    public void listBlacklist(HMR.CompletionArg<List<User>> completion) {
        final RequestId requestId = HMRContext.createRequestId();
        HMRCompletionArg<List<User>> c = new HMRCompletionArg<>(requestId, completion);

        if (completion != null) {
            LIST_BLACKLIST_COMPLETIONS.put(requestId, c);
        }
        BlacklistNative.listBlacklist(requestId.getId());
    }

    @Override
    public void isBlocked(@NonNull User user, HMR.CompletionArg<Boolean> completion) {
        final RequestId requestId = HMRContext.createRequestId();
        HMRCompletionArg<Boolean> c = new HMRCompletionArg<>(requestId, completion);

        // uid 不能小于0
        if (user == null || user.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, HMRContext.INVALID_OPERATED_UID));
            return;
        }

        if (completion != null) {
            IS_BLOCKED_COMPLETIONS.put(requestId, c);
        }
        BlacklistNative.isBlocked(requestId.getId(), user);
    }

    @Override
    public void block(@NonNull User user, HMR.Completion completion) {
        final RequestId requestId = HMRContext.createRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        // uid 不能小于0
        if (user == null || user.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, HMRContext.INVALID_OPERATED_UID));
            return;
        }

        if (completion != null) {
            BLACKLIST_COMPLETIONS.put(requestId, c);
        }
        BlacklistNative.block(requestId.getId(), user);
    }

    @Override
    public void unblock(@NonNull User user, HMR.Completion completion) {
        final RequestId requestId = HMRContext.createRequestId();
        HMRCompletion c = new HMRCompletion(requestId, completion);

        // uid 不能小于0
        if (user == null || user.getId() < 0) {
            CompletionUtils.dispatchFailure(c, new Error(ErrorEnum.INVALID_PARAMETER, HMRContext.INVALID_OPERATED_UID));
            return;
        }

        if (completion != null) {
            BLACKLIST_COMPLETIONS.put(requestId, c);
        }
        BlacklistNative.unblock(requestId.getId(), user);
    }

    @Override
    public void addListener(BlacklistListener listener) {
        if (listener != null) {
            synchronized (BLACKLIST_LISTENERS) {
                BLACKLIST_LISTENERS.add(listener);
            }
        }
        HMRContext.reportReturnCode(ReportFunction.ADD_BLACKLIST_LISTENER, HMRContext.getCurrentTime());
    }

    @Override
    public void removeListener(BlacklistListener listener) {
        if (listener != null) {
            synchronized (BLACKLIST_LISTENERS) {
                BLACKLIST_LISTENERS.remove(listener);
            }
        }
        HMRContext.reportReturnCode(ReportFunction.REMOVE_BLACKLIST_LISTENER, HMRContext.getCurrentTime());
    }

    @Override
    public void handleNotify(int type, byte[] data) {
        int notifyType = type / 100;
        if (notifyType != 9) {
            return;
        }

        try {
            if (type == BlacklistNotification.CALLBACK_ON_LIST_BLACKLIST) {
                handleCallbackOnListBlacklist(data);
            } else if (type == BlacklistNotification.CALLBACK_ON_IS_USER_BLOCKED) {
                handleCallbackOnIsUserBlocked(data);
            } else if (type == BlacklistNotification.CALLBACK_ON_BLOCK_USER) {
                handleCallbackOnBlockUser(data);
            } else if (type == BlacklistNotification.CALLBACK_ON_UNBLOCK_USER) {
                handleCallbackOnUnblockUser(data);
            } else if (type == BlacklistNotification.NOTIFY_ON_BLOCK_USER) {
                handleBlacklistNotification(data, BlacklistNotification.NOTIFY_ON_BLOCK_USER);
            } else if (type == BlacklistNotification.NOTIFY_ON_UNBLOCK_USER) {
                handleBlacklistNotification(data, BlacklistNotification.NOTIFY_ON_UNBLOCK_USER);
            } else if (type == BlacklistNotification.NOTIFY_ON_BLOCKED_BY_USER) {
                handleBlacklistNotification(data, BlacklistNotification.NOTIFY_ON_BLOCKED_BY_USER);
            } else if (type == BlacklistNotification.NOTIFY_ON_UNBLOCKED_BY_USER) {
                handleBlacklistNotification(data, BlacklistNotification.NOTIFY_ON_UNBLOCKED_BY_USER);
            }
        } catch (Exception e) {
            Log.e(TAG, "handleNotify | type: " + type + ", err: " + e.getMessage());
        }
    }

    private void handleCallbackOnListBlacklist(byte[] data) {
        BlacklistNotification.CallbackOnListBlacklist notify = new BlacklistNotification.CallbackOnListBlacklist();
        notify.unmarshall(data);
        final BlacklistNotification.OnListBlacklist callback = notify.get();
        RequestId requestId = new RequestId(callback.getRequestId());
        HMRCompletionArg<List<User>> completion = LIST_BLACKLIST_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callback.getUsers(), callback.getCode(), callback.getDesc());
        /// report上报
        HMRContext.report(ReportFunction.LIST_BLACKLIST, callback.getRequestId(), callback.getCode());
    }

    private void handleCallbackOnIsUserBlocked(byte[] data) {
        BlacklistNotification.CallbackOnIsUserBlocked notify = new BlacklistNotification.CallbackOnIsUserBlocked();
        notify.unmarshall(data);
        final BlacklistNotification.OnIsUserBlocked callback = notify.get();
        RequestId requestId = new RequestId(callback.getRequestId());
        HMRCompletionArg<Boolean> completion = IS_BLOCKED_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callback.getIsBlock(), callback.getCode(), callback.getDesc());

        // report上报
        HMRContext.report(ReportFunction.IS_BLOCKED, callback.getRequestId(), callback.getCode());
    }

    private void handleCallbackOnBlockUser(byte[] data) {
        handleBaseCallback(data, ReportFunction.BLOCK);
    }

    private void handleCallbackOnUnblockUser(byte[] data) {
        handleBaseCallback(data, ReportFunction.UNBLOCK);
    }

    private void handleBaseCallback(byte[] data, String functionName) {
        HummerNotification.NotifyBaseCallback notify = new HummerNotification.NotifyBaseCallback();
        notify.unmarshall(data);
        final HummerNotification.BaseCallback callback = notify.get();
        RequestId requestId = new RequestId(callback.getRequestId());
        HMRCompletion completion = BLACKLIST_COMPLETIONS.remove(requestId);
        HummerDispatch.dispatchCompletion(completion, callback.getCode(), callback.getDesc());
        /// report上报
        HMRContext.report(functionName, callback.getRequestId(), callback.getCode());
    }

    public void handleBlacklistNotification(final byte[] data, final int type) {
        BlacklistNotification.NotifyOnBlockUser notify = new BlacklistNotification.NotifyOnBlockUser();
        notify.unmarshall(data);
        final BlacklistNotification.OnBlockUser info = notify.get();
        HummerDispatch.runOutAction(new HummerDispatch.RunOutActionVisitor() {
            @Override
            public void visit() {
                synchronized (BLACKLIST_LISTENERS) {
                    for (BlacklistListener listener : BLACKLIST_LISTENERS) {
                        if (type == BlacklistNotification.NOTIFY_ON_BLOCK_USER) {
                            listener.onBlockUser(info.getUser());
                        } else if (type == BlacklistNotification.NOTIFY_ON_UNBLOCK_USER) {
                            listener.onUnblockUser(info.getUser());
                        } else if (type == BlacklistNotification.NOTIFY_ON_BLOCKED_BY_USER) {
                            listener.onBlockedByUser(info.getUser());
                        } else if (type == BlacklistNotification.NOTIFY_ON_UNBLOCKED_BY_USER) {
                            listener.onUnblockedByUser(info.getUser());
                        }
                    }
                }
            }
        });
    }


    private static final Set<BlacklistService.BlacklistListener> BLACKLIST_LISTENERS = new CopyOnWriteArraySet<>();

    private static final Map<RequestId, HMRCompletion> BLACKLIST_COMPLETIONS = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<List<User>>> LIST_BLACKLIST_COMPLETIONS = new ConcurrentHashMap<>();
    private static final Map<RequestId, HMRCompletionArg<Boolean>> IS_BLOCKED_COMPLETIONS = new ConcurrentHashMap<>();

    private static final String TAG = "BlacklistService";

}