package com.hummer.im;

import android.content.Context;
import android.text.TextUtils;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

import com.hummer.im._internals.bridge.helper.HummerEngine;
import com.hummer.im._internals.log.HummerLog;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.Trace;
import com.hummer.im._internals.utility.CompletionUtils;
import com.hummer.im._internals.utility.DispatchQueue;
import com.hummer.im._internals.utility.HMRCompletion;
import com.hummer.im._internals.utility.HMRContext;
import com.hummer.im._internals.utility.ReportFunction;
import com.hummer.im._internals.utility.RequestIdBuilder;
import com.hummer.im._internals.utility.ServiceProvider;
import com.hummer.im.model.RequestId;
import com.hummer.im.model.auth.TokenProvider;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccess;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.model.id.User;
import com.hummer.im.model.message.BaseMessage;
import com.hummer.im.model.message.P2PMessageOptions;
import com.hummer.im.model.message.TextMessage;
import com.hummer.im.model.option.HummerOptions;
import com.hummer.im.model.user.UserOnlineStatus;

import java.lang.ref.WeakReference;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;

import static com.hummer.im._internals.utility.HMRContext.INVALID_UID;


/**
 * Hummer类型是服务的总入口，主要包含了SDK初始化，以及用户上下文切换的功能
 * <br>实际业务接入时，必须遵循下述原则：
 * <br>1. 需要在init之前，调用Hummer.registerChannel方法注册相应的Channel对象实例
 * 如果没有注册将默认为SDK提供的自治模式（AutonomousMode）的ServiceChannel
 * <br>2. 必须在App初始化时，依赖Hummer的模块初始化之前，调用Hummer.init方法，进行初始化
 * <br>3. 在用户身份确定（成功登录）后，调用{@link HMR#open}启用IM用户上下文
 * <br>4. 在用户身份失效（退出登录）后，调用{@link HMR#close}注销IM用户上下文
 */

public final class HMR {

    public enum State {
        Unavailable,
        Opening,
        Opened,
        Closing,
        Closed,
    }

    public enum ConnectionState {
        Unavailable,
        Disconnected,
        Connecting,
        Reconnecting,
        Connected,
    }

    private HMR() {
    }

    private static boolean isInitialized() {
        return HMRContext.appId != null || HMR.state != State.Unavailable;
    }

    /**
     * Hummer SDK的初始化方法，关于运行环境的参数，欢迎垂询
     *
     * @param appContext Android应用AppContext实例对象
     * @param appId      应用appId，它是区分不同业务的重要标识。目前需要和Hummer服务提供方通过人工申请
     * @param provider   用于服务初始化时所需token的获取
     *
     * @since 2.6.11
     */
    @Deprecated
    public static void init(@NonNull final Context appContext,
                            final long appId,
                            @Nullable final TokenProvider provider) {
        long now = HMRContext.getCurrentTime();
        if (isInitialized()) {
            Log.e("HMR", Trace.method("init")
                    .info("Hummer already been initialized!", ""));
            return;
        }
        HMRContext.appId = appId;
        HMRContext.appContext = new WeakReference<>(appContext.getApplicationContext());
        // 如果 provider设置为null， 则表示使用代理模式
        HummerEngine.getInstance().init(appId, provider == null);

        mTokenProvider = provider;
        HummerEngine.setTokenProvider(provider);

        ServiceProvider.loadServicesIfNeeded("com.hummer.im");
        setHummerState(State.Closed);
        HMRContext.reportReturnCode(ReportFunction.initWithTokenProvider, now);
    }

    /**
     * Hummer SDK的初始化方法，关于运行环境的参数，欢迎垂询
     *
     * <br> 1. 业务需要在init之前register相应的Channel对象实例
     * <br> 2. 如果没有register想默认为SDK提供的自治模式（AutonomousMode）的ServiceChannel
     *
     * @param appContext Android应用AppContext实例对象
     * @param appId      应用appId，它是区分不同业务的重要标识。目前需要和Hummer服务提供方通过人工申请
     */
    public static void init(@NonNull final Context appContext,
                            final long appId) {
        Log.i(TAG, "[start] init appId: " + appId);

        long now = HMRContext.getCurrentTime();
        if (isInitialized()) {
            Log.e("HMR", Trace.method("init")
                    .info("Hummer already been initialized!", ""));
            return;
        }
        HMRContext.appId = appId;
        HMRContext.appContext = new WeakReference<>(appContext.getApplicationContext());
        HummerEngine.getInstance().init(appId);
        ServiceProvider.loadServicesIfNeeded("com.hummer.im");
        setHummerState(State.Closed);
        HMRContext.reportReturnCode(ReportFunction.init, now);

        Log.i(TAG, "[end] init appId: " + appId);
    }

    /**
     * Hummer相关配置选项，在调用init方法之前调用该接口
     *
     * <br> 该配置只对新接口生效
     *
     * @param options 选项
     */
    public static void setHummerOptions(@NonNull HummerOptions options) {
        Log.i(TAG, Trace.method("setHummerOptions").msg(options));
        if (options == null) {
            return;
        }
        HummerEngine.setHummerOptions(options);
    }

    /**
     * 切换至指定用户的工作上下文，该方法应该在上层业务的用户登录操作完成之后执行
     *
     * @param uid        用户uid
     * @param region     启用用户上下文所需的区域配置参数
     * @param tagSet     用于消息过滤
     * @param completion 异步操作回调，可为null，会在操作结束时通过其不同的回调方法返回操作结果。
     *
     * @since 2.6.11
     */
    @Deprecated
    public static void open(final long uid,
                            @NonNull final String region,
                            @Nullable final Set<String> tagSet,
                            @Nullable final HMR.Completion completion) {
        String token = "";
        if (mTokenProvider != null) {
            byte[] bytes = mTokenProvider.getToken(uid);
            if (bytes != null && bytes.length > 0) {
                token = new String(bytes);
            }
        }
        doOpen(uid, region, token.getBytes(), completion, ReportFunction.open);
    }

    /**
     * 切换至指定用户的工作上下文，该方法应该在上层业务的用户登录操作完成之后执行
     *
     * @param uid        用户uid
     * @param region     启用用户上下文所需的区域配置参数
     * @param tagSet     用于消息过滤
     * @param provider   用于服务初始化时所需token的获取
     * @param completion 异步操作回调，可为null，会在操作结束时通过其不同的回调方法返回操作结果。
     *
     * @since 2.10.0
     * @deprecated
     */
    @Deprecated
    public static void open(final long uid,
                            @NonNull final String region,
                            @Nullable final Set<String> tagSet,
                            @Nullable final TokenProvider provider,
                            @Nullable final HMR.Completion completion) {
        HummerEngine.setTokenProvider(provider);
        String token = "";
        if (provider != null) {
            byte[] bytes = provider.getToken(uid);
            if (bytes != null && bytes.length > 0) {
                token = new String(bytes);
            }
        }
        doOpen(uid, region, token.getBytes(), completion, ReportFunction.openWithTokenProvider);
    }

    /**
     * 切换至指定用户的工作上下文，该方法应该在上层业务的用户登录操作完成之后执行
     *
     * @param uid        用户uid
     * @param region     启用用户上下文所需的区域配置参数
     * @param tagSet     用于消息过滤
     * @param token      用于服务初始化时所需token的获取
     * @param completion 异步操作回调，可为null，会在操作结束时通过其不同的回调方法返回操作结果。
     *
     * @deprecated 2.16.0 废弃，由 {@link HMR#open(long, String, String, Completion)} 替代
     */
    @Deprecated
    public static void open(final long uid,
                            @NonNull final String region,
                            @Nullable final Set<String> tagSet,
                            @NonNull final String token,
                            @Nullable final HMR.Completion completion) {
        doOpen(uid, region, token.getBytes(), completion, ReportFunction.openWithToken);
    }

    /**
     * 切换至指定用户的工作上下文，该方法应该在上层业务的用户登录操作完成之后执行
     *
     * @param uid        用户uid
     * @param region     启用用户上下文所需的区域配置参数
     * @param token      用于服务初始化时所需token的获取
     * @param completion 异步操作回调，可为null，会在操作结束时通过其不同的回调方法返回操作结果。
     *
     * @since 2.16.0 新增
     */
    public static void open(final long uid,
                            @NonNull final String region,
                            @NonNull final String token,
                            @Nullable final HMR.Completion completion) {
        Log.i(TAG, ReportFunction.openWithToken + " open uid: " + uid + ", region: " + region);
        doOpen(uid, region, token.getBytes(), completion, ReportFunction.openWithToken);
    }

    private static void doOpenFailed(HMRCompletion completion, String funcName, long now, Error error) {
        hummerStateToClosed();
        HMRContext.reportReturnCode(funcName, now, error);
        CompletionUtils.dispatchFailure(completion, error);
    }

    /**
     * 切换至指定用户的工作上下文，该方法应该在上层业务的用户登录操作完成之后执行
     *
     * @param uid        用户uid
     * @param region     启用用户上下文所需的区域配置参数
     * @param token      用于服务初始化时所需token的获取
     * @param completion 异步操作回调，可为null，会在操作结束时通过其不同的回调方法返回操作结果。
     *
     * @since 2.16.22 新增
     */
    public static void open(final long uid,
                            @NonNull final String region,
                            @NonNull final byte[] token,
                            @Nullable final HMR.Completion completion) {
        Log.i(TAG, ReportFunction.openWithByteToken + " open uid: " + uid
                + ", region: " + region
                + ", token size: " + token.length);
        doOpen(uid, region, token, completion, ReportFunction.openWithByteToken);
    }

    private static void doOpen(final long uid,
                               final String region,
                               final byte[] token,
                               final HMR.Completion completion,
                               final String funcName) {
        final long now = HMRContext.getCurrentTime();
        RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        Log.i(TAG, Trace.method("open").msg("open sdk")
                                        .info("uid", uid)
                                        .info("rid", requestId.getId()));

        final HMRCompletion hmrCompletion = new HMRCompletion(requestId, completion);

        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (!isInitialized()) {
                    CompletionUtils.dispatchFailure(hmrCompletion,
                            new Error(ErrorEnum.UNINITIALIZED_EXCEPTION, "The Hummer SDK has not been initialized, please call 'HMR.init' firstly"));
                    return;
                }
                if (uid < 0) {
                    CompletionUtils.dispatchFailure(hmrCompletion, new Error(ErrorEnum.INVALID_PARAMETER, INVALID_UID));
                    return;
                }

                /*
                 * 这个判断为啥放在android平台层，显然是可以在engine处理的
                 * 原因在于：需要在调用ServiceProvider.openServices之前有这个判断
                 *      当用户登录成功后，ChatStoreService会在openService订阅消息监听，在closeService取消订阅
                 *      而如果该判断在调用ServiceProvider.openServices之后，用户登录成功后，再调用一次登录
                 *      那么openServices会调用成功，但是进入engine层，会判断用户已登录，返回错误，平台层监听到错误
                 *      就会调用 closeService，那就就取消了消息监听的订阅，消息就不会入库了
                 * 故：判断需要提前
                 * 而这个判断提前，则需要在平台层维护好用户登出、被踢、刷新token所以引起的登录用户和HMR.state的
                 */
                if (HMR.me != null || HMR.getState() != State.Closed) {
                    String eMessage = "[P]Duplicate call login, please logout first";
                    if (HMR.getState() == State.Closing) {
                        eMessage = "[P]Please wait close finished";
                    }
                    CompletionUtils.dispatchFailure(hmrCompletion, new Error(ErrorEnum.BAD_USER_ERROR, eMessage));
                    return;
                }

                HummerEngine.addKickOutHandler(KICK_OUT_HANDLER);
                setHummerState(State.Opening);
                HMR.me = new User(uid);
                HMRContext.region = region;
                HMRContext.contextId = hmrCompletion.getRequestId();
                Log.i(TAG, Trace.method("open").msg("open sdk, do it"));

                ServiceProvider.openServices(new HMRCompletion(hmrCompletion.getRequestId())
                        .onSuccess(new OnSuccess() {
                            @Override
                            public void onSuccess() {
                                HMR.Completion completionWrap = new HMR.Completion() {
                                    @Override
                                    public void onSuccess() {
                                        setHummerState(State.Opened);
                                        HMRContext.reportReturnCode(funcName, now);

                                        CompletionUtils.dispatchSuccess(hmrCompletion);
                                    }

                                    @Override
                                    public void onFailed(Error err) {
                                        doOpenFailed(hmrCompletion, funcName, now, err);
                                    }
                                };
                                HummerEngine.login(hmrCompletion.getRequestId(), uid, region, token, completionWrap);
                            }
                        })
                        .onFailure(new OnFailure() {
                            @Override
                            public void onFailure(Error error) {
                                doOpenFailed(hmrCompletion, funcName, now, error);
                            }
                        }));
            }
        });
    }

    /**
     * 关闭Hummer的功能，该操作应该在业务进行实际的用户注销（退出登录）之前执行
     *
     * @param completion 完成关闭的一个回调
     */
    public static void close(@Nullable final Completion completion) {
        final long now = HMRContext.getCurrentTime();
        RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        Log.i(TAG, Trace.method("close").msg("user close")
                .info("rid", requestId.getId()));

        final HMRCompletion hmrCompletion = new HMRCompletion(requestId, completion);

        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (!isInitialized()) {
                    CompletionUtils.dispatchFailure(hmrCompletion,
                            new Error(ErrorEnum.UNINITIALIZED_EXCEPTION, "The Hummer SDK has not been initialized, please call 'HMR.init' firstly"));
                    HMRContext.reportReturnCode(ReportFunction.close, now, ErrorEnum.UNINITIALIZED_EXCEPTION.getCode());
                    return;
                }

                setHummerState(State.Closing);
                HMR.me = null;
                HMRContext.region = null;
                HMRContext.contextId = null;
                HummerEngine.removeKickOutHandler(KICK_OUT_HANDLER);

                ServiceProvider.closeServices();
                HummerEngine.logout(hmrCompletion.getRequestId(), new Completion() {
                    @Override
                    public void onSuccess() {
                        setHummerState(State.Closed);
                        HMRContext.reportReturnCode(ReportFunction.close, now);
                        CompletionUtils.dispatchSuccess(hmrCompletion);
                    }

                    @Override
                    public void onFailed(Error err) {
                        setHummerState(State.Closed);
                        HMRContext.reportReturnCode(ReportFunction.close, now, err);
                        CompletionUtils.dispatchFailure(hmrCompletion, err);
                    }
                });
            }
        });
    }

    /**
     * 获取服务实例
     *
     * <br> Hummer包含了许多服务接口，例如PeerService, RoomService，所有服务的实例都必须通过getService方法来获取
     *
     * @return 如果正确加载、或者确实可以提供某个Service实例，则返回对应的实现实例，否则返回null
     */
    public static <Service> Service getService(@NonNull Class<Service> serviceClass) {
        // ！！该方法是提供给业务调用的，Hummer内部应直接使用{@link ServiceProvider#get}来获取service实例
        return ServiceProvider.get(serviceClass);
    }

    public enum TokenInvalidCode {
        /**
         * 过期
         */
        EXPIRED(2003);

        private int code;

        TokenInvalidCode(int code) {
            this.code = code;
        }

        public int getCode() {
            return code;
        }
    }


    public interface StateListener {
        void onUpdateHummerState(State fromState, State toState);
    }

    public interface TokenInvalidListener {
        @Deprecated
        void onHummerTokenInvalid(TokenInvalidCode code, String desc);

        void onHummerPreviousTokenExpired();
    }

    /**
     * 获取 SDK 当前状态
     *
     * @return SDK当前状态
     */
    public static State getState() {
        HMRContext.reportReturnCode(ReportFunction.HMR_getState, HMRContext.getCurrentTime());
        return state;
    }

    public static void addStateListener(@NonNull final StateListener listener) {
        HMRContext.reportReturnCode(ReportFunction.HMR_addStateListener, HMRContext.getCurrentTime());
        if (listener != null) {
            synchronized (mStateListeners) {
                mStateListeners.add(listener);
            }
            setHummerState(null);
        }
    }

    public static void removeStateListener(@NonNull final StateListener listener) {
        HMRContext.reportReturnCode(ReportFunction.HMR_removeStateListener, HMRContext.getCurrentTime());
        if (listener != null) {
            synchronized (mStateListeners) {
                mStateListeners.remove(listener);
            }
        }
    }

    private static void setHummerState(@Nullable final HMR.State state) {
        HMR.State fromState = HMR.state;
        if (state != null) {
            HMR.state = state;
        }

        notifyUpdateHummerState(fromState, HMR.state);
    }

    private static void notifyUpdateHummerState(final HMR.State fromState, final HMR.State toState) {
        DispatchQueue.main.sync(new Runnable() {
            @Override
            public void run() {
                synchronized (mStateListeners) {
                    for (final StateListener l : mStateListeners) {
                        l.onUpdateHummerState(fromState, toState);
                    }
                }
            }
        });
    }

    public static void addTokenInvalidListener(@NonNull final TokenInvalidListener listener) {
        final long now = HMRContext.getCurrentTime();
        HummerEngine.addTokenInvalidListener(listener);
        HMRContext.reportReturnCode(ReportFunction.addTokenInvalidListener, now);
    }

    public static void removeTokenInvalidListener(@NonNull final TokenInvalidListener listener) {
        final long now = HMRContext.getCurrentTime();
        HummerEngine.removeTokenInvalidListener(listener);
        HMRContext.reportReturnCode(ReportFunction.removeTokenInvalidListener, now);
    }

    public interface HummerListener {
        void onConnectionStateChanged(ConnectionState fromState, ConnectionState toState, String reason);

        void onHummerTokenWillExpired();

        /**
         * 账号被踢通知
         *
         * @param code   被踢错误码
         * @param reason 被踢原因
         */
        void onForceoutOffline(int code, String reason);

        /**
         * 当收到P2P信令消息时，会收到该事件的通知
         *
         * @param sender  发送者
         * @param message 消息
         */
        void onP2PTextMessageReceived(User sender, TextMessage message);
    }

    public static void addHummerListener(@NonNull final HummerListener listener) {
        final long now = HMRContext.getCurrentTime();
        HummerEngine.addHummerListener(listener);
        HMRContext.reportReturnCode(ReportFunction.addHummerListener, now);
    }

    public static void removeHummerListener(@NonNull final HummerListener listener) {
        final long now = HMRContext.getCurrentTime();
        HummerEngine.removeHummerListener(listener);
        HMRContext.reportReturnCode(ReportFunction.removeHummerListener, now);
    }

    public static ConnectionState getConnectionState() {
        final long now = HMRContext.getCurrentTime();
        ConnectionState state = HummerEngine.getConnectionState();
        HMRContext.reportReturnCode(ReportFunction.getConnectionState, now);
        return state;
    }

    /**
     * 获取 SDK 版本号
     *
     * @return SDK版本号
     */
    public static String getVersion() {
        final long now = HMRContext.getCurrentTime();
        String version = HummerEngine.getSdkVersion();
        HMRContext.reportReturnCode(ReportFunction.getVersion, now);
        return version;
    }

    /**
     * 获取当前Hummer工作上下文的用户对象
     *
     * @return 如果已经成功通过HMR.open方法启用了用户上下文，则返回对应的用户实例，否则返回null。可以通过返回值来判断Hummer是否已经
     * 正确开启了用户上下文。
     */
    public static User getMe() {
        return me;
    }

    /**
     * 判断某个标识对象是否当前Hummer的工作用户
     *
     * @param identity 欲比对的标识对象
     *
     * @return 如果identity对象实际User类型，且其id值与当前Hummer用户的id值一致，则返回true，否则返回false
     */
    public static boolean isMe(Identifiable identity) {
        return Identifiable.equals(me, identity);
    }

    public interface Completion {
        void onSuccess();

        void onFailed(Error err);
    }

    public interface CompletionArg<Argument> {
        void onSuccess(Argument arg);

        void onFailed(Error err);
    }

    public interface CompletionArgs<Argument1, Argument2> {
        void onSuccess(Argument1 arg1, Argument2 arg2);

        void onFailed(Error err);
    }

    /**
     * 刷新用户凭证
     *
     * @param token 待刷新的用户凭证
     *
     * @since 2.10.0
     * @deprecated
     */
    @Deprecated
    public static void refreshToken(@NonNull final String token) {
        final long now = HMRContext.getCurrentTime();
        HMR.Completion completion = new HMR.Completion() {
            @Override
            public void onSuccess() {
                HMRContext.reportReturnCode(ReportFunction.refreshToken, now);
            }

            @Override
            public void onFailed(Error err) {
                hummerStateToClosed();
                HMRContext.reportReturnCode(ReportFunction.refreshToken, now, err);
            }
        };
        HummerEngine.refreshToken(token, completion);
    }


    /**
     * 刷新用户凭证
     *
     * @param token      待刷新的用户凭证
     * @param completion 异步回调
     *
     * @return 递增的一个请求标识，主要用于查障。
     *
     * @since 2.12.0
     * @deprecated
     */
    @Deprecated
    public static void refreshToken(@NonNull final String token,
                                    @Nullable final HMR.Completion completion) {
        Log.i(TAG, ReportFunction.refreshTokenWithCompletion + "refreshToken: " + token.length());

        final long now = HMRContext.getCurrentTime();
        RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        HMRCompletion hmrCompletion = new HMRCompletion(requestId, completion);
        if (!isInitialized()) {
            CompletionUtils.dispatchFailure(hmrCompletion,
                    new Error(ErrorEnum.UNINITIALIZED_EXCEPTION, "The Hummer SDK has not been initialized, please call 'HMR.init' firstly"));
            return;
        }

        HMR.Completion completionWrap = new HMR.Completion() {
            @Override
            public void onSuccess() {
                HMRContext.reportReturnCode(ReportFunction.refreshTokenWithCompletion, now);
                if (completion != null) {
                    completion.onSuccess();
                }
            }

            @Override
            public void onFailed(Error err) {
                hummerStateToClosed();
                HMRContext.reportReturnCode(ReportFunction.refreshTokenWithCompletion, now, err);
                if (completion != null) {
                    completion.onFailed(err);
                }
            }
        };
        HummerEngine.refreshToken(token, completionWrap);
    }

    /**
     * 刷新用户凭证
     *
     * @param token      待刷新的用户凭证
     * @param completion 异步回调
     */
    public static void refreshToken1(@NonNull final String token,
                                     @Nullable final HMR.Completion completion) {
        Log.i(TAG, ReportFunction.refreshToken1WithCompletion + "refreshToken: " + token.length());

        final long now = HMRContext.getCurrentTime();
        RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        HMRCompletion hmrCompletion = new HMRCompletion(requestId, completion);
        if (!isInitialized()) {
            CompletionUtils.dispatchFailure(hmrCompletion,
                    new Error(ErrorEnum.UNINITIALIZED_EXCEPTION, "The Hummer SDK has not been initialized, please call 'HMR.init' firstly"));
            return;
        }
        HMR.Completion completionWrap = new HMR.Completion() {
            @Override
            public void onSuccess() {
                HMRContext.reportReturnCode(ReportFunction.refreshToken1WithCompletion, now);
                if (completion != null) {
                    completion.onSuccess();
                }
            }

            @Override
            public void onFailed(Error err) {
                HMRContext.reportReturnCode(ReportFunction.refreshToken1WithCompletion, now, err);
                if (completion != null) {
                    completion.onFailed(err);
                }
            }
        };
        HummerEngine.refreshToken1(token, completionWrap);
    }

    /**
     * 刷新用户凭证
     *
     * @param token      待刷新的用户凭证
     * @param completion 异步回调
     *
     * @since 2.16.21
     */
    public static void refreshToken1(@NonNull final byte[] token,
                                     @Nullable final HMR.Completion completion) {
        Log.i(TAG, ReportFunction.refreshByteToken1WithCompletion + "refreshToken: " + token.length);

        final long now = HMRContext.getCurrentTime();
        RequestId requestId = new RequestId(RequestIdBuilder.generateRequestId());
        HMRCompletion hmrCompletion = new HMRCompletion(requestId, completion);
        if (!isInitialized()) {
            CompletionUtils.dispatchFailure(hmrCompletion,
                    new Error(ErrorEnum.UNINITIALIZED_EXCEPTION, "The Hummer SDK has not been initialized, please call 'HMR.init' firstly"));
            return;
        }
        HMR.Completion completionWrap = new HMR.Completion() {
            @Override
            public void onSuccess() {
                HMRContext.reportReturnCode(ReportFunction.refreshByteToken1WithCompletion, now);
                if (completion != null) {
                    completion.onSuccess();
                }
            }

            @Override
            public void onFailed(Error err) {
                HMRContext.reportReturnCode(ReportFunction.refreshByteToken1WithCompletion, now, err);
                if (completion != null) {
                    completion.onFailed(err);
                }
            }
        };
        HummerEngine.refreshToken1(token, completionWrap);
    }

    public enum HMRLogLevel {
        /**
         * 输出所有级别日志
         */
        HMR_LOG_LEVEL_VERBOSE(0),

        /**
         * 输出调试信息及以上级别日志
         */
        HMR_LOG_LEVEL_DEBUG(1),

        /**
         * 输出关键信息及以上级别日志
         */
        HMR_LOG_LEVEL_INFO(2),

        /**
         * 输出告警信息及以上级别日志
         */
        HMR_LOG_LEVEL_WARNING(3),

        /**
         * 输出错误信息及以上级别日志
         */
        HMR_LOG_LEVEL_ERROR(4),

        /**
         * 输出release及以上级别日志
         */
        HMR_LOG_LEVEL_RELEASE(10);

        private int level;

        HMRLogLevel(int level) {
            this.level = level;
        }

        public int getLevel() {
            return level;
        }

        public static HMRLogLevel toLevel(int level) {
            for (HMRLogLevel type : HMRLogLevel.values()) {
                if (type.getLevel() == level) {
                    return type;
                }
            }
            return HMR_LOG_LEVEL_VERBOSE;
        }
    }

    /**
     * 设置日志的保存路径
     *
     * <br> 请不要使用根目录当日志路径
     * <br> 最好在调用{@link HMR#init}之前设置
     * <br> 请尽量保持设置的路径是不变的
     *
     * @param loggerPath 日志路径
     *
     * @return true: 设置成功
     */
    public static boolean setLoggerFilePath(@NonNull String loggerPath) {
        long now = HMRContext.getCurrentTime();
        if (TextUtils.isEmpty(loggerPath)) {
            Log.e(TAG, Trace.method("setLoggerFilePath").msg("path must be not null").info("path", loggerPath));
            HMRContext.reportReturnCode(ReportFunction.SET_LOGGER_FILE_PATH, now, ErrorEnum.INVALID_PARAMETER.getCode());
            return false;
        }

        Log.i(TAG, "setLoggerFilePath: " + loggerPath);
        HummerLog.instance().setLogFilePath(loggerPath);
        HMRContext.reportReturnCode(ReportFunction.SET_LOGGER_FILE_PATH, now);
        return true;
    }

    /**
     * 设置日志输出级别
     *
     * @param logLevel 日志等级
     *
     * @return true: 方法调用成功；false：方法调用失败
     *
     * @remark 不调用此接口，使用默认日志等级 HMR_LOG_LEVEL_VERBOSE
     */
    public static boolean setLogLevel(HMRLogLevel logLevel) {
        long now = HMRContext.getCurrentTime();

        if (logLevel == null) {
            Log.e(TAG, Trace.method("setLogLevel").msg("level must be not null"));
            HMRContext.reportReturnCode(ReportFunction.SET_LOG_LEVEL, now, ErrorEnum.INVALID_PARAMETER.getCode());
            return false;
        }
        Log.i(TAG, Trace.method("setLogLevel").info("level", logLevel.getLevel()));
        HummerLog.instance().setLogLevel(logLevel.getLevel());

        HMRContext.reportReturnCode(ReportFunction.SET_LOG_LEVEL, now);
        return true;
    }

    /**
     * 手动上报日志
     *
     * @param remark 上报时的备注信息
     *
     * @return true: 方法调用成功；false：方法调用失败
     */
    public static boolean uploadLogsManually(String remark) {
        long now = HMRContext.getCurrentTime();

        HummerLog.instance().uploadLog(remark);

        HMRContext.reportReturnCode(ReportFunction.UPLOAD_LOGS_MANUALLY, now);
        return true;
    }

    /**
     * 设置日志输出回调
     *
     * @param callback 回调接口实例
     *
     * @return true: 方法调用成功；false：方法调用失败
     *
     * @remark SDK回调日志消息给应用程序，由应用程序代理输出日志文件
     */
    public static boolean setLogCallback(HMRLogCallback callback) {
        long now = HMRContext.getCurrentTime();

        HummerLog.instance().setLogCallback(callback);

        HMRContext.reportReturnCode(ReportFunction.SET_LOG_CALLBACK, now);
        return true;
    }

    public interface HMRLogCallback {
        /**
         * 日志输出回调，设置此回调后，日志信息将通过此接口回调
         *
         * @param level 待输出日志级别
         * @param msg   待输出日志内容
         */
        void onHmrLogWithLevel(HMRLogLevel level, String msg);
    }

    /**
     * 创建文本消息
     *
     * @param text 文本内容
     *
     * @return TextMessage
     */
    public static TextMessage createTextMessage(String text) {
        return HummerEngine.createTextMessage(text);
    }

    /**
     * 发送P2P单播消息
     *
     * @param receiver   信令消息的接受者标识
     * @param message    需要发送的信令消息
     * @param options    发送消息所需要的配置信息
     * @param completion 回调
     */
    public static void sendP2PMessage(@NonNull User receiver,
                                      @NonNull BaseMessage message,
                                      P2PMessageOptions options,
                                      HMR.Completion completion) {
        HummerEngine.sendP2PMessage(receiver, message, options, completion);
    }

    /**
     * 批量查询用户在线状态，可以查询具体用户是否在线
     * <br> 批量查询的最大量为200
     *
     * @param users      需要查询是否在线的用户ID列表
     * @param completion 回调
     */
    public static void fetchUserOnlineStatus(@NonNull Set<User> users,
                                             HMR.CompletionArg<Set<UserOnlineStatus>> completion) {
        HummerEngine.fetchUserOnlineStatus(users, completion);
    }

    private final static HummerEngine.KickOutHandler KICK_OUT_HANDLER = new HummerEngine.KickOutHandler() {

        @Override
        public void onHummerKickedResult(Error error) {
            hummerStateToClosed();
        }
    };

    private static void hummerStateToClosed() {
        HMR.me = null;
        HMRContext.region = null;
        setHummerState(State.Closed);
    }

    private static final String TAG = "HMR";

    private static State state = State.Unavailable;

    private static TokenProvider mTokenProvider = null;

    private static final CopyOnWriteArraySet<StateListener> mStateListeners = new CopyOnWriteArraySet<>();

    private static User me = null;
}
