package com.hummer.im;

import java.util.HashSet;
import java.util.Set;

/**
 * 错误枚举
 *
 * @author hummer组
 */
public enum ErrorEnum {

    /**
     * kOk
     */
    OK(0, "成功"),

    /**
     * kUnknownError
     */
    UNKNOWN_ERROR(1, "未知错误"),

    /**
     * KLoginKickOut = 100; // 登录互踢
     */
    LOGIN_KICK_OUT(100, "您的帐号在别处登录，您被迫下线"),

    TERMINAL_LIMIT(101, "超出终端类限制，被其他端踢下线"),

    DEVICE_COUNT_LIMIT(102, "超出设备数限制，被其他端踢下线"),

    /* 客户端错误 */

    /**
     * 通用客户端错误，如果在下面的错误码表中没有更匹配的错误类型，则会使用该错误码进行表示
     */
    CLIENT_EXCEPTIONS(1000, "客户端错误"),

    /**
     * kUninitializedException = 1001;
     */
    UNINITIALIZED_EXCEPTION(1001, "没有初始化"),

    /**
     * 请求Hummer服务时，提供了无效的错误参数
     */
    SDK_INVALID_PARAMETERS(1002, "参数有误"),

    /**
     * kDBError
     */
    DB_ERROR(1003, "访问本地数据库错误"),

    /**
     * 如果发起需要网络服务，但是请求时网络不可达，则会返回该错误，建议提示用户检查网络连接并重试
     */
    NETWORK_NOT_FOUND(1004, "网络不可达"),

    /**
     * kOperationTimeout
     */
    OPERATION_TIMEOUT(1005, "操作超时"),

    /**
     * 网络连接超时，通常是由于网络虽然连通，但是因为网络条件比较恶劣导致的，建议提示用户检查网络连接并重试
     */
    CONNECTION_TIMEOUT(1006, "网络连接超时"),

    /**
     * kConnectionException
     */
    CONNECTION_EXCEPTION(1007, "连接异常"),

    /**
     * 服务请求调用过于频繁，建议业务端需要进行频率控制
     */
    THROTTLING(1008, "服务请求调用过于频繁"),

    /**
     * kUnauthorizedException = 1009; // 没有登录
     */
    UNAUTHORIZED_EXCEPTION(1009, "鉴权失败"),

    /**
     * Hummer内调用第三方服务发生错误
     */
    THIRD_PARTY_SERVICE_ERROR(1010, "调用第三方服务发生错误"),

    /**
     * kBadUserError = 1011;
     * 如果业务没有正确处理用户上下文切换，例如业务已经注销了用户，但是没有调用Hummer.close，则会产生该错误
     */
    BAD_USER_ERROR(1011, "未正确切换用户上下文"),


    /* 通用错误 */

    /**
     * 传输协议异常，例如协议版本错误等
     */
    PROTOCOL_EXCEPTIONS(2000, "传输协议异常"),

    /**
     * kInvalidParameter
     */
    INVALID_PARAMETER(2001, "无效参数"),

    /**
     * kInvalidTokenException
     */
    INVALID_TOKEN_EXCEPTION(2002, "Token校验错误"),

    /**
     * kExpiredTokenException
     */
    EXPIRED_TOKEN_EXCEPTION(2003, "Token过期"),

    /**
     * kResourceNotFoundException
     */
    RESOURCE_NOT_FOUND_EXCEPTION(2004, "资源不存在"),

    /**
     * 请求访问的资源已存在，通常在创建房间等场景出现
     */
    RESOURCE_ALREADY_EXIST(2005, "请求访问的资源已存在"),

    /**
     * 资源、关系数量超出了限定值
     */
    LIMIT_EXCEEDED(2006, "资源、关系数量超出了限定值"),

    /**
     * kMsgSizeLimitExceeded
     */
    MSG_SIZE_LIMIT_EXCEEDED(2007, "消息长度超上限"),

    COUNT_LIMIT_EXCEEDED(2008, "数量超出了上限"),

    /**
     * kAppidUnavailable = 2018;   // appid不可用
     */
    APP_ID_UNAVAILABLE(2018, "无效的appId"),

    /**
     * kPermissionDeny = 3000; //权限验证失败
     */
    PERMISSION_DENY(3000, "权限验证失败"),

    /**
     * 用户被列入黑名单，无法获得CIM服务
     */
    BLACK_LISTED(3001, "用户被列入黑名单"),

    TEMPORARILY_DENIED_EXCEPTION(3002, "暂时没有权限"),

    FORBIDDEN_EXCEPTION(3003, "操作被禁止"),

    USER_FORBIDDEN_EXCEPTION(3004, "用户操作被禁止"),

    BANNED_EXCEPTION(3005, "操作被封禁"),

    CHALLENGE_EXCEPTION(3006, "需要输入参数进行验证"),

    /**
     * kContentCensorshipDeny = 3007;   //内容审查失败
     */
    CONTENT_CENSORSHIP_DENY(3007, "内容审查失败"),

    OPERATED_BY_MYSELF_DENY(3011, "不允许自己操作自己"),

    OWNER_CANNOT_BE_OPERATED(3012, "owner不允许被操作"),

    /**
     * kInternalServerError = 4000; //服务器错误
     */
    INTERNAL_SERVER_ERROR(4000, "服务器错误"),

    /**
     * 暂时无法提供服务，一般为服务器进程重启等原因导致
     */
    SERVICE_UNAVAILABLE(4001, "暂时无法提供服务"),

    BUSINESS_SERVER_ERROR(4002, "业务服务异常"),

    /**
     * kOperationTooOften = 4003; //操作过于频繁
     */
    OPERATION_TOO_OFTEN(4003, "操作过于频繁"),

    REPETITIVE_OPERATION(4004, "重复操作"),

    /* ChatRoom错误码 */

    CHAT_ROOM_NOT_EXIST(5001, "聊天室不存在"),

    OPERATOR_USER_NOT_IN_CHAT_ROOM(5003, "操作者不在聊天室"),

    OPERATED_USER_NOT_IN_CHAT_ROOM(5004, "被操作者不在聊天室"),

    CHAR_ROOM_USER_NOT_ADMIN(5012, "用户不是管理员"),

    USER_NOT_IN_CHANNEL(5101, "用户不在频道"),

    CHANNEL_SIZE_LIMIT_EXCEEDED(5109, "频道数量超过上限"),

    CHANNEL_USER_ALREADY_JOINED(5110, "重复加入");

    //业务自定义错误(大于等于10000)
    //kCustomErrorStart = 10000;

    private final int code;

    private final String desc;

    ErrorEnum(int code, String desc) {
        this.code = code;
        this.desc = desc;
    }

    public int getCode() {
        return code;
    }

    public String getDesc() {
        return desc;
    }

    private static final Set<Integer> codeSet = new HashSet<>();

    static {
        for (ErrorEnum errorEnum : ErrorEnum.values()) {
            codeSet.add(errorEnum.getCode());
        }
    }

    /**
     * 判读errorCode是否在枚举值之内
     *
     * <br> 业务自定义返回码，直接返回true
     *
     * @param errorCode 错误码
     * @return true: 包含；false: 不包含
     */
    public static boolean contains(int errorCode) {
        if (errorCode >= CUSTOM_ERROR_START) {
            return true;
        }
        return codeSet.contains(errorCode);
    }

    /**
     * 业务自定义错误码从 10000 开始
     */
    private static final int CUSTOM_ERROR_START = 10000;
}
