package com.hummer.im._internals;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.services.user.UserService;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.db.DBActions;
import com.hummer.im.db.DBService;
import com.hummer.im.model.Chat;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.states.Failed;
import com.hummer.im.model.chat.states.Preparing;
import com.hummer.im.model.chat.store.FetchingClauses;
import com.hummer.im.model.chat.store.RemovingClauses;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccess;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.RichCompletionArg;
import com.hummer.im.model.id.AppSession;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.model.id.User;
import com.hummer.im.service.ChatService;
import com.hummer.im.service.ChatStoreService;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.hummer.im.Error.Code.InvalidParameters;

public final class ChatStoreServiceImpl implements ChatStoreService,
        ChatService.MessageListener,
        ChatService.StateListener,
        ServiceProvider.Service {

    private static final String TAG = "ChatStoreService";

    // Mark - Implements: ServiceProvider.Service

    @Override
    public Class[] staticDependencies() {
        return new Class[]{ChatService.class};
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class, ChatUpgradeService.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return new Class[]{ChatService.class};
    }

    @Override
    public void initService() {
    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        // 1. 缓存会话到内存
        DBService.Action loadConversations = new DBActions()
                .createTableIfNeeded(BeanConversation.class)
                .queryAll(BeanConversation.class,
                        new DBActions.QueryAcceptor<BeanConversation>() {
                            @Override
                            public void onQueryResults(List<BeanConversation> indexes) {
                                for (BeanConversation index : indexes) {
                                    Chat c = BeanConversation.toConversation(index);
                                    chats.put(c.getTarget(), c);
                                }
                            }
                        });

        // 2. 清理生命周期不完整的遗留消息，将它们都标记为失败状态
        DBService.Action handlePendingMessages = new DBActions()
                .createTableIfNeeded(BeanMessage.pendingConfig())
                .custom(new ActGetPendingChatMessages(new ActGetPendingChatMessages.Acceptor() {
                    @Override
                    public void onAcquired(List<Message> messages) {
                        Error recovered = new Error(Error.Code.UndefinedExceptions, "从异常中恢复", null);
                        for (Message message : messages) {
                            message.setState(new Failed(recovered));

                            Identifiable target = message.getTarget();
                            if (target == null) {
                                return;
                            }

                            Chat chat = ChatStoreServiceImpl.this.getChat(target);
                            if (chat != null) {
                                ChatStoreServiceImpl.this.addOrUpdateMessage(chat, message, null);
                            }
                        }
                    }
                }))
                .deleteAll(BeanMessage.pendingConfig());

        HMR.getService(DBService.class).execute(
                new DBActions()
                        // 尝试修复旧版本导致的有会话数据
                        .custom(new ActFixConversationsIfNeeded())

                        .custom(loadConversations)
                        .custom(handlePendingMessages)
                        .run("Hummer.Chat.AddListener", new Runnable() {
                            @Override
                            public void run() {
                                // 3. Add listener only when succeed
                                HMR.getService(ChatService.class).addStateListener(null,
                                        ChatStoreServiceImpl.this
                                );

                                HMR.getService(ChatService.class).addMessageListener(null, ChatStoreServiceImpl.this);

                                ChatStoreServiceImpl.this.notifyUpdateConversations();
                            }
                        }),
                completion);
    }

    @Override
    public void closeService() {
        HMR.getService(ChatService.class).removeMessageListener(null, ChatStoreServiceImpl.this);
        chats.clear();
        notifyUpdateConversations();
    }

    // Mark - Implements: ChatStoreService

    @Override
    public void createChat(@NonNull Identifiable target, @NonNull final HMR.CompletionArg<Chat> completion) {
        final RichCompletionArg rCompletion = new RichCompletionArg(completion, "ChatStoreService::createChat");
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        Chat chat = getChat(target);

        if (chat == null) {
            chat = new Chat(target, null, 0L, null, System.currentTimeMillis());

            final Chat finalChat = chat;
            addChat(chat, new HMR.Completion() {
                @Override
                public void onSuccess() {
                    CompletionUtils.dispatchSuccess(rCompletion, finalChat);
                }

                @Override
                public void onFailed(Error error) {
                    CompletionUtils.dispatchFailure(rCompletion, error);
                }
            });
        } else {
            CompletionUtils.dispatchFailure(rCompletion, new Error(InvalidParameters, "会话已存在"));
        }
    }

    @Override
    public @Nullable
    Chat getChat(@NonNull Identifiable target) {
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            Log.e(TAG, Trace.once("getChat").msg("User not login, or using anonymous mode"));
            return null;
        }

        return chats.get(target);
    }

    @Override
    public @NonNull
    List<Chat> getChats() {
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            Log.e(TAG, Trace.once("getChats").msg("User not login, or using anonymous mode"));
            return new ArrayList<>();
        }

        return new ArrayList<>(chats.values());
    }

    @Override
    public void addChat(@NonNull final Chat chat, @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatStoreService::addChat");
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        final Identifiable target = chat.getTarget();
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .guard(new Error(InvalidParameters, "会话已存在"),
                                new DBActions.Guarder() {
                                    @Override
                                    public boolean shouldBreak() {
                                        return chats.containsKey(target);
                                    }
                                })
                        .createTableIfNeeded(BeanMessage.conversationConfig(target))
                        .create(BeanConversation.fromConversation(chat), null)
                        .run("Hummer.AddConversation", new Runnable() {
                            @Override
                            public void run() {
                                chats.put(target, chat);
                                ChatStoreServiceImpl.this.notifyAfterAddingConversation(chat);
                            }
                        }),
                rCompletion);
    }

    @Override
    public void removeChat(@NonNull final Chat chat, @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatStoreService::removeChat");
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        final Identifiable target = chat.getTarget();
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .guard(new Error(InvalidParameters, "会话不存在"),
                                new DBActions.Guarder() {
                                    @Override
                                    public boolean shouldBreak() {
                                        return !chats.containsKey(target);
                                    }
                                })
                        .delete(BeanConversation.fromConversation(chat), null)
                        .dropTableIfExist(BeanMessage.conversationConfig(target))
                        .run("Hummer.DetachConversation", new Runnable() {
                            @Override
                            public void run() {
                                // 1. 从内存缓存中进行移除
                                chats.remove(target);
                                ChatStoreServiceImpl.this.notifyAfterRemovingConversation(chat);
                            }
                        }),
                rCompletion);
    }

    @Override
    public void updateChat(@NonNull final Chat chat, @NonNull HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatStoreService::updateChat");
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        final Identifiable target = chat.getTarget();
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .guard(new Error(InvalidParameters, "会话不存在"),
                                new DBActions.Guarder() {
                                    @Override
                                    public boolean shouldBreak() {
                                        return !chats.containsKey(target);
                                    }
                                })
                        .update(BeanConversation.fromConversation(chat), null)
                        .run("Hummer.NotifyUpdateConversation", new Runnable() {
                            @Override
                            public void run() {
                                ChatStoreServiceImpl.this.notifyUpdateConversation(chat);
                            }
                        }),
                rCompletion);
    }

    @Override
    public void resetUnread(@NonNull final Chat chat, @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatStoreService::resetUnread");
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        chat.setUnreadNum(0L);
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .custom(new ActResetChatUnreadNum(chat.getTarget()))
                        .run("Hummer.resetUnread", new Runnable() {
                            @Override
                            public void run() {
                                chats.put(chat.getTarget(), chat);
                                ChatStoreServiceImpl.this.notifyUpdateConversation(chat);
                            }
                        }), rCompletion);
    }

    @Override
    public void addOrUpdateMessage(@NonNull final Chat chat,
                                   @NonNull final Message message,
                                   @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatStoreService::addOrUpdateMessage");
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        HMR.getService(DBService.class).execute(new ActAddOrUpdateMessage(chat, message,
                new ActAddOrUpdateMessage.Callback() {
                    @Override
                    public void afterSaved(boolean isAdding) {
                        if (isAdding) {
                            // 当发送消息用户非当前登录用户
                            if (!HMR.isMe(message.getSender())) {
                                chat.setUnreadNum(chat.getUnreadNum() + 1);
                            }

                            chat.setLatestMsg(message);
                            chat.setTimestamp(message.getTimestamp());
                            chats.put(chat.getTarget(), chat);

                            HMR.getService(ChatStoreService.class).updateChat(chat,
                                    new HMR.Completion() {
                                        @Override
                                        public void onSuccess() {
                                            ChatStoreServiceImpl.this.notifyAfterMessageAdding(chat, message);
                                        }

                                        @Override
                                        public void onFailed(Error err) {

                                        }
                                    });
                        } else {
                            if (chat.getLatestMsg().getUuid().equals(message.getUuid())) {
                                HMR.getService(ChatStoreService.class).updateChat(chat,
                                        new HMR.Completion() {
                                            @Override
                                            public void onSuccess() {
                                                ChatStoreServiceImpl.this.notifyUpdateMessage(chat, message);
                                            }

                                            @Override
                                            public void onFailed(Error err) {

                                            }
                                        });
                            } else {
                                ChatStoreServiceImpl.this.notifyUpdateMessage(chat, message);
                            }
                        }
                    }
                }), rCompletion);
    }

    @Override
    public void fetchMessages(@NonNull Chat chat,
                              @NonNull FetchingClauses clauses,
                              @Nullable final HMR.CompletionArg<List<Message>> completion) {
        final RichCompletionArg rCompletion = new RichCompletionArg(completion, "ChatStoreService::fetchMessages");
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        final ActFetchMessages fetch = new ActFetchMessages(chat, clauses);
        HMR.getService(DBService.class).execute(fetch, new RichCompletion("ChatStoreService::fetchMessages")
                .onSuccess(new OnSuccess() {
                    @Override
                    public void onSuccess() {
                        CompletionUtils.dispatchSuccess(rCompletion, fetch.results);
                    }
                })
                .onFailure(new OnFailure() {
                    @Override
                    public void onFailure(Error error) {
                        CompletionUtils.dispatchFailure(rCompletion, error);
                    }
                }));
    }

    @Override
    public void removeMessages(@NonNull final Chat chat,
                               @NonNull final RemovingClauses clauses,
                               @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion, "ChatStoreService::removeMessages");
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        if (clauses.messages != null) {
            notifyBeforeMessagesRemoving(chat, clauses.messages);
        }
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .custom(new ActRemoveMessages(chat, clauses))
                        .custom(new ActUpdateChatLatestMsg(chat))
                        .run("Hummer.NotifyAfterRemovingMessages", new Runnable() {
                            @Override
                            public void run() {
                                if (clauses.messages == null) {
                                    ChatStoreServiceImpl.this.notifyAfterMessagesClear(chat);
                                }
                            }
                        }), rCompletion);
    }

    @Override
    public void addChatListener(@NonNull final ChatListener listener) {
        HMRContext.work.async("ChatStore::addChatListener", new Runnable() {
            @Override
            public void run() {
                chatListeners.add(listener);
                Log.i(TAG, Trace.once().method("addChatListener")
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", chatListeners.size()));
            }
        });
    }

    @Override
    public void removeChatListener(@NonNull final ChatListener listener) {
        HMRContext.work.async("ChatStore::removeChatListener", new Runnable() {
            @Override
            public void run() {
                chatListeners.remove(listener);
                Log.i(TAG, Trace.once().method("removeChatListener")
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", chatListeners.size()));
            }
        });
    }

    @Override
    public void addMessageListener(@Nullable final Chat chat, @NonNull final MessageListener listener) {
        HMRContext.work.async("ChatStore::addMessageListener", new Runnable() {
            @Override
            public void run() {
                if (chat == null) {
                    commonMsgListeners.add(listener);
                    return;
                }

                Set<MessageListener> listeners = msgListeners.get(chat.getTarget());
                if (listeners == null) {
                    listeners = new HashSet<>();
                    msgListeners.put(chat.getTarget(), listeners);
                }

                listeners.add(listener);

                Log.i(TAG, Trace.once().method("addMessageListener")
                        .info("chat", chat)
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", ChatStoreServiceImpl.this.chatListeners.size()));
            }
        });
    }

    @Override
    public void removeMessageListener(@Nullable final Chat chat, @NonNull final MessageListener listener) {
        HMRContext.work.async("ChatStore::removeMessageListener", new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.once().method("removeMessageListener")
                        .info("chat", chat)
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", ChatStoreServiceImpl.this.chatListeners.size()));

                if (chat == null) {
                    commonMsgListeners.remove(listener);
                    return;
                }

                Set<MessageListener> listeners = msgListeners.get(chat.getTarget());
                if (listeners == null) {
                    return;
                }

                listeners.remove(listener);
            }
        });
    }

    private void notifyAfterAddingConversation(@NonNull final Chat chat) {
        DispatchQueue.main.async("ChatStore::notifyAfterAddingConversation", new Runnable() {
            @Override
            public void run() {
                for (ChatListener l : chatListeners) {
                    l.afterCreatingChat(chat);
                    l.afterUpdateChats(new ArrayList<>(chats.values()));
                }
            }
        });
    }

    private void notifyAfterRemovingConversation(@NonNull final Chat chat) {
        DispatchQueue.main.async("ChatStore::notifyAfterRemovingConversation", new Runnable() {
            @Override
            public void run() {
                for (ChatListener l : chatListeners) {
                    l.beforeRemovingChat(chat);
                    l.afterUpdateChats(new ArrayList<>(chats.values()));
                }
            }
        });
    }

    private void notifyUpdateConversations() {
        DispatchQueue.main.sync("ChatStore::notifyUpdateConversations", new Runnable() {
            @Override
            public void run() {
                for (ChatListener l : chatListeners) {
                    l.afterUpdateChats(new ArrayList<>(chats.values()));
                }
            }
        });
    }

    private void notifyUpdateConversation(@NonNull final Chat chat) {
        if (chatListeners.size() == 0) {
            return;
        }

        DispatchQueue.main.sync("ChatStore::notifyUpdateConversation", new Runnable() {
            @Override
            public void run() {
                for (ChatListener l : chatListeners) {
                    l.afterUpdateChat(chat);
                }
            }
        });
    }

    interface MessageListenersVisitor {
        void visit(MessageListener listener);
        String visitorName();
    }

    private void iterateMessageListeners(final Chat chat, final MessageListenersVisitor visitor) {
        DispatchQueue.main.sync("ChatStore::" + visitor.visitorName(), new Runnable() {
            @Override
            public void run() {
                // 避免同一个listener被通知多次，先将它们收集到一个set内，然后在发起访问
                Set<MessageListener> listeners = new HashSet<>(commonMsgListeners);

                Set<MessageListener> specificListeners = msgListeners.get(chat.getTarget());
                if (specificListeners != null) {
                    listeners.addAll(specificListeners);
                }

                for (MessageListener l : listeners) {
                    visitor.visit(l);
                }
            }
        });
    }

    private void notifyAfterMessageAdding(@NonNull final Chat chat, @NonNull final Message message) {
        iterateMessageListeners(chat, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.afterAddingMessage(chat, message);
            }

            @Override
            public String visitorName() {
                return "notifyAfterMessageAdding";
            }
        });
    }

    private void notifyUpdateMessage(@NonNull final Chat chat, final Message message) {
        iterateMessageListeners(chat, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.afterUpdateMessage(chat, message);
            }

            @Override
            public String visitorName() {
                return "notifyUpdateMessage";
            }
        });
    }

    private void notifyBeforeMessagesRemoving(@NonNull final Chat chat, final List<Message> messages) {
        iterateMessageListeners(chat, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.beforeRemovingMessages(chat, messages);
            }

            @Override
            public String visitorName() {
                return "notifyBeforeMessagesRemoving";
            }
        });
    }

    private void notifyAfterMessagesClear(@NonNull final Chat chat) {
        iterateMessageListeners(chat, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.afterClearMessages(chat);
            }

            @Override
            public String visitorName() {
                return "notifyAfterMessagesClear";
            }
        });
    }

    @Override
    public void setChatFilter(Filter filter) {
        this.filter = filter;
    }

    // Mark - ChatService.Listener

    @Override
    public void beforeSendingMessage(@NonNull final Message message) {
        HMRContext.work.async("ChatStore::beforeSendingMessage", new Runnable() {
            @Override
            public void run() {
                if (ChatStoreServiceImpl.this.shouldIgnoreMessage(message)) {
                    return;
                }

                Chat chat = ChatStoreServiceImpl.this.setupConversationIfNeeded(message);

                if (chat == null) {
                    // TODO: Log
                    return;
                }

                // 这里应该就立即插入数据了，而不是等到发送完成，否则在弱网
                // 情况下，还可能会导致要过很长一段时间，消息才会显示在界面上
                ChatStoreServiceImpl.this.addOrUpdateMessage(chat, message, null);

                // 添加到未决消息表
                HMR.getService(DBService.class).execute(new DBActions().create(
                        BeanMessage.fromMessage(message, System.currentTimeMillis()),
                        BeanMessage.pendingConfig()));
            }
        });
    }

    @Override
    public void onUpdateMessageState(@NonNull Message message, @NonNull Message.State state) {
        // 避免频繁的DB操作，以及避免阻塞当前线程，对Progress的进度值不操作更新DB处理
        if (shouldIgnoreMessage(message)) {
            return;
        }

        if (state instanceof Preparing) {
            return;
        }

        Identifiable target = message.getTarget();
        if (target == null) {
            return;
        }

        Chat chat = getChat(target);

        if (chat == null) {
            return;
        }

        addOrUpdateMessage(chat, message, null);
    }

    @Override
    public void afterSendingMessage(@NonNull final Message message) {
        HMRContext.work.async("ChatStore::afterSendingMessage", new Runnable() {
            @Override
            public void run() {
                // 从未决消息表中移除已经处理完成的消息
                if (ChatStoreServiceImpl.this.shouldIgnoreMessage(message)) {
                    return;
                }

                HMR.getService(DBService.class).execute(new DBActions()
                        .deleteById(BeanMessage.pendingConfig(), message.getUuid()));
            }
        });
    }

    @Override
    public void beforeReceivingMessage(@NonNull Message message) {
    }

    @Override
    public void afterReceivingMessage(@NonNull final Message message) {
        HMRContext.work.async("ChatStore::afterReceivingMessage", new Runnable() {
            @Override
            public void run() {
                if (ChatStoreServiceImpl.this.shouldIgnoreMessage(message)) {
                    return;
                }

                Chat chat = ChatStoreServiceImpl.this.setupConversationIfNeeded(message);
                if (chat == null) {
                    // TODO: Log
                    return;
                }

                ChatStoreServiceImpl.this.addOrUpdateMessage(chat, message, null);
            }
        });
    }

    private boolean shouldIgnoreMessage(@NonNull Message message) {
        return !filter.shouldAcceptMessage(message);
    }

    private Chat setupConversationIfNeeded(Message message) {
        if (message.getReceiver() instanceof AppSession) {
            Log.i(TAG, Trace.once().method("parseBroadCastAppSessionChatMessage").msg("setupConversationIfNeeded")
                    .info("receiver type", ((AppSession) message.getReceiver()).getType())
                    .info("id", ((AppSession) message.getReceiver()).getId()));
        }

        final Identifiable target = message.getTarget();
        if (target == null) {
            return null;
        }

        Chat chat = getChat(target);
        if (chat != null) {
            // 如果已经存在，则无需新增会话
            return chat;
        }

        final Chat finalChat = new Chat(target, null, 0L, message, message.getTimestamp());
        chats.put(target, finalChat);
        HMR.getService(DBService.class).execute(new DBActions()
                .createTableIfNeeded(BeanMessage.conversationConfig(target))
                .createIfNotExists(BeanConversation.fromConversation(finalChat), null)
                .run("Hummer.Run.SetupConversationIfNeeded",
                        new Runnable() {
                            @Override
                            public void run() {
                                Log.w(TAG, Trace.once().msg("Chat Created!"));
                                ChatStoreServiceImpl.this.notifyAfterAddingConversation(finalChat);
                            }
                        }));

        return finalChat;
    }

    private final Map<Identifiable, Chat> chats = new HashMap<>();
    private final Set<ChatListener> chatListeners = new HashSet<>();
    private final Map<Identifiable, Set<MessageListener>> msgListeners = new HashMap<>();
    private final Set<MessageListener> commonMsgListeners = new HashSet<>();

    private Filter filter = new Filter() {
        @Override
        public boolean shouldAcceptMessage(Message message) {
            return message.getTarget() instanceof User;
        }
    };
}
