package com.hummer.im._internals;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im._internals.shared.CodecManager;
import com.hummer.im.model.Chat;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.id.IDFactory;
import com.hummer.im.model.id.Identifiable;
import com.j256.ormlite.field.DataType;
import com.j256.ormlite.field.DatabaseField;
import com.j256.ormlite.table.DatabaseTable;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

@SuppressWarnings({"WeakerAccess", "unused"})
@DatabaseTable(tableName = "conversation_indexes")
public class BeanConversation implements Serializable {

    static final String FieldName = "name";
    static final String FieldExtra = "extra";
    static final String FIELD_UNREAD_NUM = "unread_num";
    static final String FIELD_LATEST_MSG = "latest_msg";
    static final String FIELD_TIMESTAMP = "timestamp";
    static final String CODEC_TYPE_NAME = "chat_latest_message";

    @DatabaseField(id = true, columnName = FieldName, unique = true)
    String name;

    @DatabaseField(columnName = FieldExtra)
    String extra;

    /**
     * 未读数
     */
    @DatabaseField(columnName = FIELD_UNREAD_NUM, dataType = DataType.LONG)
    long unreadNum;

    /**
     * 最后一条消息
     */
    @DatabaseField(columnName = FIELD_LATEST_MSG, dataType = DataType.STRING, defaultValue = "")
    String latestMsg;

    /**
     * 时间戳， 用户排序
     */
    @DatabaseField(columnName = FIELD_TIMESTAMP, dataType = DataType.LONG)
    long timestamp;


    public BeanConversation() {
    }

    public BeanConversation(@NonNull String name,
                            @Nullable String extra,
                            long unreadNum,
                            @NonNull String latestMsg,
                            long timestamp) {
        this.name = name;
        this.extra = extra;
        this.unreadNum = unreadNum;
        this.latestMsg = latestMsg;
        this.timestamp = timestamp;
    }

    @Override
    public String toString() {
        return "BeanConversation{" + name + "}";
    }

    static BeanConversation fromConversation(@NonNull Chat chat) {
        BeanConversation index = new BeanConversation();
        index.name  = IDFactory.makeString(chat.getTarget());
        index.extra = chat.getExtra();
        index.unreadNum = chat.getUnreadNum();
        if (chat.getLatestMsg() != null) {
            index.latestMsg = codecs.encode(chat.getLatestMsg());
        } else {
            index.latestMsg = "";
        }
        index.timestamp = chat.getTimestamp();
        return index;
    }

    static Chat toConversation(@NonNull final BeanConversation dbConversation) {
        Identifiable target = IDFactory.makeId(dbConversation.name);
        Message latestMessage = null;
        if (dbConversation.latestMsg != null && dbConversation.latestMsg.length() > 0) {
            latestMessage = codecs.decode(CODEC_TYPE_NAME, dbConversation.latestMsg);
        }
        return new Chat(target,
                dbConversation.extra,
                dbConversation.unreadNum,
                latestMessage,
                dbConversation.timestamp);
    }

    static List<BeanConversation> fromConversations(@NonNull final List<Chat> chats) {
        List<BeanConversation> dbConversations = new ArrayList<>();
        for (Chat c : chats) {
            dbConversations.add(fromConversation(c));
        }

        return dbConversations;
    }

    static List<Chat> toConversations(@NonNull final List<BeanConversation> dbConversations) {
        List<Chat> chats = new ArrayList<>();

        for (BeanConversation c : dbConversations) {
            chats.add(toConversation(c));
        }

        return chats;
    }

    public static final CodecManager<String, String, Message> codecs = new CodecManager<>();
}
