package com.hummer.im._internals;

import com.j256.ormlite.android.apptools.OrmLiteSqliteOpenHelper;
import com.j256.ormlite.dao.Dao;
import com.j256.ormlite.dao.DaoManager;
import com.j256.ormlite.table.DatabaseTableConfig;
import com.j256.ormlite.table.TableUtils;
import com.hummer.im.db.DBService;
import com.hummer.im.model.id.IDFactory;
import com.hummer.im.model.id.Identifiable;

import java.sql.SQLException;
import java.util.List;

public final class ActFixConversationsIfNeeded implements DBService.Action {
    @Override
    @SuppressWarnings("unchecked")
    public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
        Dao<BeanConversation, ?> conversationsDao = daoSet.create(null, BeanConversation.class);

        List<BeanConversation> results;

        try {
            results = conversationsDao.queryForAll();
        } catch (SQLException t) {
            // 'conversation_indexes' 表不存在，不需要修复
            return;
        }

        // 1. 对遗失的消息表进行补建
        // 2. 移除无效会话（中东地区语言数字建立的相关数据）
        for (BeanConversation bean : results) {
            if (bean.name == null) {
                continue;
            }

            Identifiable target = IDFactory.makeId(bean.name);
            if (target == null) {
                dropIllegalConversation(helper, daoSet, bean);
                continue;
            }

            recoverMissingMessageTable(helper, daoSet, target);
        }
    }

    private void recoverMissingMessageTable(OrmLiteSqliteOpenHelper helper,
                                            DBService.DaoSet daoSet,
                                            Identifiable target) {
        DatabaseTableConfig<BeanMessage> tableConfig = BeanMessage.conversationConfig(target);
        try {
            Dao<BeanMessage, ?> dao = daoSet.create(tableConfig, null);

            // 提高查询效率，使用*匹配，而不是指定uuid
            dao.queryRaw("SELECT * FROM " + tableConfig.getTableName() + " LIMIT 1");
        } catch (SQLException t) {
            try {
                // 如果消息表不在，则进行补建
                TableUtils.createTableIfNotExists(helper.getConnectionSource(), tableConfig);
            } catch (SQLException e) {
                // 如果修补失败……
            }

            DaoManager.clearDaoCache();
        }
    }

    @SuppressWarnings("unchecked")
    private void dropIllegalConversation(OrmLiteSqliteOpenHelper helper,
                                         DBService.DaoSet daoSet,
                                         BeanConversation bean) {
        // 如果是无法识别的target（例如中东地区数字导致的会话），则应移除该记录
        try {
            Dao<BeanConversation, ?> dao = daoSet.create(null, BeanConversation.class);
            ((Dao<BeanConversation, String>) dao).deleteById(bean.name);
        } catch (SQLException e) {
            // 唔……不明所以
        }

        try {
            DatabaseTableConfig<BeanMessage> tbConfig = new DatabaseTableConfig<>();
            String tableName = "conversation_" + bean.name;
            tbConfig.setTableName(tableName);
            tbConfig.setDataClass(BeanMessage.class);

            TableUtils.dropTable(helper.getConnectionSource(), tbConfig, true);
            DaoManager.clearDaoCache();
        } catch (SQLException e) {
            // 如果还失败，也做不了什么
        }
    }

    @Override
    public String toString() {
        return "Hummer.FixConversationsIfNeeded";
    }
}
