package com.hummer.im._internals;

import com.hummer.im._internals.shared.CodecManager;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im.service.ChatStoreService;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.states.StateFactory;
import com.hummer.im.model.id.IDFactory;

import org.json.JSONException;
import org.json.JSONObject;

import java.util.HashMap;
import java.util.Map;

@SuppressWarnings("unused")
public final class SPChatStoreModuleLoader implements ServiceProvider.ModuleLoader {

    @Override
    public void initModule() {
        registerChatMessageCodecs();
    }

    @Override
    public Map<Class, ServiceProvider.Service> getServices() {
        return new HashMap<Class, ServiceProvider.Service>() {{
            put(ChatStoreService.class,  new ChatStoreServiceImpl());
            put(ChatUpgradeService.class, new ChatUpgradeService());
        }};
    }

    private static void registerChatMessageCodecs() {
        BeanMessage.codecs.register(generateCodec("hmr_text_message"));
        BeanMessage.codecs.register(generateCodec("chat_message"));
        BeanConversation.codecs.register(generateLatestMessageCodec());
    }

    private static CodecManager.Codec<String, String, Message> generateLatestMessageCodec() {
        // 这两个字段是用来兼容 Hamo 版本情况下本地文本消息的情况
        final String legacyType = "hmr_text_message";
        final String indexText  = "text";

        final String indexContentType = "contentType";
        final String indexContent     = "content";
        final String indexUUID        = "uuid";
        final String indexTimestamp   = "ts";
        final String indexSender      = "sender";
        final String indexReceiver    = "receiver";
        final String indexState       = "state";

        return new CodecManager.Codec<String, String, Message>() {
            @Override
            public String getDataType() {
                return BeanConversation.CODEC_TYPE_NAME;
            }

            @Override
            public Class<? extends Message> getModelClass() {
                return Message.class;
            }

            @Override
            public String encode(Message message) throws JSONException {
                JSONObject json = new JSONObject();

                json.put(indexUUID,        message.getUuid());
                json.put(indexTimestamp,   message.getTimestamp());
                json.put(indexSender,      IDFactory.makeString(message.getSender()));
                json.put(indexReceiver,    IDFactory.makeString(message.getReceiver()));
                json.put(indexState,       StateFactory.makeString(message.getState()));
                json.put(indexContentType, Content.getDataType(message.getContent()));
                json.put(indexContent,     Content.makeString(message.getContent()));

                return json.toString();
            }

            @Override
            public Message decode(String content) throws JSONException {
                JSONObject json = new JSONObject(content);

                Message message = new Message();
                message.setUuid(json.getString(indexUUID));
                message.setTimestamp(json.getLong(indexTimestamp));
                message.setSender(IDFactory.makeId(json.getString(indexSender)));
                message.setReceiver(IDFactory.makeId(json.getString(indexReceiver)));
                message.setState(StateFactory.makeState(json.getString(indexState)));

                if (legacyType.equals(this.getDataType())) { // 兼容旧版本的文本情况
                    // 0 是文本消息内容的类型值，这里为了避免直接依赖上层模块中的TextChatContent，所以使用隐性的
                    // 协议值依赖
                    message.setContent(Content.makeContent(0, json.getString(indexText)));
                } else {
                    int contentType = json.getInt(indexContentType);
                    message.setContent(Content.makeContent(contentType, json.getString(indexContent)));
                }

                return message;
            }
        };
    }

    private static
    CodecManager.Codec<String, BeanMessage.DBMessageData, Message> generateCodec(final String dataType) {
        // 这两个字段是用来兼容 Hamo 版本情况下本地文本消息的情况
        final String legacyType = "hmr_text_message";
        final String indexText  = "text";

        final String indexContentType = "contentType";
        final String indexContent     = "content";
        final String indexUUID        = "uuid";
        final String indexTimestamp   = "ts";
        final String indexSender      = "sender";
        final String indexReceiver    = "receiver";
        final String indexState       = "state";

        return new CodecManager.Codec<String, BeanMessage.DBMessageData, Message>() {
            @Override
            public String getDataType() {
                return dataType;
            }

            @Override
            public Class<? extends Message> getModelClass() {
                return Message.class;
            }

            @Override
            public BeanMessage.DBMessageData encode(Message chatMessage) throws JSONException {
                JSONObject json = new JSONObject();

                json.put(indexUUID,        chatMessage.getUuid());
                json.put(indexTimestamp,   chatMessage.getTimestamp());
                json.put(indexSender,      IDFactory.makeString(chatMessage.getSender()));
                json.put(indexReceiver,    IDFactory.makeString(chatMessage.getReceiver()));
                json.put(indexState,       StateFactory.makeString(chatMessage.getState()));
                json.put(indexContentType, Content.getDataType(chatMessage.getContent()));
                json.put(indexContent,     Content.makeString(chatMessage.getContent()));

                return new BeanMessage.DBMessageData(chatMessage.getUuid(), json.toString());
            }

            @Override
            public Message decode(BeanMessage.DBMessageData data) throws JSONException {
                JSONObject json = new JSONObject(data.content);

                Message message = new Message();
                message.setUuid(json.getString(indexUUID));
                message.setTimestamp(json.getLong(indexTimestamp));
                message.setSender(IDFactory.makeId(json.getString(indexSender)));
                message.setReceiver(IDFactory.makeId(json.getString(indexReceiver)));
                message.setState(StateFactory.makeState(json.getString(indexState)));

                if (legacyType.equals(this.getDataType())) { // 兼容旧版本的文本情况
                    // 0 是文本消息内容的类型值，这里为了避免直接依赖上层模块中的TextChatContent，所以使用隐性的
                    // 协议值依赖
                    message.setContent(Content.makeContent(0, json.getString(indexText)));
                } else {
                    int contentType = json.getInt(indexContentType);
                    message.setContent(Content.makeContent(contentType, json.getString(indexContent)));
                }

                return message;
            }
        };
    }
}
