package com.hummer.im._internals;

import com.hummer.im.model.chat.Message;
import com.hummer.im.model.id.Identifiable;
import com.j256.ormlite.field.DataType;
import com.j256.ormlite.field.DatabaseField;
import com.j256.ormlite.table.DatabaseTableConfig;
import com.hummer.im.model.id.IDFactory;
import com.hummer.im._internals.shared.CodecManager;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

@SuppressWarnings({"WeakerAccess", "unused"})
public final class BeanMessage implements Serializable {

    static final String FieldKey        = "key";
    static final String FieldTimestamp  = "ts";
    static final String FieldType       = "type";
    static final String FieldContent    = "content";
    static final String FieldDelFlag    = "deleted";

    @DatabaseField(columnName = FieldKey, dataType = DataType.STRING, id = true)
    String key;

    @DatabaseField(columnName = FieldTimestamp, dataType = DataType.LONG, canBeNull = false)
    long ts;

    @DatabaseField(columnName = FieldType, dataType = DataType.STRING, canBeNull = false)
    String type;

    @DatabaseField(columnName = FieldContent, dataType = DataType.STRING, canBeNull = false)
    String content;

    @DatabaseField(columnName = FieldDelFlag, dataType = DataType.BOOLEAN, canBeNull = false)
    boolean deleted;

    public BeanMessage() {
    }

    private BeanMessage(String key, String type, String content, Long ts) {
        this.content = content;
        this.key     = key;
        this.type    = type;
        this.deleted = false;

        if (ts != null) {
            this.ts = ts;
        }
    }

    @Override
    public String toString() {
        return "BeanMessage{" + key + ", " + type + ", " + content + "}";
    }

    static DatabaseTableConfig<BeanMessage> conversationConfig(Identifiable target) {
        DatabaseTableConfig<BeanMessage> config = new DatabaseTableConfig<>();
        String tableName = "conversation_" + IDFactory.makeString(target);
        config.setTableName(tableName);
        config.setDataClass(BeanMessage.class);
        return config;
    }

    static DatabaseTableConfig<BeanMessage> pendingConfig() {
        DatabaseTableConfig<BeanMessage> config = new DatabaseTableConfig<>();
        String tableName = "pending_chat_messages";
        config.setTableName(tableName);
        config.setDataClass(BeanMessage.class);
        return config;
    }

    static Message toMessage(BeanMessage dbMessage) {
        return codecs.decode(dbMessage.type, new DBMessageData(dbMessage.key, dbMessage.content));
    }

    static List<Message> toMessages(List<BeanMessage> dbMessages) {
        List<Message> messages = new ArrayList<>();

        for (BeanMessage dbMsg : dbMessages) {
            Message message = toMessage(dbMsg);
            if (message != null) {
                messages.add(message);
            }
        }

        return messages;
    }

    static List<BeanMessage> fromMessages(List<Message> messages) {
        List<BeanMessage> dbItems = new ArrayList<>();

        for (Message message : messages) {
            dbItems.add(fromMessage(message, System.currentTimeMillis()));
        }

        return dbItems;
    }

    static BeanMessage fromMessage(Message message, Long timestamp) {
        DBMessageData data = codecs.encode(message);
        return new BeanMessage(
                data.key,
                codecs.getDataType(message),
                data.content,
                timestamp
        );
    }

    public static class DBMessageData {
        public final String key;
        public final String content;

        public DBMessageData(String key, String content) {
            this.key     = key;
            this.content = content;
        }

        @Override
        public String toString() {
            return "DBMessageData{" + key + ", " + content + "}";
        }
    }

    public static final CodecManager<String, DBMessageData, Message> codecs = new CodecManager<>();
}
