package com.hummer.im.service;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.HMR;
import com.hummer.im.model.Chat;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.store.FetchingClauses;
import com.hummer.im.model.chat.store.RemovingClauses;
import com.hummer.im.model.id.Identifiable;

import java.util.List;
import java.util.Set;

/**
 * 会话及消息管理服务接口
 */
public interface ChatStoreService {

    /**
     * 获取所有会话列表
     *
     * <br> 结果已排序。排序规则：先根据权重(priority)从大到小排序；权重相同，根据消息时间戳从大到小排
     *
     * @return 会话列表
     */
    List<Chat> getChats();

    /**
     * 根据Tag获取会话列表
     *
     * <br> 结果已排序。排序规则：先根据权重(priority)从大到小排序；权重相同，根据消息时间戳从大到小排
     *
     * @param tag 标签
     * @return 会话列表
     */
    List<Chat> getChatsByTag(@NonNull String tag);

    /**
     * 根据targetId获取一个会话
     *
     * @param target 会话的目标对象，可能取值请参阅{@link Message#getTarget()}
     * @return 会话
     */
    Chat getChat(@NonNull Identifiable target);

    /**
     * 根据会话目标创建一个会话实例，如果该会话已存在，视为调用失败，会执行completion的onFailed回调。
     * 因此，如果不确定会话是否存在，应调用getConversation来尝试获取对应的会话。
     *
     * @param target 会话目标，该目标可以是Group, User等类型的实例
     */
    void createChat(@NonNull Identifiable target, @NonNull HMR.CompletionArg<Chat> completion);

    /**
     * 从会话列表中添加一个会话
     *
     * @param chat 待加入的会话
     * @param completion   操作完成的回调
     */
    void addChat(@NonNull Chat chat, @Nullable HMR.Completion completion);

    /**
     * 从会话列表中删除某个会话
     *
     * @param chat 待删除会话
     * @param completion   操作完成的回调
     */
    void removeChat(@NonNull Chat chat, @Nullable HMR.Completion completion);

    /**
     * 更新会话信息
     *
     * @param chat 需要更新会话的信息
     * @param completion 操作完成的回调
     */
    void updateChat(@NonNull Chat chat, @NonNull HMR.Completion completion);

    /**
     * 重置未读数
     *
     * @param chat 需要更新会话的信息
     * @param completion   操作完成的回调
     */
    void resetUnread(@NonNull Chat chat, @Nullable HMR.Completion completion);

    /**
     * 给会话中添加消息，如果该消息存在，则会更新其内容
     *
     * @param chat 需要添加消息的会话
     * @param message 欲添加的消息对象
     * @param completion 回调
     */
    void addOrUpdateMessage(@NonNull Chat chat,
                            @NonNull Message message,
                            @Nullable HMR.Completion completion);

    /**
     * 从指定的会话中根据策略获取消息列表
     *
     * @param chat 需要获取消息的会话
     * @param clauses    描述消息获取的条件集合
     * @param completion 回调
     */
    void fetchMessages(@NonNull Chat chat,
                       @NonNull FetchingClauses clauses,
                       @Nullable HMR.CompletionArg<List<Message>> completion);

    /**
     * 根据uuids查询消息
     *
     * <br> NOTE: 删除在数据库中存在的uuid，如果对应的uuid不存在相应的消息，直接忽略该条uuid
     *
     * @param chat       对应会话
     * @param uuids      消息的uuid列表
     * @param completion 回调
     */
    void fetchMessagesByUuids(@NonNull Chat chat,
                              @NonNull Set<String> uuids,
                              @NonNull HMR.CompletionArg<List<Message>> completion);

    /**
     * 从指定会话中根据策略删除消息列表
     *
     * @param chat 需要删除消息的会话
     * @param clauses    描述消息移除的条件集合
     * @param completion 操作回调
     */
    void removeMessages(@NonNull Chat chat,
                        @NonNull RemovingClauses clauses,
                        @Nullable HMR.Completion completion);

    /**
     * 设置优先级
     *
     * @param chat       会话
     * @param priority   权重优先级
     * @param completion 回调
     */
    void setPriority(@NonNull Chat chat, int priority, @Nullable HMR.Completion completion);

    /**
     * 设置会话标签
     *
     * <p>
     * 直接覆盖已有的标签
     *      已设置标签：a, b, c
     *      要设置的标签：d
     *      成功后会话标签：d
     * </p>
     *
     * @param chat       需要设置标签的会话
     * @param tags       会话标签列表， 空集合（not null）表示清空标签
     * @param completion 回调
     */
    void setTags(@NonNull Chat chat, @NonNull Set<String> tags, @Nullable HMR.Completion completion);

    /**
     * 添加会话标签
     *
     * <p>
     * 1. 在已有的标签上追加标签
     *      已设置标签：a, b
     *      要追加的标签：a, c, d
     *      成功后会话标签：a, b, c, d
     * 2. 不管追加的标签是否都存在，都会去设置，回调告知成功失败
     * </p>
     *
     * @param chat       需要设置标签的会话
     * @param tags       会话标签列表
     * @param completion 回调
     */
    void addTags(@NonNull Chat chat, @NonNull Set<String> tags, @Nullable HMR.Completion completion);

    /**
     * 移除会话标签
     *
     * <p>
     * 1. 在已有的标签上删除标签
     *      已设置标签：a, b, c, d
     *      要删除的标签：a, c, e
     *      成功后会话标签：b, d
     * 2. 如果要删除的标签不存在，会直接忽略该标签，回调告知设置成功与否
     * </p>
     *
     * @param chat       需要移除标签的会话
     * @param tags       标签列表
     * @param completion 回调
     */
    void removeTags(@NonNull Chat chat, @NonNull Set<String> tags, @Nullable HMR.Completion completion);

    /**
     * ChatStoreService的消息过滤器，Hummer默认情况下只会记录P2P的消息。如果业务希望其它特定消息，如AppSession
     * 消息也进入本地消息记录，则应自行接管消息过滤逻辑。注意，如果通过setChatFilter接管了过滤器，则也应该对P2P
     * 消息是否应进入本地记录进行判断。
     */
    interface Filter {
        /**
         * 过滤回调方法，ChatStoreService每次进行消息处理时都会回调该接口，并完全过滤掉那些返回false的消息
         * @param message 需要进行过滤与否判断的消息对象
         * @return 当不希望该消息进入本地记录时，返回false，否则返回true
         */
        boolean shouldAcceptMessage(Message message);
    }

    void setChatFilter(Filter filter);


    /**
     * ConversationListener用于进行会话内行为监听
     * <p>
     * {@link ChatStoreService#addChatListener(ChatListener)}
     * {@link ChatStoreService#removeChatListener(ChatListener)}
     */
    interface MessageListener {

        /**
         * 当消息入库后，会产生该回调
         *
         * @param chat 产生回调的会话对象
         * @param message      消息对象
         */
        void afterAddingMessage(Chat chat, Message message);

        /**
         * 在会话把消息状态变更状态入库后，会产生该回调
         *
         * @param chat 产生回调的会话
         * @param message      状态变更的消息
         */
        void afterUpdateMessage(Chat chat, Message message);

        /**
         * 在会话即将移除消息前，会产生该回调
         *
         * @param chat 产生回调的会话
         * @param messages     即将移除的消息
         */
        void beforeRemovingMessages(Chat chat, List<Message> messages);

        /**
         * 当会话内消息被清空时，会产生该回调
         *
         * @param chat 被清空消息的会话
         */
        void afterClearMessages(Chat chat);
    }

    /**
     * 用于进行添加/删除/更新Conversation以及可以监听所有会话里面消息变化的的事件监听器
     * <p>
     * 如果需要监听所有会话下的消息变化可以通过该接口去以及调用一下方法去监听
     * {@link ChatStoreService#addChatListener(ChatListener)}
     * {@link ChatStoreService#removeChatListener(ChatListener)}
     */
    interface ChatListener {
        /**
         * 当添有新增会话时，会触发该回调
         * @param chat 当前新增的会话
         */
        void afterCreatingChat(@NonNull Chat chat);

        /**
         * 当会话被移除时，会触发该回调
         * @param chat 被移除的会话
         */
        void beforeRemovingChat(@NonNull Chat chat);

        /**
         * 当会话属性（例如extra）发生变化时，会触发该回调
         * @param chat 发生属性变更的会话
         */
        void afterUpdateChat(@NonNull Chat chat);

        /**
         * 会话列表发生变化时，会触发该回调。变化指的是其中某个会话的属性、新增会话、移除会话等。
         * @param chats 会话列表
         */
        void afterUpdateChats(@NonNull List<Chat> chats);
    }

    /**
     * 增加指定的监听对象
     *
     * @param listener 欲增加的监听对象
     */
    void addChatListener(@NonNull ChatListener listener);

    /**
     * 移除特定的监听对象
     *
     * @param listener 欲移除的监听对象
     */
    void removeChatListener(@NonNull ChatListener listener);

    /**
     * 添加消息监听器
     *
     * @param chat 指定监听的会话，如果conversation为null，则会收到所有会话的消息
     * @param listener 欲增加的监听对象
     */
    void addMessageListener(@Nullable Chat chat, @NonNull MessageListener listener);

    /**
     * 移除特定的监听对象
     *
     * @param chat 该监听器当前监听的会话对象
     * @param listener     欲移除的监听对象
     */
    void removeMessageListener(@Nullable Chat chat, @NonNull MessageListener listener);



}
