package com.hummer.im._internals;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.chatsvc.IdentifiableHelper;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.services.user.UserService;
import com.hummer.im._internals.shared.DispatchQueue;
import com.hummer.im._internals.shared.ServiceProvider;
import com.hummer.im._internals.shared.statis.Util;
import com.hummer.im.db.DBActions;
import com.hummer.im.db.DBService;
import com.hummer.im.model.Chat;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.states.Failed;
import com.hummer.im.model.chat.states.Preparing;
import com.hummer.im.model.chat.store.FetchingClauses;
import com.hummer.im.model.chat.store.RemovingClauses;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.OnFailure;
import com.hummer.im.model.completion.OnSuccess;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.completion.RichCompletionArg;
import com.hummer.im.model.id.AppSession;
import com.hummer.im.model.id.Identifiable;
import com.hummer.im.model.id.User;
import com.hummer.im.service.ChatService;
import com.hummer.im.service.ChatStoreService;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import static com.hummer.im.Error.Code.InvalidParameters;

public final class ChatStoreServiceImpl implements ChatStoreService,
        ChatService.MessageListener,
        ChatService.StateListener,
        ServiceProvider.Service {

    private static final String TAG = "ChatStoreService";

    // Mark - Implements: ServiceProvider.Service

    @Override
    public Class[] staticDependencies() {
        return new Class[]{ChatService.class};
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class, ChatUpgradeService.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return new Class[]{ChatService.class};
    }

    @Override
    public void initService() {
    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        // 1. 缓存会话到内存
        DBService.Action loadConversations = new DBActions()
                .createTableIfNeeded(BeanConversation.class)
                .queryAll(BeanConversation.class,
                        new DBActions.QueryAcceptor<BeanConversation>() {
                            @Override
                            public void onQueryResults(List<BeanConversation> indexes) {
                                synchronized (chats) {
                                    for (BeanConversation index : indexes) {
                                        Chat c = BeanConversation.toConversation(index);
                                        if (c != null) {
                                            chats.add(c);
                                        }
                                    }
                                }
                            }
                        });

        // 2. 清理生命周期不完整的遗留消息，将它们都标记为失败状态
        DBService.Action handlePendingMessages = new DBActions()
                .createTableIfNeeded(BeanMessage.pendingConfig())
                .custom(new ActGetPendingChatMessages(new ActGetPendingChatMessages.Acceptor() {
                    @Override
                    public void onAcquired(List<Message> messages) {
                        Error recovered = new Error(Error.Code.UndefinedExceptions, "从异常中恢复", null);
                        for (Message message : messages) {
                            message.setState(new Failed(recovered));

                            Identifiable target = message.getTarget();
                            if (target == null) {
                                return;
                            }

                            Chat chat = ChatStoreServiceImpl.this.getChat(target);
                            if (chat != null) {
                                ChatStoreServiceImpl.this.addOrUpdateMessage(chat, message, null);
                            }
                        }
                    }
                }))
                .deleteAll(BeanMessage.pendingConfig());

        HMR.getService(DBService.class).execute(
                new DBActions()
                        // 尝试修复旧版本导致的有会话数据
                        .custom(new ActFixConversationsIfNeeded())

                        .custom(loadConversations)
                        .custom(handlePendingMessages)
                        .run("Hummer.Chat.AddListener", new Runnable() {
                            @Override
                            public void run() {
                                // 3. Add listener only when succeed
                                HMR.getService(ChatService.class).addStateListener(null,
                                        ChatStoreServiceImpl.this
                                );

                                HMR.getService(ChatService.class).addMessageListener(null, ChatStoreServiceImpl.this);

                                ChatStoreServiceImpl.this.notifyUpdateChats();
                            }
                        }),
                completion);
    }

    @Override
    public void closeService() {
        HMR.getService(ChatService.class).removeMessageListener(null, ChatStoreServiceImpl.this);
        synchronized (chats) {
            chats.clear();

        }
        notifyUpdateChats();
    }

    // Mark - Implements: ChatStoreService

    @Override
    public void createChat(@NonNull Identifiable target, @NonNull final HMR.CompletionArg<Chat> completion) {
        final RichCompletionArg rCompletion = new RichCompletionArg<>(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        Chat chat = getChat(target);

        if (chat == null) {
            // 新增会话，优先级为0, tags为空
            chat = ChatUtils.newChat(target, null, 0L, null,
                    System.currentTimeMillis(), 0, new LinkedHashSet<String>());
            if (chat == null) {
                CompletionUtils.dispatchFailure(rCompletion,
                        new Error(Error.Code.UndefinedExceptions, "new chat error"));
                return;
            }

            final Chat finalChat = chat;
            addChat(chat, new HMR.Completion() {
                @Override
                public void onSuccess() {
                    CompletionUtils.dispatchSuccess(rCompletion, finalChat);
                }

                @Override
                public void onFailed(Error error) {
                    CompletionUtils.dispatchFailure(rCompletion, error);
                }
            });
        } else {
            CompletionUtils.dispatchFailure(rCompletion, new Error(InvalidParameters, "会话已存在"));
        }
    }

    @Override
    public @Nullable
    Chat getChat(@NonNull Identifiable target) {
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            Log.e(TAG, Trace.method("getChat").msg("User not login, or using anonymous mode"));
            return null;
        }

        synchronized (chats) {
            for (Chat chat : chats) {
                if (Objects.equals(target, chat.getTarget())) {
                    return chat;
                }
            }
        }
        return null;
    }

    @Override
    public @NonNull
    List<Chat> getChats() {
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            Log.e(TAG, Trace.method("getChats").msg("User not login, or using anonymous mode"));
            return new ArrayList<>();
        }

        sort();
        return new ArrayList<>(chats);
    }

    @Override
    public List<Chat> getChatsByTag(@NonNull String tag) {
        List<Chat> chatTagList = new ArrayList<>();
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            Log.e(TAG, Trace.method("getChatsByTag").msg("User not login, or using anonymous mode"));
            return chatTagList;
        }

        List<Chat> chatList = getChats();
        for (Chat chat : chatList) {
            if (chat.getTags().contains(tag)) {
                chatTagList.add(chat);
            }
        }
        return chatTagList;
    }

    @Override
    public void addChat(@NonNull final Chat chat, @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        final Identifiable target = chat.getTarget();
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .guard(new Error(InvalidParameters, "会话已存在"),
                                new DBActions.Guarder() {
                                    @Override
                                    public boolean shouldBreak() {
                                        return getChat(target) != null;
                                    }
                                })
                        .createTableIfNeeded(BeanChatMessage.chatMessageTableConfig(target))
                        .create(BeanConversation.fromConversation(chat), null)
                        .run("Hummer.AddConversation", new Runnable() {
                            @Override
                            public void run() {
                                synchronized (chats) {
                                    chats.add(chat);
                                }
                                ChatStoreServiceImpl.this.notifyAfterAddingChats(chat);
                            }
                        }),
                rCompletion);
    }

    @Override
    public void removeChat(@NonNull final Chat chat, @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        final Identifiable target = chat.getTarget();
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .guard(new Error(InvalidParameters, "会话不存在"),
                                new DBActions.Guarder() {
                                    @Override
                                    public boolean shouldBreak() {
                                        return getChat(target) == null;
                                    }
                                })
                        .delete(BeanConversation.fromConversation(chat), null)
                        .custom(new ActRemoveMessages(chat, new RemovingClauses().setClearAll()))
                        .dropTableIfExist(BeanMessage.conversationConfig(target))
                        .run("Hummer.DetachConversation", new Runnable() {
                            @Override
                            public void run() {
                                // 1. 从内存缓存中进行移除
                                synchronized (chats) {
                                    chats.remove(chat);
                                }
                                ChatStoreServiceImpl.this.notifyAfterRemovingChats(chat);
                            }
                        }),
                rCompletion);
    }

    @Override
    public void updateChat(@NonNull final Chat chat, @NonNull HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        final Identifiable target = chat.getTarget();
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .guard(new Error(InvalidParameters, "会话不存在"),
                                new DBActions.Guarder() {
                                    @Override
                                    public boolean shouldBreak() {
                                        return getChat(target) == null;
                                    }
                                })
                        .update(BeanConversation.fromConversation(chat), null)
                        .run("Hummer.NotifyUpdateConversation", new Runnable() {
                            @Override
                            public void run() {
                                ChatStoreServiceImpl.this.notifyUpdateChat(chat);
                            }
                        }),
                rCompletion);
    }

    @Override
    public void resetUnread(@NonNull final Chat chat, @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        chat.setUnreadNum(0L);
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .custom(new ActResetChatUnreadNum(chat.getTarget()))
                        .run("Hummer.resetUnread", new Runnable() {
                            @Override
                            public void run() {
                                ChatStoreServiceImpl.this.notifyUpdateChat(chat);
                            }
                        }), rCompletion);
    }

    @Override
    public void addOrUpdateMessage(@NonNull final Chat chat,
                                   @NonNull final Message message,
                                   @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        HMR.getService(DBService.class).execute(new ActAddOrUpdateMessage(chat, message,
                new ActAddOrUpdateMessage.Callback() {
                    @Override
                    public void afterSaved(boolean isAdding) {
                        if (isAdding) {
                            // 当发送消息用户非当前登录用户
                            if (!HMR.isMe(message.getSender())) {
                                chat.setUnreadNum(chat.getUnreadNum() + 1);
                            }

                            if (chat.getLatestMsg() != null) {
                                String latestMsgId = Util.getMsgId(chat.getLatestMsg().getTimestamp(),
                                        chat.getLatestMsg().getUuid());
                                String newMsgId = Util.getMsgId(message.getTimestamp(), message.getUuid());
                                // -1 表示latestMsgId 小
                                if (latestMsgId.compareTo(newMsgId) < 0) {
                                    chat.setLatestMsg(message);
                                    chat.setTimestamp(message.getTimestamp());
                                }
                            } else {
                                chat.setLatestMsg(message);
                                chat.setTimestamp(message.getTimestamp());
                            }

                            HMR.getService(ChatStoreService.class).updateChat(chat,
                                    new HMR.Completion() {
                                        @Override
                                        public void onSuccess() {
                                            ChatStoreServiceImpl.this.notifyAfterMessageAdding(chat, message);
                                        }

                                        @Override
                                        public void onFailed(Error err) {

                                        }
                                    });
                        } else {
                            String latestMsgId = Util.getMsgId(chat.getLatestMsg().getTimestamp(),
                                    chat.getLatestMsg().getUuid());
                            String newMsgId = Util.getMsgId(message.getTimestamp(), message.getUuid());
                            // 如果uuid相同需要更新最后一条消息。如果新消息的时间戳比旧消息的大，也需要更新
                            // msgid是timestamp和uuid的整合，相比较能满足上述两种条件
                            // latestMsgId = newMsgId 则时间戳相同，uuid也相同
                            // -1 表示latestMsgId 小
                            if (latestMsgId.compareTo(newMsgId) <= 0) {
                                chat.setLatestMsg(message);
                                HMR.getService(ChatStoreService.class).updateChat(chat,
                                        new HMR.Completion() {
                                            @Override
                                            public void onSuccess() {
                                                ChatStoreServiceImpl.this.notifyUpdateMessage(chat, message);
                                            }

                                            @Override
                                            public void onFailed(Error err) {

                                            }
                                        });
                            } else {
                                ChatStoreServiceImpl.this.notifyUpdateMessage(chat, message);
                            }
                        }
                    }
                }), rCompletion);
    }

    @Override
    public void fetchMessages(@NonNull Chat chat,
                              @NonNull FetchingClauses clauses,
                              @Nullable final HMR.CompletionArg<List<Message>> completion) {
        final RichCompletionArg rCompletion = new RichCompletionArg<>(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        final ActFetchMessages fetch = new ActFetchMessages(chat, clauses);
        HMR.getService(DBService.class).execute(fetch, new RichCompletion()
                .onSuccess(new OnSuccess() {
                    @Override
                    public void onSuccess() {
                        CompletionUtils.dispatchSuccess(rCompletion, fetch.results);
                    }
                })
                .onFailure(new OnFailure() {
                    @Override
                    public void onFailure(Error error) {
                        CompletionUtils.dispatchFailure(rCompletion, error);
                    }
                }));
    }

    @Override
    public void fetchMessagesByUuids(@NonNull final Chat chat,
                                     @NonNull Set<String> uuids,
                                     @NonNull final HMR.CompletionArg<List<Message>> completion) {
        final RichCompletionArg rCompletion
                = new RichCompletionArg<>(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        final ActFetchMessagesByUuids fetch = new ActFetchMessagesByUuids(chat, uuids);
        HMR.getService(DBService.class)
                .execute(fetch, new RichCompletion()
                        .onSuccess(new OnSuccess() {
                            @Override
                            public void onSuccess() {
                                CompletionUtils.dispatchSuccess(rCompletion, fetch.messages);
                            }
                        })
                        .onFailure(new OnFailure() {
                            @Override
                            public void onFailure(Error error) {
                                CompletionUtils.dispatchFailure(rCompletion, error);
                            }
                        }));
    }

    @Override
    public void removeMessages(@NonNull final Chat chat,
                               @NonNull final RemovingClauses clauses,
                               @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        if (clauses.messages != null) {
            notifyBeforeMessagesRemoving(chat, clauses.messages);
        }
        HMR.getService(DBService.class).execute(
                new DBActions()
                        .custom(new ActRemoveMessages(chat, clauses))
                        .custom(new ActUpdateChatLatestMsg(chat))
                        .run("Hummer.NotifyAfterRemovingMessages", new Runnable() {
                            @Override
                            public void run() {
                                if (clauses.messages == null) {
                                    ChatStoreServiceImpl.this.notifyAfterMessagesClear(chat);
                                }
                            }
                        }), rCompletion);
    }

    @Override
    public void setPriority(@NonNull final Chat chat, int priority, @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        chat.setPriority(priority);
        HMR.getService(DBService.class).execute(new DBActions()
                .custom(new ActUpdateChatPriority(chat.getTarget(), chat.getPriority()))
                .run("ChatStoreService::setPriority", new Runnable() {
                    @Override
                    public void run() {
                        ChatStoreServiceImpl.this.notifyUpdateChat(chat);
                    }
                }), rCompletion);
    }

    @Override
    public void setTags(@NonNull final Chat chat,
                        @NonNull Set<String> tags,
                        @Nullable HMR.Completion completion) {
        handleTags(SET_TAGS, chat, tags, completion);
    }

    @Override
    public void addTags(@NonNull final Chat chat,
                        @NonNull Set<String> tags,
                        @Nullable HMR.Completion completion) {
        handleTags(ADD_TAGS, chat, tags, completion);
    }

    @Override
    public void removeTags(@NonNull final Chat chat,
                           @NonNull Set<String> tags,
                           @Nullable HMR.Completion completion) {
        handleTags(REMOVE_TAGS, chat, tags, completion);
    }

    private void handleTags(int operation,
                            @NonNull final Chat chat,
                            @NonNull Set<String> tags,
                            @Nullable HMR.Completion completion) {
        RichCompletion rCompletion = new RichCompletion(completion);
        if (HMR.getMe() == null || HMR.getMe().isAnonymous()) {
            CompletionUtils.dispatchFailure(rCompletion,
                    new Error(Error.Code.BadUser, "Hummer not open, or using anonymous mode"));
            return;
        }

        if (operation == SET_TAGS) {
            chat.setTags(tags);
        } else if (operation == ADD_TAGS) {
            chat.getTags().addAll(tags);
        } else if (operation == REMOVE_TAGS) {
            chat.getTags().removeAll(tags);
        }

        HMR.getService(DBService.class).execute(new DBActions()
                .custom(new ActUpdateChatTag(chat.getTarget(), chat.getTags()))
                .run(new Runnable() {
                    @Override
                    public void run() {
                        ChatStoreServiceImpl.this.notifyUpdateChat(chat);
                    }
                }), rCompletion);
    }

    @Override
    public void addChatListener(@NonNull final ChatListener listener) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                synchronized (chatListeners) {
                    chatListeners.add(listener);
                }
                Log.i(TAG, Trace.method("addChatListener")
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", chatListeners.size()));
            }
        });
    }

    @Override
    public void removeChatListener(@NonNull final ChatListener listener) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                synchronized (chatListeners) {
                    chatListeners.remove(listener);
                }
                Log.i(TAG, Trace.method("removeChatListener")
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", chatListeners.size()));
            }
        });
    }

    @Override
    public void addMessageListener(@Nullable final Chat chat, @NonNull final MessageListener listener) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (chat == null) {
                    commonMsgListeners.add(listener);
                    return;
                }

                Set<MessageListener> listeners = msgListeners.get(chat.getTarget());
                if (listeners == null) {
                    listeners = new HashSet<>();
                    msgListeners.put(chat.getTarget(), listeners);
                }

                listeners.add(listener);

                Log.i(TAG, Trace.method("addMessageListener")
                        .info("chat", chat)
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", ChatStoreServiceImpl.this.chatListeners.size()));
            }
        });
    }

    @Override
    public void removeMessageListener(@Nullable final Chat chat, @NonNull final MessageListener listener) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.method("removeMessageListener")
                        .info("chat", chat)
                        .info("listener", listener.getClass().getSimpleName())
                        .info("size", ChatStoreServiceImpl.this.chatListeners.size()));

                if (chat == null) {
                    commonMsgListeners.remove(listener);
                    return;
                }

                Set<MessageListener> listeners = msgListeners.get(chat.getTarget());
                if (listeners == null) {
                    return;
                }

                listeners.remove(listener);
            }
        });
    }

    private void notifyAfterAddingChats(@NonNull final Chat chat) {
        DispatchQueue.main.async(new Runnable() {
            @Override
            public void run() {
                synchronized (chatListeners) {
                    for (ChatListener l : chatListeners) {
                        l.afterCreatingChat(chat);
                        l.afterUpdateChats(getChats());
                    }
                }
            }
        });
    }

    private void notifyAfterRemovingChats(@NonNull final Chat chat) {
        DispatchQueue.main.async(new Runnable() {
            @Override
            public void run() {
                synchronized (chatListeners) {
                    for (ChatListener l : chatListeners) {
                        l.beforeRemovingChat(chat);
                        l.afterUpdateChats(getChats());
                    }
                }
            }
        });
    }

    private void notifyUpdateChats() {
        DispatchQueue.main.sync(new Runnable() {
            @Override
            public void run() {
                synchronized (chatListeners) {
                    for (ChatListener l : chatListeners) {
                        l.afterUpdateChats(getChats());
                    }
                }
            }
        });
    }

    private void notifyUpdateChat(@NonNull final Chat chat) {
        if (chatListeners.size() == 0) {
            return;
        }

        DispatchQueue.main.sync(new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.method("notifyUpdateChat")
                        .info("target", chat.getTarget().getId()));

                synchronized (chatListeners) {
                    for (ChatListener l : chatListeners) {
                        l.afterUpdateChat(chat);
                    }
                }
            }
        });
    }

    interface MessageListenersVisitor {
        void visit(MessageListener listener);

        String visitorName();
    }

    private void iterateMessageListeners(final Chat chat, final MessageListenersVisitor visitor) {
        DispatchQueue.main.sync(new Runnable() {
            @Override
            public void run() {
                // 避免同一个listener被通知多次，先将它们收集到一个set内，然后在发起访问
                Set<MessageListener> listeners = new HashSet<>(commonMsgListeners);

                Set<MessageListener> specificListeners = msgListeners.get(chat.getTarget());
                if (specificListeners != null) {
                    listeners.addAll(specificListeners);
                }

                for (MessageListener l : listeners) {
                    visitor.visit(l);
                }
            }
        });
    }

    private void notifyAfterMessageAdding(@NonNull final Chat chat, @NonNull final Message message) {
        Log.i(TAG, Trace.method("notifyAfterMessageAdding")
                .info("target", chat.getTarget()).info("uuid", message.getUuid()));

        iterateMessageListeners(chat, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.afterAddingMessage(chat, message);
            }

            @Override
            public String visitorName() {
                return "notifyAfterMessageAdding";
            }
        });
    }

    private void notifyUpdateMessage(@NonNull final Chat chat, final Message message) {
        iterateMessageListeners(chat, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.afterUpdateMessage(chat, message);
            }

            @Override
            public String visitorName() {
                return "notifyUpdateMessage";
            }
        });
    }

    private void notifyBeforeMessagesRemoving(@NonNull final Chat chat, final List<Message> messages) {
        iterateMessageListeners(chat, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.beforeRemovingMessages(chat, messages);
            }

            @Override
            public String visitorName() {
                return "notifyBeforeMessagesRemoving";
            }
        });
    }

    private void notifyAfterMessagesClear(@NonNull final Chat chat) {
        iterateMessageListeners(chat, new MessageListenersVisitor() {
            @Override
            public void visit(MessageListener listener) {
                listener.afterClearMessages(chat);
            }

            @Override
            public String visitorName() {
                return "notifyAfterMessagesClear";
            }
        });
    }

    @Override
    public void setChatFilter(Filter filter) {
        this.filter = filter;
    }

    // Mark - ChatService.Listener

    @Override
    public void beforeSendingMessage(@NonNull final Message message) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (mustIgnoreMessage(message)) {
                    return;
                }

                if (ChatStoreServiceImpl.this.shouldIgnoreMessage(message)) {
                    return;
                }

                Chat chat = ChatStoreServiceImpl.this.setupConversationIfNeeded(message);

                if (chat == null) {
                    Log.w(TAG, Trace.method("beforeSendingMessage").msg("chat is null"));
                    return;
                }

                // 这里应该就立即插入数据了，而不是等到发送完成，否则在弱网
                // 情况下，还可能会导致要过很长一段时间，消息才会显示在界面上
                ChatStoreServiceImpl.this.addOrUpdateMessage(chat, message, null);

                // 添加到未决消息表
                HMR.getService(DBService.class).execute(new DBActions().create(
                        BeanMessage.fromMessage(message, System.currentTimeMillis()),
                        BeanMessage.pendingConfig()));
            }
        });
    }

    @Override
    public void onUpdateMessageState(@NonNull final Message message, @NonNull final Message.State state) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (mustIgnoreMessage(message)) {
                    return;
                }

                // 避免频繁的DB操作，以及避免阻塞当前线程，对Progress的进度值不操作更新DB处理
                if (shouldIgnoreMessage(message)) {
                    return;
                }

                if (state instanceof Preparing) {
                    return;
                }

                Identifiable target = message.getTarget();
                if (target == null) {
                    return;
                }

                Chat chat = getChat(target);
                if (chat == null) {
                    return;
                }

                addOrUpdateMessage(chat, message, null);
            }
        });
    }

    @Override
    public void afterSendingMessage(@NonNull final Message message) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (mustIgnoreMessage(message)) {
                    return;
                }

                // 从未决消息表中移除已经处理完成的消息
                if (ChatStoreServiceImpl.this.shouldIgnoreMessage(message)) {
                    return;
                }

                HMR.getService(DBService.class).execute(new DBActions()
                        .deleteById(BeanMessage.pendingConfig(), message.getUuid()));
            }
        });
    }

    @Override
    public void beforeReceivingMessage(@NonNull Message message) {
    }

    @Override
    public void afterReceivingMessage(@NonNull final Message message) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (mustIgnoreMessage(message)) {
                    return;
                }

                if (ChatStoreServiceImpl.this.shouldIgnoreMessage(message)) {
                    return;
                }

                Chat chat = ChatStoreServiceImpl.this.setupConversationIfNeeded(message);
                if (chat == null) {
                    // TODO: Log
                    return;
                }

                ChatStoreServiceImpl.this.addOrUpdateMessage(chat, message, null);
            }
        });
    }

    @Override
    public void onRevokeMessage(@NonNull final Message message) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (shouldIgnoreMessage(message)) {
                    return;
                }

                Identifiable target = message.getTarget();
                if (target == null) {
                    return;
                }

                final Chat chat = getChat(target);
                if (chat == null) {
                    return;
                }

                // 如果消息完整就直接通知
                if (message.getContent() != null) {
                    addOrUpdateMessage(chat, message, null);
                    return;
                }

                Set<String> uuids = new HashSet<>();
                uuids.add(message.getUuid());
                // 为啥要查一次，因为通知过来的消息不全，没有办法复用更新接口
                fetchMessagesByUuids(chat, uuids, new HMR.CompletionArg<List<Message>>() {
                    @Override
                    public void onSuccess(List<Message> messages) {
                        if (messages == null || messages.size() != 1) {
                            Log.e(TAG, Trace.method("onRevokeMessage")
                                    .info("message", messages));
                            return;
                        }

                        Message fullMessage = messages.get(0);
                        fullMessage.setState(message.getState());

                        addOrUpdateMessage(chat, fullMessage, null);
                    }

                    @Override
                    public void onFailed(Error err) {
                        Log.e(TAG, Trace.method("onRevokeMessage")
                                .info("fetchMessagesByUuids", err));
                    }
                });

            }
        });
    }

    @Override
    public void onDeleteMessage(@NonNull Message message) {
        // TODO: delete from db
    }

    private boolean shouldIgnoreMessage(@NonNull Message message) {
        return !filter.shouldAcceptMessage(message);
    }

    private Chat setupConversationIfNeeded(Message message) {
        if (message.getReceiver() instanceof AppSession) {
            Log.i(TAG, Trace.method("parseBroadCastAppSessionChatMessage").msg("setupConversationIfNeeded")
                    .info("receiver type", ((AppSession) message.getReceiver()).getType())
                    .info("id", message.getReceiver().getId()));
        }

        final Identifiable target = message.getTarget();
        if (target == null) {
            return null;
        }

        Chat chat = getChat(target);
        if (chat != null) {
            // 如果已经存在，则无需新增会话
            return chat;
        }

        final Chat finalChat = ChatUtils.newChat(target, null, 0L, message,
                message.getTimestamp(), 0, new LinkedHashSet<String>());
        if (finalChat == null) {
            return null;
        }

        synchronized (chats) {
            chats.add(finalChat);
        }
        HMR.getService(DBService.class).execute(new DBActions()
                .createTableIfNeeded(BeanChatMessage.chatMessageTableConfig(target))
                .createIfNotExists(BeanConversation.fromConversation(finalChat), null)
                .run("Hummer.Run.SetupConversationIfNeeded",
                        new Runnable() {
                            @Override
                            public void run() {
                                Log.w(TAG, Trace.method("setupConversationIfNeeded").msg("Chat Created!"));
                                ChatStoreServiceImpl.this.notifyAfterAddingChats(finalChat);
                            }
                        }));

        return finalChat;
    }

    private void sort() {
        if (chats.size() < 2) {
            return;
        }

        synchronized (chats) {
            Collections.sort(chats, new Comparator<Chat>() {
                @Override
                public int compare(Chat c1, Chat c2) {
                    if (c1.getPriority() == c2.getPriority()) {
                        long x = c2.getTimestamp();
                        long y = c1.getTimestamp();
                        return (x < y) ? -1 : ((x == y) ? 0 : 1);
                    } else {
                        int x = c2.getPriority();
                        int y = c1.getPriority();
                        return (x < y) ? -1 : ((x == y) ? 0 : 1);
                    }
                }
            });
        }
    }

    private boolean mustIgnoreMessage(final Message message) {
        // 例如chatroom消息就解析不出来，目前忽略
        String target = IdentifiableHelper.makeStringFrom(message.getTarget());
        return target == null;
    }

    private final List<Chat> chats = new ArrayList<>();
    private final Set<ChatListener> chatListeners = new HashSet<>();
    private final Map<Identifiable, Set<MessageListener>> msgListeners = new HashMap<>();
    private final Set<MessageListener> commonMsgListeners = new HashSet<>();

    private Filter filter = new Filter() {
        @Override
        public boolean shouldAcceptMessage(Message message) {
            return message.getTarget() instanceof User;
        }
    };

    private static final int SET_TAGS = 1;
    private static final int ADD_TAGS = 2;
    private static final int REMOVE_TAGS = 3;

}
