package com.hummer.im._internals;

import android.text.TextUtils;

import com.hummer.im._internals.chatsvc.IdentifiableHelper;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im._internals.shared.statis.Util;
import com.hummer.im.model.chat.Content;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.chat.PushContent;
import com.hummer.im.model.chat.states.StateFactory;
import com.hummer.im.model.id.IDFactory;
import com.hummer.im.model.id.Identifiable;
import com.j256.ormlite.field.DatabaseField;
import com.j256.ormlite.table.DatabaseTableConfig;

import org.json.JSONException;
import org.json.JSONObject;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import static com.j256.ormlite.field.DataType.BOOLEAN;
import static com.j256.ormlite.field.DataType.INTEGER;
import static com.j256.ormlite.field.DataType.LONG;
import static com.j256.ormlite.field.DataType.STRING;

/**
 * 会话消息数据库实体映射表
 */
public final class BeanChatMessage implements Serializable {

    static final String FIELD_ID = "id";
    static final String FIELD_SENDER = "sender";
    static final String FIELD_RECEIVER = "receiver";
    static final String FIELD_TIMESTAMP = "timestamp";
    static final String FIELD_STATE = "state";
    static final String FIELD_APP_EXTRA = "app_extra";
    static final String FIELD_KV_EXTRA = "kv_extra";
    static final String FIELD_PUSH_CONTENT = "push_content";
    static final String FIELD_CONTENT_TYPE = "content_type";
    static final String FIELD_CONTENT = "content";
    static final String FIELD_NEW_CONTENT = "new_content";
    static final String FIELD_MSG_ID = "msg_id";
    static final String FIELD_DELETED = "deleted";

    /**
     * 唯一标识
     */
    @DatabaseField(columnName = FIELD_ID, dataType = STRING, id = true)
    String id;

    /**
     * 消息发送者
     */
    @DatabaseField(columnName = FIELD_SENDER, dataType = STRING, canBeNull = false, index = true)
    String sender;

    /**
     * 消息接收者
     */
    @DatabaseField(columnName = FIELD_RECEIVER, dataType = STRING, canBeNull = false, index = true)
    String receiver;

    /**
     * 消息时间戳
     */
    @DatabaseField(columnName = FIELD_TIMESTAMP, dataType = LONG, canBeNull = false)
    long timestamp;

    /**
     * 状态: Init、Preparing、Delivering、Archived、Failed
     */
    @DatabaseField(columnName = FIELD_STATE, dataType = STRING, canBeNull = false)
    String state;

    /**
     * 扩展消息
     */
    @DatabaseField(columnName = FIELD_APP_EXTRA, dataType = STRING, canBeNull = false, defaultValue = "")
    String appExtra;

    /**
     * 扩展消息：key-value形式，之后取代appExtra扩展消息
     */
    @DatabaseField(columnName = FIELD_KV_EXTRA, dataType = STRING, canBeNull = false, defaultValue = "")
    String kvExtra;

    @DatabaseField(columnName = FIELD_PUSH_CONTENT, dataType = STRING, canBeNull = false, defaultValue = "")
    String pushContent;

    /**
     * 消息类型：文本、图片、语音、视频等
     */
    @DatabaseField(columnName = FIELD_CONTENT_TYPE, dataType = INTEGER, canBeNull = false)
    int contentType;

    @DatabaseField(columnName = FIELD_CONTENT, dataType = STRING, canBeNull = false)
    String content;

    @DatabaseField(columnName = FIELD_NEW_CONTENT, dataType = STRING, canBeNull = false, defaultValue = "")
    String newContent;

    @DatabaseField(columnName = FIELD_DELETED, dataType = BOOLEAN, canBeNull = false)
    boolean deleted;


    /**
     * 这个厉害了： timestamp + uuid
     */
    @DatabaseField(columnName = FIELD_MSG_ID, dataType = STRING, canBeNull = false, index = true)
    String msgId;

    public BeanChatMessage() {

    }

    private BeanChatMessage(String id,
                            String sender,
                            String receiver,
                            long timestamp,
                            String state,
                            String appExtra,
                            String kvExtra,
                            String pushContent,
                            int contentType,
                            String content,
                            String newContent,
                            String msgId) {
        this.id = id;
        this.sender = sender;
        this.receiver = receiver;
        this.timestamp = timestamp;
        this.state = state;
        this.appExtra = appExtra;
        this.kvExtra = kvExtra;
        this.pushContent = pushContent;
        this.contentType = contentType;
        this.content = content;
        this.newContent = newContent;
        this.deleted = false;
        this.msgId = msgId;
    }

    static DatabaseTableConfig<BeanChatMessage> chatMessageTableConfig(Identifiable target) {
        return chatMessageTableConfig(IdentifiableHelper.hash(target));
    }

    static DatabaseTableConfig<BeanChatMessage> chatMessageTableConfig(int suffix) {
        DatabaseTableConfig<BeanChatMessage> config = new DatabaseTableConfig<>();
        String tableName = "chat_message_" + suffix;
        config.setTableName(tableName);
        config.setDataClass(BeanChatMessage.class);

        return config;
    }


    static Message toMessage(BeanChatMessage dbMessage) {
        Message message = new Message();
        message.setUuid(dbMessage.id);
        message.setTimestamp(dbMessage.timestamp);
        message.setSender(IDFactory.makeId(dbMessage.sender));
        message.setReceiver(IDFactory.makeId(dbMessage.receiver));
        message.setState(StateFactory.makeState(dbMessage.state));
        message.setPushContent(PushContent.makeContent(dbMessage.pushContent));

        // 如果newContent为空，表示content有值，如果是AppContent类型，那么编解码的方式直接走string
        // 如果newContent有值，编解码方式走新的Base64方式
        // 新增newContent的原因在于兼容之前因为string存数据后转成byte[]，会多出字节的问题
        // 之前直接new String(), 而业务服务器的编码则是UTF-16BE或者UTF-16LE(占4个字节)。导致有误
        if (TextUtils.isEmpty(dbMessage.newContent)) {
            message.setContent(Content.makeStringContent(dbMessage.contentType, dbMessage.content));
        } else {
            message.setContent(Content.makeContent(dbMessage.contentType, dbMessage.newContent));
        }
        message.setAppExtra(dbMessage.appExtra);

        if (!TextUtils.isEmpty(dbMessage.kvExtra)) {
            try {
                JSONObject json = new JSONObject(dbMessage.kvExtra);
                Map<String, String> kvExtra = new HashMap<>(json.length());
                Iterator<String> keys = json.keys();
                while (keys.hasNext()) {
                    String key = keys.next();
                    kvExtra.put(key, json.getString(key));
                }

                message.setKvExtra(kvExtra);
            } catch (JSONException e) {
                Log.e(TAG, Trace.method("toMessage")
                        .info("kvExtra", dbMessage.kvExtra)
                        .info("parse error", e.getMessage()));
            }
        } else {
            message.setKvExtra(Collections.<String, String>emptyMap());
        }

        return message;
    }

    static List<Message> toMessages(List<BeanChatMessage> chatMessages) {
        List<Message> messages = new ArrayList<>();

        for (BeanChatMessage chatMessage : chatMessages) {
            Message message = toMessage(chatMessage);
            if (message != null) {
                messages.add(message);
            }
        }

        return messages;
    }

    static BeanChatMessage fromMessage(Message message) {
        long timestamp = message.getTimestamp() < 1L ? System.currentTimeMillis() : message.getTimestamp();

        String kvExtra = "";
        if (message.getKvExtra() != null && message.getKvExtra().size() > 0) {
            JSONObject json = new JSONObject(message.getKvExtra());
            kvExtra = json.toString();
        }

        // 目前消息为ChatRoom的makeString会返回null, 原因在于chatroom还没有注册相应的解析器
        return new BeanChatMessage(message.getUuid(),
                IDFactory.makeString(message.getSender()),
                IDFactory.makeString(message.getReceiver()),
                timestamp,
                StateFactory.makeString(message.getState()),
                message.getAppExtra() == null ? "" : message.getAppExtra(),
                kvExtra,
                PushContent.makeString(message.getPushContent()),
                Content.getDataType(message.getContent()),
                "",
                Content.makeString(message.getContent()),
                Util.getMsgId(timestamp, message.getUuid()));
    }



    @Override
    public String toString() {
        return "BeanChatMessage{" +
                "id='" + id + '\'' +
                '}';
    }

    private static final String TAG = "BeanChatMessage";
}
