package com.hummer.im._internals;

import com.hummer.im.Error;
import com.hummer.im.HMR;
import com.hummer.im._internals.log.Log;
import com.hummer.im._internals.log.trace.Trace;
import com.hummer.im.db.DBActions;
import com.hummer.im.db.DBService;
import com.hummer.im.model.chat.Message;
import com.hummer.im.model.completion.CompletionUtils;
import com.hummer.im.model.completion.RichCompletion;
import com.hummer.im.model.id.IDFactory;
import com.hummer.im.model.id.Identifiable;
import com.j256.ormlite.android.apptools.OrmLiteSqliteOpenHelper;
import com.j256.ormlite.dao.Dao;
import com.j256.ormlite.table.DatabaseTableConfig;

import java.sql.SQLException;
import java.util.List;

public class ActChatUpgrade implements DBService.Action {

    private static String TAG = "ActChatUpgrade";

    @Override
    public void process(OrmLiteSqliteOpenHelper helper, DBService.DaoSet daoSet) throws SQLException {
        Log.i(TAG, Trace.method("process").msg("Upgrade conversation table"));
        final int oldVersion = HMR.getService(DBService.class).getOldVersion();
        final int newVersion = HMR.getService(DBService.class).getNewVersion();

        if (oldVersion == newVersion) {
            CompletionUtils.dispatchSuccess(completion);
            return;
        }

        final Dao<BeanConversation, ?> dao = daoSet.create(null, BeanConversation.class);

        // oldVersion == 2; newVersion == 4;
        if (oldVersion < newVersion) {
            if (oldVersion < 2) {
                upgradeToVersion2(daoSet, dao);

            } else if (oldVersion < 3) {
                upgradeToVersion3(daoSet, dao);

            } else if (oldVersion < 4) {
                upgradeToVersion4(daoSet, dao);

            }
        }
    }

    private void upgradeToVersion2(final DBService.DaoSet daoSet,
                                   final Dao<BeanConversation, ?> dao) throws SQLException {
        String addLatestMsg = "ALTER TABLE `" + dao.getTableName() + "` ADD COLUMN "
                + BeanConversation.FIELD_LATEST_MSG + " TEXT;";
        dao.executeRaw(addLatestMsg);
        String addUnreadNum = "ALTER TABLE `" + dao.getTableName() + "` ADD COLUMN "
                + BeanConversation.FIELD_UNREAD_NUM + " INTEGER;";
        dao.executeRaw(addUnreadNum);
        String addTimestamp = "ALTER TABLE `" + dao.getTableName() + "` ADD COLUMN "
                + BeanConversation.FIELD_TIMESTAMP + " REAL;";
        dao.executeRaw(addTimestamp);

        HMR.getService(DBService.class).execute(new DBActions()
                .queryAll(BeanConversation.class,
                        new DBActions.QueryAcceptor<BeanConversation>() {
                            @Override
                            public void onQueryResults(List<BeanConversation> indexes) {
                                for (BeanConversation index : indexes) {
                                    HMR.getService(DBService.class)
                                            .execute(new ActUpdateChatLatestMsg(
                                                    BeanConversation.toConversation(index)));
                                }

                                try {
                                    upgradeToVersion3(daoSet, dao);
                                } catch (SQLException e) {
                                    Error error = new Error(Error.Code.IOError, "数据库访问异常" + e.getMessage());
                                    Log.e(TAG, error, Trace.method("execute").msg(e.getMessage()));
                                    CompletionUtils.dispatchFailure(completion, error);
                                }
                            }
                        }));
    }

    private void upgradeToVersion3(final DBService.DaoSet daoSet, Dao<BeanConversation, ?> dao) throws SQLException {

        RichCompletion dbComp = new RichCompletion();

        String addPriorityField = "ALTER TABLE " + dao.getTableName() + " ADD COLUMN "
                + BeanConversation.FIELD_PRIORITY + " INTEGER DEFAULT 0 ";
        dao.executeRawNoArgs(addPriorityField);

        String addTagsField = "ALTER TABLE " + dao.getTableName() + " ADD COLUMN "
                + BeanConversation.FIELD_TAGS + " TEXT DEFAULT '[]' ";
        dao.executeRawNoArgs(addTagsField);

        mergeTable(daoSet, dao);
    }

    private void upgradeToVersion4(final DBService.DaoSet daoSet, Dao<BeanConversation, ?> dao) throws SQLException {
        // 64张消息表新增字段newContent
        String addNewContentField = "ALTER TABLE %s ADD COLUMN "
                + BeanChatMessage.FIELD_NEW_CONTENT + " TEXT DEFAULT ''";

        boolean flag = true;
        for (int i = 0; i < 64; i++) {
            final DatabaseTableConfig<BeanChatMessage> tableConfig
                    = BeanChatMessage.chatMessageTableConfig(i);

            Dao<BeanChatMessage, ?> chatDao = daoSet.create(tableConfig, BeanChatMessage.class);

            if (flag) {
                // 加字段
                try {
                    chatDao.executeRawNoArgs(String.format(addNewContentField, tableConfig.getTableName()));
                } catch (SQLException e) {
                    Log.e(TAG, Trace.method("upgradeToVersion4").msg("column `new_content` is exist, ignore"));
                    flag = false;
                }
            }

            // pushContent设置为空字符串
            // update chat_message_? set pushcontent = ''
            chatDao.updateRaw("update " + tableConfig.getTableName()
                    + " set " + BeanChatMessage.FIELD_PUSH_CONTENT + " = ''");
        }

        CompletionUtils.dispatchSuccess(completion);
    }

    /**
     * 并表操作
     */
    private void mergeTable(final DBService.DaoSet daoSet, final Dao<BeanConversation, ?> dao) {
        /*
         * 1. 分表，新增64张表
         * 2. 获取所有的会话列表
         * 3. 根据会话获取到对应旧的消息表
         * 4. 遍历消息表，将消息存入到新的消息表
         */
        DBActions actions = new DBActions();
        for (int i = 0; i < 64; i++) {
            actions.custom(new DBActions()
                    .createTableIfNeeded(BeanChatMessage.chatMessageTableConfig(i)));
        }
        HMR.getService(DBService.class).execute(actions);

        HMR.getService(DBService.class).execute(new DBActions().queryAll(BeanConversation.class,
                new DBActions.QueryAcceptor<BeanConversation>() {
                    @Override
                    public void onQueryResults(List<BeanConversation> indexes) {
                        for (BeanConversation index : indexes) {
                            Identifiable target = IDFactory.makeId(index.name);
                            final DatabaseTableConfig<BeanChatMessage> newMsgTableConfig
                                    = BeanChatMessage.chatMessageTableConfig(target);
                            DatabaseTableConfig<BeanMessage> oldMsgTableConfig
                                    = BeanMessage.conversationConfig(target);

                            HMR.getService(DBService.class).execute(new DBActions()
                                    .queryAll(oldMsgTableConfig, new DBActions.QueryAcceptor<BeanMessage>() {
                                        @Override
                                        public void onQueryResults(List<BeanMessage> result) {

                                            List<Message> messages = BeanMessage.toMessages(result);
                                            for (Message msg : messages) {
                                                HMR.getService(DBService.class).execute(new DBActions()
                                                        .create(BeanChatMessage.fromMessage(msg), newMsgTableConfig));
                                            }
                                        }
                                    }));
                        }

                        try {
                            upgradeToVersion4(daoSet, dao);
                        } catch (SQLException e) {
                            Error error = new Error(Error.Code.IOError, "数据库访问异常" + e.getMessage());
                            Log.e(TAG, error, Trace.method("execute").msg(e.getMessage()));
                            CompletionUtils.dispatchFailure(completion, error);
                        }
                    }
                }));
    }

    ActChatUpgrade(RichCompletion completion) {
        this.completion = completion;
    }

    private RichCompletion completion;
}
