package com.hummer._internals.yyp.packet;

import android.annotation.SuppressLint;

import java.io.UnsupportedEncodingException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

public class Unpack {

    protected ByteBuffer buffer;

    public Unpack(byte[] bytes) {
        this(bytes, 0, bytes.length);
    }

    public Unpack(byte[] bytes, int offset, int length) {
        buffer = ByteBuffer.wrap(bytes, offset, length);
        buffer.order(ByteOrder.LITTLE_ENDIAN);
    }

    public int size() {
        return buffer.remaining();
    }

    public Uint32 popUint32() {
        return new Uint32(buffer.getInt());
    }

    public int popInt() {
        return buffer.getInt();
    }

    public long popLong() {
        return buffer.getLong();
    }

    public Uint8 popUint8() {
        return new Uint8(buffer.get());
    }

    public Uint16 popUint16() {
        return new Uint16(buffer.getShort());
    }

    public Int64 popInt64() {
        return new Int64(buffer.getLong());
    }

    public Uint64 popUint64() {
        return new Uint64(buffer.getLong());
    }

    public boolean popBoolean() {
        return buffer.get() == 1 ? true : false;
    }

    public byte[] popBytes() {
        short size = buffer.getShort();
        byte[] dst = new byte[toUnsigned(size)];
        buffer.get(dst);
        return dst;
    }

    public int toUnsigned(short s) {
        return s & 0xFFFF;
    }

    public String popString() {
        try {
            byte[] dst = popBytes();
            return new String(dst, "utf-8");
        } catch (UnsupportedEncodingException e) {
            throw new UnpackException();
        }
    }

    public String popString(String encoding) {
        try {
            byte[] dst = popBytes();
            return new String(dst, encoding);
        } catch (UnsupportedEncodingException e) {
            throw new UnpackException();
        }
    }

    public String popString32() {
        try {
            int size = buffer.getInt();
            byte[] dst = new byte[size];
            buffer.get(dst);
            return new String(dst, "utf-8");
        } catch (UnsupportedEncodingException e) {
            throw new UnpackException();
        }
    }

    public byte[] popBytes32() {
        Uint32 size = new Uint32(buffer.getInt());
        byte[] dst = new byte[(int) size.longValue()];
        buffer.get(dst);
        return dst;
    }

    @Override
    public String toString() {
        StringBuilder builder = new StringBuilder();
        builder.append("Pack [buffer=");
        builder.append(bufferString());
        builder.append("]");
        return builder.toString();
    }

    @SuppressLint("DefaultLocale")
    private String bufferString() {
        byte[] b = new byte[buffer.remaining()];
        int oldPosition = buffer.position();
        buffer.get(b);
        buffer.position(oldPosition);
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < b.length; i++) {
            sb.append(Integer.toHexString(b[i] & 0xFF).toUpperCase()).append(" ");
        }
        return sb.toString();
    }

    /**
     * 读取复合对像
     *
     * @return
     */
    public Map<String, String> popMapByKeyStringAndValueString() {
        Map<String, String> resultMap = new HashMap<>();
        //获取Map的长度
        int size = buffer.getInt();
        for (int pos = 0; pos < size; pos++) {
            //获取String Key
            String key = this.popString();
            //读取Set的长度
            String value = this.popString();
            resultMap.put(key, value);
        }
        return resultMap;
    }

    /**
     * 读取复合对像
     *
     * @return
     */
    public Map<String, Set<Uint64>> popMapByKeyStringAndValueUint64Set() {
        Map<String, Set<Uint64>> resultMap = new HashMap<>();
        //获取Map的长度
        int size = buffer.getInt();
        for (int pos = 0; pos < size; pos++) {
            //获取String Key
            String key = this.popString();
            //读取Set的长度
            int setSize = buffer.getInt();
            Set<Uint64> uint64Set = new HashSet<>();
            for (int setPos = 0; setPos < setSize; setPos++) {
                //获取Uinit64的Set
                uint64Set.add(this.popUint64());
            }
            resultMap.put(key, uint64Set);
        }
        return resultMap;
    }

    /**
     * 获取Uint64Set
     *
     * @return
     */
    public Set<Uint64> popUint64Set() {
        Set<Uint64> uint64Set = new HashSet<>();
        int setSize = buffer.getInt();
        for (int setPos = 0; setPos < setSize; setPos++) {
            //获取Uinit64的Set
            uint64Set.add(this.popUint64());
        }
        return uint64Set;
    }
}
