package com.hummer._internals.yyp;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.Error;
import com.hummer.ErrorEnum;
import com.hummer._internals.channel.Channel;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.report.StatisticsReporter;
import com.hummer._internals.shared.StringChain;
import com.hummer._internals.yyp.packet.Marshallable;
import com.hummer._internals.yyp.packet.Receiver;
import com.hummer._internals.yyp.packet.Sender;
import com.hummer._internals.yyp.packet.Uint32;

import java.lang.reflect.Field;
import java.lang.reflect.ParameterizedType;
import java.util.Locale;

public abstract class YYPRPC<Response extends Marshallable> implements Channel.RPC {

    public abstract Marshallable requestObj();

    public abstract void handleSuccess(@NonNull Response res) throws Throwable;

    public abstract String getServiceName();

    @Override
    public String protoType() {
        return null;
    }

    @Override
    public byte[] getRequestBytes() {
        Sender sender = new Sender(requestObj());
        sender.endPack();
        startTs = System.currentTimeMillis();
        Log.i("YYPRPC", flow.method(getFunctionName())
                .msg("-> {" + requestObj().toString() + "}"));
        return sender.getBytes();
    }

    @Override
    @SuppressWarnings("unchecked")
    public void handleResponse(@NonNull byte[] responseBytes) throws Throwable {
        Class responseClass = (Class) ((ParameterizedType) this.getClass()
                .getGenericSuperclass())
                .getActualTypeArguments()[0];

        Response res = (Response) responseClass.newInstance();
        Receiver receiver = new Receiver(responseBytes);
        receiver.unmarshallWrap2(res);

        Field resCodeField;
        try {
            resCodeField = res.getClass().getDeclaredField("res");
        } catch (Exception e) {
            resCodeField = res.getClass().getDeclaredField("rescode");
        }

        if (resCodeField == null) {
            Log.e(TAG, Trace.method("handleResponse").msg("resCodeField is <null>"));
        }

        int resCode = ((Uint32) resCodeField.get(res)).intValue();

        Log.i(TAG, Trace.method("YYPRPC.handleResponse")
                .info("code", resCode)
                .info("resCls", res.getClass().getSimpleName()));

        long rtt = System.currentTimeMillis() - startTs;
        if (resCode == ErrorEnum.OK.getCode()) {
            handleSuccess(res);
            Log.i("YYPRPC", flow.method(getFunctionName())
                    .msg("<- Succeed {" + res.toString() + "}"));
        } else {
            Field msgField = res.getClass().getDeclaredField("msg");
            String msg = "service error";

            if (msgField != null) {
                msg = msgField.get(res) == null ? msg : (String) msgField.get(res);
            }

            Error error = Error.ofError(resCode, msg);
            handleError(res, error);
            Log.e("YYPRPC", error, flow.method(getFunctionName())
                    .msg("<- Failed {" + new StringChain()
                            .add("logId",    flow.logId)
                            .add("response", res.toString())
                            .add("msg", msg)
                            .toString() + "}"));
        }

        StatisticsReporter.batchReportReturnCode(getFunctionName(), rtt, resCode);
    }

    @Override
    public String serviceName() {
        return String.format(Locale.US, "svc_%s", getServiceName());
    }

    public void handleError(@Nullable Response res, @NonNull Error error) {
        handleError(error);
    }

    private long startTs;
    private Trace.Flow flow = new Trace.Flow();
    private static final String TAG = "YYPRPC";
}
