package com.hummer._internals.utility;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;
import android.text.TextUtils;

import com.hummer.BuildConfig;
import com.hummer.Error;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer.model.completion.OnFailure;
import com.hummer.model.completion.OnSuccess;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * CIM包含了许多服务接口，例如ChatService, ……但它们的具体实现是通过注册的方式来设置的。
 * Java 6开始，提供SPI的，例如：
 *
 * @see <a href="https://juejin.im/post/5af952fdf265da0b9e652de3">Java SPI机制详解</a>
 * 但是，Android中使用起来较为复杂，因此，这里使用Service类型简单地实现了SPI的能力，让用户可以通过纯接口的
 * 方式获得CIM内部提供的服务，以此实现接口和实现的分离。
 */
public class ServiceProvider {

    private static final String TAG = "ServiceProvider";

    public interface ModuleLoader {
        void initModule();

        void deInitModule();

        Map<Class, Service> getServices();
    }

    public interface Service {

        Class[] staticDependencies();

        Class[] inherentDynamicDependencies();

        Class[] plantingDynamicDependencies();

        void initService();

        void openService(@NonNull RichCompletion completion);

        void closeService();

        void deInitService();
    }

    public static void openServices(
            @Nullable final RichCompletion completion) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                final List<Service> sortedServices = topologicalSort(dynamicDependencies);
                Log.i(TAG, Trace.method("openServices").msg(namesForServices(sortedServices)));

                openServices(sortedServices, 0, completion);
            }
        });
    }

    public static void loadServicesIfNeeded(Context appContext, String packageName) {
        if (isLoaded) {
            return;
        }

        Set<Class<?>> loaderClasses = ReflectionExt.loadClasses(
                appContext,
                BuildConfig.BUILD_VERSION,
                packageName,
                Object.class,
                new ReflectionExt.Filter() {
                    @Override
                    public boolean shouldAccept(String className) {
                        String simpleName = className.substring(className.lastIndexOf(".") + 1);
                        return simpleName.startsWith("SP") && simpleName.endsWith("ModuleLoader");
                    }
                }
        );

        try {
            logLoadedModules(loaderClasses);
            registerServicesByModuleLoaders(loaderClasses);
        } catch (Exception e) {
            Log.e(TAG, Trace.method("loadServicesIfNeeded")
                    .msg("Fail loading services" + e.getLocalizedMessage()));
        }

        logDependencies("Static", staticDependencies);
        logDependencies("Dynamic", dynamicDependencies);

        final List<Service> sortedServices = topologicalSort(staticDependencies);

        for (Service service : sortedServices) {
            service.initService();
            Log.i(TAG, Trace.method("initService").msg(simpleServiceName(service.getClass())));
        }

        isLoaded = true;
    }

    public static void unLoadServices() {
        if (!isLoaded) {
            return;
        }

        final List<Service> sortedServices = topologicalSort(staticDependencies);

        Log.i(TAG, Trace.method("unLoadServices")
                .msg(namesForServices(sortedServices)));

        // unLoad的顺序应该和load的顺序是镜像对称的
        Collections.reverse(sortedServices);

        for (Service service : sortedServices) {
            service.deInitService();
            Log.i(TAG, Trace.method("unLoadServices").msg(simpleServiceName(service.getClass())));
        }

        for (ModuleLoader loader : loaders) {
            loader.deInitModule();
        }

        loaders.clear();
        services.clear();
        staticDependencies.clear();
        dynamicDependencies.clear();

        isLoaded = false;
    }

    private static void registerServicesByModuleLoaders(Set<Class<?>> loaderClasses)
            throws InstantiationException, IllegalAccessException {
        for (Class<?> loaderClazz : loaderClasses) {
            ModuleLoader loader = (ModuleLoader) loaderClazz.newInstance();

            loader.initModule();

            loaders.add(loader);

            Map<Class, Service> services = loader.getServices();
            for (Map.Entry<Class, Service> entry : services.entrySet()) {
                ServiceProvider.register(entry.getKey(), entry.getValue());
            }
        }
    }

    private static void logLoadedModules(Set<Class<?>> loaderClasses) {
        List<String> moduleNames = new ArrayList<>();
        for (Class<?> loaderClazz : loaderClasses) {
            moduleNames.add(simpleModuleName(loaderClazz.getSimpleName()));
        }

        Log.i(TAG, Trace.method("loadedModules")
                .msg(TextUtils.join(", ", moduleNames)));
    }

    // 使用mermaid DSL进行依赖输出，以便通过常用的markdown工具可以对依赖关系进行可视化分析
    // 例如：https://stackedit.io
    private static void logDependencies(String name, Map<Class, Set<Class>> dependencies) {
        StringBuilder sb = new StringBuilder();

        sb.append("```mermaid\n").append("graph TD\n");

        for (Map.Entry<Class, Set<Class>> oneService : dependencies.entrySet()) {
            Class from = oneService.getKey();

            if (oneService.getValue().size() == 0) {
                sb.append(simpleServiceName(from)).append("\n");
            }

            for (Class toClazz : oneService.getValue()) {
                sb.append(simpleServiceName(from))
                        .append("-->")
                        .append(simpleServiceName(toClazz))
                        .append("\n");
            }
        }

        sb.append("```");
        Log.i(TAG, Trace.method("dependencies").msg("%s\n%s", name, sb.toString()));
    }

    private static String simpleServiceName(Class clazz) {
        String name = clazz.getSimpleName();

        if (name.endsWith("Service")) {
            name = name.substring(0, name.length() - "Service".length());
        }

        return name;
    }

    private static String namesForServices(List<Service> services) {
        List<String> names = new ArrayList<>();
        for (Service svc : services) {
            names.add(svc.getClass().getSimpleName());
        }

        return TextUtils.join(", ", names);
    }

    private static String simpleModuleName(String fullName) {
        if (!fullName.startsWith("SP")) {
            Log.e(TAG, Trace.method("simpleModuleName")
                    .info("SP", fullName));
        }

        if (!fullName.endsWith("ModuleLoader")) {
            Log.e(TAG, Trace.method("simpleModuleName")
                    .info("ModuleLoader", fullName));
        }

        return fullName.substring(2, fullName.length() - 12);
    }

    @SuppressWarnings("unchecked")
    private static void openServices(
            final List<Service> allServices,
            final int atPosition,
            final RichCompletion completion) {
        if (atPosition >= allServices.size()) {
            CompletionUtils.dispatchSuccess(completion);
            return;
        }

        final Service service = allServices.get(atPosition);
        service.openService(new RichCompletion()
                .onSuccess(new OnSuccess() {
                    @Override
                    public void onSuccess() {
                        Log.i(TAG, Trace.method("openService")
                                .msg("Success - %s", service.getClass().getSimpleName()));

                        openServices(allServices, atPosition + 1, completion);
                    }
                })
                .onFailure(new OnFailure() {
                    @Override
                    public void onFailure(Error error) {
                        Log.e(TAG, error, Trace.method("openService")
                                .msg("Failed - %s", service.getClass().getSimpleName()));

                        // Unwind services
                        for (int pos = atPosition - 1; pos >= 0; --pos) {
                            allServices.get(pos).closeService();
                        }

                        CompletionUtils.dispatchFailure(completion, error);
                    }
                })
        );
    }

    public static void closeServices() {
        final List<Service> sortedServices = topologicalSort(dynamicDependencies);

        Log.i(TAG, Trace.method("closeServices")
                .msg(namesForServices(sortedServices)));

        // close的顺序应该和open的顺序是镜像对称的
        Collections.reverse(sortedServices);

        for (Service service : sortedServices) {
            service.closeService();

            Log.i(TAG, Trace.method("closeService").msg(service.getClass().getSimpleName()));
        }
    }

    /**
     * 获取Interface类型Service的实现实例，在此之前，必须通过register方法注册了该接口的实例
     *
     * @param serviceClass service的类型，例如ChatService.class
     * @return 如果找到了serviceClass对应的服务，则返回其服务实例
     */
    @SuppressWarnings("unchecked")
    public static <Interface> Interface get(Class<Interface> serviceClass) {
        if (serviceClass == null) {
            Log.e(TAG, Trace.method("get")
                    .info("Exception", "null"));
            return null;
        }

        for (Map.Entry<Class, Service> impl : services.entrySet()) {
            if (serviceClass.isAssignableFrom(impl.getKey())) {
                return (Interface) impl.getValue();
            }
        }

        Service service = services.get(serviceClass);

        if (service == null) {
            Log.e(TAG, Trace.method("get")
                    .info("Service Not Found:", serviceClass.getName()));
        }

        return (Interface) service;
    }

    @SuppressWarnings("WeakerAccess")
    public static void register(Class<?> interfaceClass, Service service) {
        services.put(interfaceClass, service);

        staticDependencies.put(interfaceClass, setFromArray(service.staticDependencies()));

        acquireDynamicDependencies(interfaceClass).addAll(setFromArray(service.inherentDynamicDependencies()));

        for (Class clazz : setFromArray(service.plantingDynamicDependencies())) {
            acquireDynamicDependencies(clazz).add(interfaceClass);
        }
    }

    private static Set<Class> acquireDynamicDependencies(Class<?> serviceClass) {
        Set<Class> dependencies = dynamicDependencies.get(serviceClass);
        if (dependencies == null) {
            dependencies = new HashSet<>();
            dynamicDependencies.put(serviceClass, dependencies);
        }

        return dependencies;
    }

    private static Set<Class> setFromArray(Class[] classes) {
        if (classes == null) {
            return new HashSet<>();
        } else {
            return new HashSet<>(Arrays.asList(classes));
        }
    }

    // 拓扑排序原理请见：
    // https://en.wikipedia.org/wiki/Topological_sorting
    private static @NonNull
    List<Service> topologicalSort(Map<Class, Set<Class>> graph) {
        Map<Class, Set<Class>> outEdges = cloneDependencies(graph);
        Map<Class, Set<Class>> inEdges = new HashMap<>();

        int edges = 0;

        for (Class node : outEdges.keySet()) {
            inEdges.put(node, new HashSet<Class>());
        }

        List<Class> leafClasses = new ArrayList<>();

        for (Map.Entry<Class, Set<Class>> entry : outEdges.entrySet()) {
            Class from = entry.getKey();

            for (Class to : entry.getValue()) {
                inEdges.get(to).add(from);
                edges += 1;
            }

            if (entry.getValue().size() == 0) {
                leafClasses.add(from);
            }
        }

        ArrayList<Class> sortedClasses = new ArrayList<>();

        while (!leafClasses.isEmpty()) {
            Class n = leafClasses.remove(0);
            sortedClasses.add(n);

            for (Class from : inEdges.get(n)) {
                outEdges.get(from).remove(n);
                edges -= 1;
                if (outEdges.get(from).size() == 0) {
                    leafClasses.add(from);
                }
            }
        }

        if (edges != 0) {
            Log.e(TAG, Trace.method("topologicalSort")
                    .info("服务依赖异常，请检查它们的依赖关系", ""));
        }

        ArrayList<Service> sortedServices = new ArrayList<>();
        for (Class clazz : sortedClasses) {
            sortedServices.add(services.get(clazz));
        }

        return sortedServices;
    }

    private static Map<Class, Set<Class>> cloneDependencies(Map<Class, Set<Class>> originDependencies) {
        Map<Class, Set<Class>> copy = new HashMap<>();

        for (Map.Entry<Class, Set<Class>> entry : originDependencies.entrySet()) {
            copy.put(entry.getKey(), new HashSet<>(entry.getValue()));
        }

        return copy;
    }

    private static boolean isLoaded = false;
    private static final Set<ModuleLoader> loaders = new HashSet<>();
    private static final HashMap<Class, Service> services = new HashMap<>();
    private static final HashMap<Class, Set<Class>> staticDependencies = new HashMap<>();
    private static final HashMap<Class, Set<Class>> dynamicDependencies = new HashMap<>();
}
