package com.hummer._internals.utility;

import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.Error;
import com.hummer.HMR;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer.model.RequestId;
import com.hummer.model.completion.OnFailure;
import com.hummer.model.completion.OnSuccessArg;

public final class RichCompletionArg<T> {

    public RichCompletionArg() {
        Looper looper = Looper.myLooper();
        if (looper == Looper.getMainLooper()) {
            this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(Looper.getMainLooper()));
        } else if (looper == HMRContext.work.getHandler().getLooper()) {
            // 回到work队列
            this.dispatchQueue = HMRContext.work;
        } else {
            // 默认使用Direct分派
            this.dispatchQueue = DispatchQueue.direct;
        }
    }

    public RichCompletionArg(@NonNull final RequestId requestId, @Nullable final HMR.CompletionArg<T> completion) {
        this();
        onSuccess(new OnSuccessArg<T>() {
            @Override
            public void onSuccess(T result) {
                if (completion != null) {
                    Log.i(TAG, Trace.method("RichCompletionArg").msg("Success")
                            .info("hummerRequestId", requestId));
                    completion.onSuccess(requestId, result);
                }
            }
        }).onFailure(new OnFailure() {
            @Override
            public void onFailure(Error error) {
                if (completion != null) {
                    Log.i(TAG, Trace.method("RichCompletionArg").msg("Fail")
                            .info("hummerRequestId", requestId)
                            .info("Error", error.toString()));
                    completion.onFailed(requestId, error);
                }
            }
        });
    }

    public RichCompletionArg(@NonNull Handler handler) {
        this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(handler.getLooper()));
    }

    public RichCompletionArg<T> onSuccess(@NonNull OnSuccessArg<T> success) {
        this.successHandler = success;
        return this;
    }

    public RichCompletionArg<T> onFailure(@NonNull OnFailure failure) {
        this.failureHandler = failure;
        return this;
    }

    void dispatchSuccess(@NonNull T result) {

        final T finalResult = result;
        dispatchQueue.async(new Runnable() {
            @Override
            public void run() {
                if (successHandler != null) {
                    successHandler.onSuccess(finalResult);
                } else {
                    Log.i(TAG, Trace.method("dispatchSuccess")
                            .msg("dispatchQueue async, success handler = null"));
                }
            }
        });
    }

    void dispatchFailure(@NonNull Error error) {
        final Error finalError = error;
        dispatchQueue.async(new Runnable() {
            @Override
            public void run() {
                if (failureHandler != null) {
                    failureHandler.onFailure(finalError);
                } else {
                    Log.i(TAG, Trace.method("dispatchFailure")
                            .msg("dispatchQueue async, failure handler = null"));
                }
            }
        });
    }

    private static final String TAG = "RichCompletionArg";
    private final DispatchQueue dispatchQueue;
    private OnFailure failureHandler;
    private OnSuccessArg<T> successHandler;
}
