package com.hummer._internals.utility;

import android.os.Handler;
import android.os.Looper;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.Error;
import com.hummer.HMR;
import com.hummer.model.RequestId;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer.model.completion.OnFailure;
import com.hummer.model.completion.OnSuccess;

import java.util.LinkedList;
import java.util.List;

public final class RichCompletion {

    public RichCompletion() {
        Looper looper = Looper.myLooper();
        if (looper == Looper.getMainLooper()) {
            this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(Looper.getMainLooper()));
        } else if (looper == HMRContext.work.getHandler().getLooper()) {
            // 回到work队列
            this.dispatchQueue = HMRContext.work;
        } else {
            // 默认使用Direct分派
            this.dispatchQueue = DispatchQueue.direct;
        }

    }

    public RichCompletion(@NonNull final RequestId requestId, @Nullable final HMR.Completion completion) {
        this();

        this.onSuccess(new OnSuccess() {
            @Override
            public void onSuccess() {
                if (completion != null) {
                    Log.i(TAG, Trace.method("RichCompletion").msg("Success")
                            .info("hummerRequestId", requestId));
                    completion.onSuccess(requestId);
                }
            }
        }).onFailure(new OnFailure() {
            @Override
            public void onFailure(Error error) {
                if (completion != null) {
                    Log.i(TAG, Trace.method("RichCompletion").msg("Fail")
                            .info("hummerRequestId", requestId)
                            .info("Error", error.toString()));
                    completion.onFailed(requestId, error);
                }
            }
        });
    }

    public RichCompletion(@NonNull Handler handler) {
        this.dispatchQueue = new DispatchQueue(new DispatchQueue.LooperHandlerProvider(handler.getLooper()));
    }

    public RichCompletion decorate(@Nullable RichCompletion decorated) {
        if (this.decorated == null) {
            Log.e(TAG, Trace.method("decorate")
                    .info("只允许对Completion进行一次装饰", this.decorated));
            return this;
        }

        this.decorated = decorated;
        return this;
    }

    public RichCompletion beforeSuccess(@NonNull Runnable runnable) {
        this.preSuccess.add(runnable);
        return this;
    }

    public RichCompletion onSuccess(@NonNull OnSuccess success) {
        this.successHandler = success;
        return this;
    }

    public RichCompletion afterSuccess(@NonNull Runnable runnable) {
        this.postSuccess.add(0, runnable);
        return this;
    }

    public RichCompletion beforeFailure(@NonNull Runnable runnable) {
        this.preFailure.add(runnable);
        return this;
    }

    public RichCompletion onFailure(@NonNull OnFailure failure) {
        this.failureHandler = failure;
        return this;
    }

    public RichCompletion afterFailure(@NonNull Runnable runnable) {
        this.postFailure.add(0, runnable);
        return this;
    }

    void dispatchSuccess() {
        dispatchQueue.async(new Runnable() {
            @Override
            public void run() {
                dispatchActions(preSuccess);

                if (successHandler != null) {
                    successHandler.onSuccess();
                } else {
                    Log.i(TAG, Trace.method("dispatchSuccess")
                            .msg("dispatchSuccess, successHandler == null"));
                }

                CompletionUtils.dispatchSuccess(decorated);

                dispatchActions(postSuccess);
            }
        });
    }

    void dispatchFailure(@NonNull final Error error) {
        dispatchQueue.async(new Runnable() {
            @Override
            public void run() {
                dispatchActions(preFailure);

                if (failureHandler != null) {
                    failureHandler.onFailure(error);
                } else {
                    Log.i(TAG, Trace.method("dispatchFailure")
                            .msg("dispatchFailure, failureHandler == null"));
                }

                CompletionUtils.dispatchFailure(decorated, error);

                dispatchActions(postFailure);
            }
        });
    }

    private static void dispatchActions(@NonNull List<Runnable> actions) {
        for (Runnable r : actions) {
            r.run();
        }
    }

    private static final String TAG = "RichCompletion";

    private final DispatchQueue dispatchQueue;
    private List<Runnable> preSuccess = new LinkedList<>();
    private OnSuccess successHandler;
    private List<Runnable> postSuccess = new LinkedList<>();

    private List<Runnable> preFailure = new LinkedList<>();
    private OnFailure failureHandler;
    private List<Runnable> postFailure = new LinkedList<>();

    private RichCompletion decorated;
}
