package com.hummer._internals.utility;

import android.content.Context;
import android.content.SharedPreferences;
import android.support.annotation.NonNull;

import com.hummer.Error;
import com.hummer.ErrorEnum;
import com.hummer.HMR;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.user.UserService;

import java.util.Locale;

public final class PrefStorage implements ServiceProvider.Service {

    private static final String TAG = "PrefStorage";
    private static PrefStorage sharedStorage;
    private static int NumberOfClosing = 0;
    private static int NumberOfOpening = 0;
    private SharedPreferences preferences = null;
    private Long me = null;

    // Prevent user initiation
    private PrefStorage() {
    }

    public static PrefStorage storage() {
        // 不会产生多线程竞争问题，因此不需要多线程锁、Double Check Locking等模式介入
        if (sharedStorage == null) {
            sharedStorage = new PrefStorage();
        }

        return sharedStorage;
    }

    public <ResultType> ResultType execute(Query<ResultType> query) {
        if (!isValidContext()) {
            Log.w(TAG, Trace.method("execute").msg("Bad Context : %s ", serviceSnapshot()));
            return null;
        }

        return query.run(this.preferences);
    }

    public void execute(Edit edit) {
        if (!isValidContext()) {
            Log.w(TAG, Trace.method("execute").msg("Bad Context : %s ", serviceSnapshot()));
            return;
        }

        SharedPreferences.Editor editor = preferences.edit();
        edit.run(editor);
        editor.apply();
    }

    private boolean isValidContext() {
        return (this.preferences != null) && HMR.isMe(this.me);
    }

    private String serviceSnapshot() {
        return "me: " + HMR.getMe()
                + ", isRunning: " + (preferences != null)
                + ", opening: " + NumberOfOpening
                + ", closing: " + NumberOfClosing;
    }

    public interface Query<ResultType> {
        ResultType run(SharedPreferences pref);
    }

    public interface Edit {
        void run(SharedPreferences.Editor editor);
    }

    @Override
    public Class[] staticDependencies() {
        return null;
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return new Class[0];
    }

    @Override
    public void initService() {

    }

    @Override
    public void deInitService() {

    }

    @Override
    public void openService(@NonNull RichCompletion completion) {
        PrefStorage.NumberOfOpening += 1;

        if (this.me != null || this.preferences != null) {
            Log.w(TAG, Trace.method("login").msg("Nonsense opening: " + serviceSnapshot()));
            return;
        }

        this.me = HMR.getMe();

        if (this.me == null) {
            Log.e("PrefStorage", Trace.method("openService")
                    .info("无效的Me对象:" + HMR.getMe() + ", " + serviceSnapshot(), null));
            CompletionUtils.dispatchFailure(completion, new Error(ErrorEnum.BAD_USER_ERROR, "User not login."));
            return;
        }

        HMRContext.Region env = HMRContext.region;
        String serviceName = String.format(Locale.US, "svc_%s_cim_proxy", env.area);

        String prefName = String.format(Locale.US, "im_%d_%d_%s",
                HMRContext.getAppId(),
                this.me,
                serviceName);
        this.preferences = HMRContext.getAppContext().getSharedPreferences(prefName, Context.MODE_PRIVATE);

        Log.i(TAG, Trace.method("login").info("prefName", prefName));

        if (this.preferences == null) {
            Log.e("PrefStorage", Trace.method("openService")
                    .info("无法创建Preferences文件 | " + serviceSnapshot(), null));
        }
        CompletionUtils.dispatchSuccess(completion);
    }

    @Override
    public void closeService() {
        PrefStorage.NumberOfClosing += 1;

        Log.i(TAG, Trace.method("logout"));

        if (!isValidContext()) {
            Log.w(TAG, Trace.method("logout").msg("Nonsense closing: ", serviceSnapshot()));
            return;
        }

        this.preferences = null;
        this.me = null;
    }
}


