package com.hummer._internals.utility;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.google.protobuf.MessageLite;
import com.hummer.Error;
import com.hummer.ErrorEnum;
import com.hummer.HMR;
import com.hummer._internals.channel.Channel;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.report.StatisticsReporter;
import com.hummer._internals.shared.StringChain;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.util.Locale;

public abstract class IMRPC<Request, RequestBuilder, Response> implements Channel.RPC {

    public abstract void buildHummerRequest(@NonNull RequestBuilder builder) throws Throwable;

    public String describeHummerRequest(Request req) {
        return req.toString();
    }

    public abstract void handleHummerSuccess(@NonNull Response res) throws Throwable;

    public abstract void handleHummerError(@Nullable Response res, @NonNull Error error);

    public abstract String getHummerFunction();

    public String describeHummerResponse(@NonNull Response res) {
        return res.toString();
    }

    @Override
    public String serviceName() {
        HMRContext.Region env = HMRContext.region;
        return String.format(Locale.US, "svc_%s_cim_proxy", env.area);
    }

    @Override
    public String getFunctionName() {
        return FunctionNamePrefix + "." + getHummerFunction();
    }

    @Override
    public String protoType() {
        return "yy-rp-pb";
    }

    public long getLogId() {
        return flow.logId;
    }

    @Override
    @SuppressWarnings("unchecked")
    public byte[] getRequestBytes() throws Throwable {
        Class requestClazz = (Class) ((ParameterizedType) this.getClass()
                .getGenericSuperclass())
                .getActualTypeArguments()[0];

        Method newBuilder = requestClazz.getMethod("newBuilder");

        RequestBuilder builder = (RequestBuilder) newBuilder.invoke(null);
        if (builder == null) {
            Log.e("IMRPC", Trace.method("getRequestBytes")
                    .info("RequestBuilder", null));
        }

        assert builder != null;
        Method setLogId = builder.getClass().getMethod("setLogId", long.class);
        if (setLogId == null) {
            Log.e("IMRPC", Trace.method("getRequestBytes")
                    .info("Method setLogId", null));
        }
        assert setLogId != null;
        setLogId.invoke(builder, getLogId());

        Method setAppId = builder.getClass().getMethod("setAppId", long.class);
        if (setAppId == null) {
            Log.e("IMRPC", Trace.method("getRequestBytes")
                    .info("Method setAppId", null));
        }
        assert setAppId != null;
        setAppId.invoke(builder, HMRContext.appId);

        try {
            Method setSelfUid = builder.getClass().getMethod("setSelfUid", long.class);
            setSelfUid.invoke(builder, HMR.getMe());
        } catch (NoSuchMethodException e) {
            // setSelfUid 不是必须的
        }

        Method build = builder.getClass().getMethod("build");

        buildHummerRequest(builder);
        MessageLite request = (MessageLite) build.invoke(builder);

        Log.i("IMRPC", flow.method(getHummerFunction())
                .msg("-> {" + describeHummerRequest((Request) request)) + "}");

        startTs = System.currentTimeMillis();

        return request.toByteArray();
    }

    @Override
    public void handleError(@NonNull Error err) {
        Log.e("IMRPC", err, flow.method(getHummerFunction())
                .msg("<- Failed {" + new StringChain()
                        .add("logId", getLogId())
                        .toString() + "}"));
        handleHummerError(null, err);
    }

    @Override
    @SuppressWarnings("unchecked")
    public void handleResponse(@NonNull byte[] responseBytes) throws Throwable {
        Class responseClass = (Class) ((ParameterizedType) this.getClass()
                .getGenericSuperclass())
                .getActualTypeArguments()[2];

        if (responseClass == null) {
            Log.e("IMRPC", Trace.method("handleResponse")
                    .info("错误的HummerRPC继承实现: ", null));
        }

        assert responseClass != null;
        Method newBuilder = responseClass.getMethod("newBuilder");
        if (newBuilder == null) {
            Log.e("IMRPC", Trace.method("handleResponse")
                    .info("Method newBuilder: ", null));
        }

        assert newBuilder != null;
        Object builder = newBuilder.invoke(null);
        if (builder == null) {
            Log.e("IMRPC", Trace.method("handleResponse")
                    .info("Object builder: ", null));
        }

        assert builder != null;
        Method mergeFrom = builder.getClass().getMethod("mergeFrom", byte[].class);
        if (mergeFrom == null) {
            Log.e("IMRPC", Trace.method("handleResponse")
                    .info("Method mergeFrom: ", null));
        }

        Method build = builder.getClass().getMethod("build");
        if (build == null) {
            Log.e("IMRPC", Trace.method("handleResponse")
                    .info("Method build: ", null));
        }

        assert mergeFrom != null;
        mergeFrom.invoke(builder, (Object) responseBytes);
        assert build != null;
        Response response = (Response) build.invoke(builder);

        Method getCode = response.getClass().getMethod("getCode");
        Method getMsg = response.getClass().getMethod("getMsg");
        if (getCode == null) {
            Log.e("IMRPC", Trace.method("handleResponse")
                    .info("Method getCode: ", "null"));
        }

        if (getMsg == null) {
            Log.e("IMRPC", Trace.method("handleResponse")
                    .info("Method getMsg: ", "null"));
        }

        assert getCode != null;
        Integer code = (Integer) getCode.invoke(response);
        assert getMsg != null;
        String msg = (String) getMsg.invoke(response);


        long rtt = System.currentTimeMillis() - startTs;

        if (code == ErrorEnum.OK.getCode()) {
            Log.i("IMRPC", flow.method(getHummerFunction())
                    .msg("<- Succeed {" + describeHummerResponse(response)) + "}");

            handleHummerSuccess(response);
        } else {
            rtt = 0;

            Error error = Error.ofError(code, msg);

            Log.e("IMRPC", error, flow.method(getHummerFunction())
                    .msg("<- Failed {" + new StringChain()
                            .add("logId", getLogId())
                            .add("response", describeHummerResponse(response))
                            .toString() + "}"));

            handleHummerError(response, error);
        }

        StatisticsReporter.batchReportReturnCode(getHummerFunction(), rtt, code);
    }

    private Trace.Flow flow = new Trace.Flow();
    private long startTs;
    private static final String FunctionNamePrefix = "cim.proto.ProxyService";
}
