package com.hummer._internals.utility;

import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.support.annotation.NonNull;

import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;

import java.util.concurrent.Semaphore;

public final class DispatchQueue {

    public static final DispatchQueue main   = new DispatchQueue(new LooperHandlerProvider(Looper.getMainLooper()));
    public static final DispatchQueue direct = new DispatchQueue(new DispatchQueue.WorkerHandler("hmr_callBack"));
    private static final String TAG = "DispatchQueue";

    public interface HandlerProvider {
        Handler getHandler();
    }

    public static class LooperHandlerProvider implements HandlerProvider {

        public LooperHandlerProvider(@NonNull Looper looper) {
            this.handler = new Handler(looper);
        }

        @Override
        public Handler getHandler() {
            return handler;
        }

        private final Handler handler;
    }

    public static class WorkerHandler implements HandlerProvider {
        public WorkerHandler(String threadName) {
            this.threadName = threadName;
        }

        @Override
        public Handler getHandler() {
            HandlerThread thread = new HandlerThread(threadName);
            thread.start();
            return new Handler(thread.getLooper());
        }

        private final String threadName;
    }

    public void async(@NonNull final Runnable runnable) {
        if (handler == null) {
            Log.i(TAG, Trace.method("async")
                    .msg("runnable, run"));
            runnable.run();
            return;
        }

        handler.post(new Runnable() {
            @Override
            public void run() {
                runnable.run();
            }
        });
    }

    public void sync(@NonNull final Runnable runnable) {
        if (handler == null) {
            runnable.run();
            return;
        }

        boolean sameLooper = Looper.myLooper() == handler.getLooper();

        if (sameLooper) {
            runnable.run();
        } else {
            final Semaphore semaphore = new Semaphore(0);
            handler.post(new Runnable() {
                @Override
                public void run() {
                    runnable.run();
                    semaphore.release();
                }
            });
            semaphore.acquireUninterruptibly(1);
        }
    }

    public void asyncAfter(int delayMillis, @NonNull Runnable runnable) {
        if (handler == null) {
            try {
                Thread.sleep(delayMillis);
            } catch (InterruptedException e) {
                // 无需处理
            }

            runnable.run();
        } else {
            handler.postDelayed(runnable, delayMillis);
        }
    }

    public Handler getHandler() {
        return handler;
    }

    private final Handler handler;

    @SuppressWarnings("unused") // Meant not for calling
    public DispatchQueue() {
        handler = null;
    }

    public DispatchQueue(HandlerProvider provider) {
        this.handler = provider.getHandler();
    }
}
