package com.hummer._internals.report;

import android.os.Build;
import android.text.TextUtils;
import android.util.SparseIntArray;

import com.hummer.HMR;
import com.hummer._internals.channel.Channel;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.report.statis.InternetUtils;
import com.hummer._internals.report.statis.ReportMessage;
import com.hummer._internals.utility.HMRContext;

import java.util.HashMap;
import java.util.Map;

public final class StatisticsReporter {

    private static Map<Integer, String> codeNames = new HashMap<Integer, String>() {{
        put(Codes.ExceptionalDispatch, "ExceptionalDispatch");
        put(Codes.NotResetIsDraining, "NotResetIsDraining");
        put(Codes.ImpossibleScene, "ImpossibleScene");
    }};

    // Mark - 海度上报

    // 消息拉取异常上报
    public static void report(final int errCode, final Fields fields) {
        final String codeName = codeNames.get(errCode);
        if (codeName == null) {
            // 忽略错误（无法识别）的上报内容
            return;
        }

        int frequency = Frequencies.get(errCode, 0);
        boolean customizedFrequency = (frequency > 0);

        if (customizedFrequency) {
            boolean shouldReport = false;
            int count = RequestedCounts.get(errCode, 0);
            count += 1;

            if (count >= frequency) {
                shouldReport = true;
                count = 0;
            }

            RequestedCounts.put(errCode, count);

            if (!shouldReport) {
                // 仅仅是更新请求次数，不作上报
                return;
            }
        }

        Log.w("MessageService", Trace.method("reportException")
                .info("type", codeName)
                .info("fields", fields.toString()));

        HashMap<String, HiidoReporter.Field> fieldsMap = new HashMap<String, HiidoReporter.Field>() {{
            // 固有字段（必备）
            put("errCode", HiidoReporter.Field.from(errCode));
            put("errType", HiidoReporter.Field.from(codeName));
            put("appId", HiidoReporter.Field.from(HMRContext.getAppId()));
            put("sys", HiidoReporter.Field.from(2L)); // For Android
            put("operateSystem", HiidoReporter.Field.from(Build.VERSION.RELEASE));
            put("phoneModel", HiidoReporter.Field.from(getDeviceName()));
            put("sdkver", HiidoReporter.Field.from(HMR.getVersion()));

            // 可空字段
            put("logId", HiidoReporter.Field.from(fields.mLogId));
            put("localSeqId", HiidoReporter.Field.from(fields.localSeqId));
            put("maxSeqId", HiidoReporter.Field.from(fields.maxSeqId));
            put("remoteSeqId", HiidoReporter.Field.from(fields.remoteSeqId));
            put("errInfo", HiidoReporter.Field.from(fields.errInfo));
        }};

        Long me = HMR.getMe();
        if (me != null) {
            fieldsMap.put("uid", HiidoReporter.Field.from(me));
        }

        HiidoReporter.report("cim2", fieldsMap);
    }

    // 错误上报
    public static void report(final Fields fields) {
        HashMap<String, HiidoReporter.Field> fieldsMap = new HashMap<String, HiidoReporter.Field>() {{
            put("appId", HiidoReporter.Field.from(HMRContext.appId));
            put("sys", HiidoReporter.Field.from(2L));        // For Android
        }};

        Long me = HMR.getMe();
        if (me != null) {
            fieldsMap.put("uid", HiidoReporter.Field.from(me));
        }

        fieldsMap.put("localSeqId", HiidoReporter.Field.from(fields.localSeqId));
        fieldsMap.put("maxSeqId", HiidoReporter.Field.from(fields.maxSeqId));
        fieldsMap.put("remoteSeqId", HiidoReporter.Field.from(fields.remoteSeqId));
        fieldsMap.put("errCode", HiidoReporter.Field.from(fields.errCode));
        fieldsMap.put("errInfo", HiidoReporter.Field.from(fields.errInfo));

        if (fields.errCode == Codes.UndefinedError) {
            fieldsMap.put("errType", HiidoReporter.Field.from("UndefinedError"));
        } else if (fields.errCode >= Codes.PullMsgNull && fields.errCode <= Codes.ParseMsgErr) {
            String[] types = {"PullMsgNull", "LocalMaxRemote", "ContinuePullErr", "ParseMsgErr"};
            fieldsMap.put("errType", HiidoReporter.Field.from(types[fields.errCode - 1]));
        }

        HiidoReporter.report(act, fieldsMap);
    }

    private static final String act = "cim2";

    // 消息发送回复成功上报
    public static void report2SendMsg(final ReportMessage message) {
        try {
            if (TextUtils.isEmpty(message.getUuid())
                    || TextUtils.isEmpty(message.getOffline())
                    || TextUtils.isEmpty(message.getTimestamp())) {
                Log.i(TAG, Trace.method("report2SendMsg")
                        .msg("something is null")
                        .info("uuid", message.getUuid())
                        .info("timestamp", message.getTimestamp())
                        .info("offline", message.getOffline()));
                return;
            }

            HashMap<String, String> fieldsMap = new HashMap<>(16);
            fieldsMap.put("appId", HiidoReporter.Field.from(HMRContext.appId).toString());

            fieldsMap.put("sUid", HiidoReporter.Field.from(message.getFromUid()).toString());

            fieldsMap.put("sdkVersion", HiidoReporter.Field.from(HMR.getVersion()).toString());
            fieldsMap.put("sNet", HiidoReporter.Field.from(InternetUtils.getNetworkState()).toString());
            // fieldsMap.put("sRtt", HiidoReporter.Field.from(sendRtt)); // 发送消息时的时延（后期由ServiceSDK提供）
            fieldsMap.put("sRegion", HiidoReporter.Field.from(HMRContext.region.area).toString());
            fieldsMap.put("sModel", HiidoReporter.Field.from(getDeviceName()).toString());
            fieldsMap.put("rUid", HiidoReporter.Field.from(message.getToUid()).toString());
            fieldsMap.put("msgUuid", HiidoReporter.Field.from(message.getUuid()).toString());
            fieldsMap.put("offline", HiidoReporter.Field
                    .from("yes".equals(message.getOffline()) ? "1" : "0")
                    .toString());

            HiidoReporter.reportAct("hmrp2psendmsg", null, null, fieldsMap);
        } catch (Exception ex) {
            Log.e(TAG, "report error " + ex.toString());
        }
    }

    /**
     * 上报接收到的消息
     * @param message 需要上报的消息
     */
    public static void report2RecvMsg(final ReportMessage message) {
        try {
            if (TextUtils.isEmpty(message.getUuid())
                    || TextUtils.isEmpty(message.getOffline())
                    || TextUtils.isEmpty(message.getTimestamp())) {
                Log.i(TAG, Trace.method("report2RecvMsg")
                        .msg("something is null")
                        .info("uuid", message.getUuid())
                        .info("timestamp", message.getTimestamp())
                        .info("offline", message.getOffline()));
                return;
            }

            long timestamp = Long.parseLong(message.getTimestamp());
            long endTs = HMR.getService(Channel.class).getAlignmentServerTs();
            if (endTs == 0 || timestamp == 0) {
                Log.i(TAG, Trace.method("report2RecvMsg")
                        .msg("endTs is error")
                        .info("uuid", message.getUuid())
                        .info("timestamp", timestamp)
                        .info("endTs", endTs));
            }

            long rtt = endTs - timestamp;
            // 时延上报鹰眼: 对于rtt < 0, 内部有保护
            StatisticsReporter.batchReportReturnCode("P2PMessageDelay", rtt, 0);

            if (rtt <= 0) {
                Log.i(TAG, Trace.method("report2RecvMsg")
                        .info("report timestamp exception", message.getUuid())
                        .info("rtt", rtt));
                rtt = 0;
            }

            Log.d(TAG, Trace.method("reportMessageRTT")
                    .info("uuid", message.getUuid())
                    .info("start timestamp", message.getTimestamp())
                    .info("end timestamp", endTs)
                    .info("rtt", rtt));

            HashMap<String, String> fieldsMap = new HashMap<>();
            fieldsMap.put("appId", HiidoReporter.Field.from(HMRContext.appId).toString());

            fieldsMap.put("rUid", HiidoReporter.Field.from(message.getToUid()).toString());
            fieldsMap.put("sUid", HiidoReporter.Field.from(message.getFromUid()).toString());

            fieldsMap.put("sdkVersion", HiidoReporter.Field.from(HMR.getVersion()).toString());
            fieldsMap.put("rNet", HiidoReporter.Field.from(InternetUtils.getNetworkState()).toString());
            // fieldsMap.put("rRtt", HiidoReporter.Field.from(sendRtt)); // 接收消息时的时延（后期由ServiceSDK提供）
            fieldsMap.put("rRegion", HiidoReporter.Field.from(HMRContext.region.area).toString());
            fieldsMap.put("rModel", HiidoReporter.Field.from(getDeviceName()).toString());
            fieldsMap.put("msgUuid", HiidoReporter.Field.from(message.getUuid()).toString());
            fieldsMap.put("offline", HiidoReporter.Field
                    .from("yes".equals(message.getOffline()) ? "1" : "0")
                    .toString());
            fieldsMap.put("msgCost", HiidoReporter.Field.from(rtt).toString());

            HiidoReporter.reportAct("hmrp2precvmsg", null, null, fieldsMap);
        } catch (Exception ex) {
            Log.e(TAG, "report2RecvMsg error " + ex.toString());
        }
    }

    // 鹰眼上报
    public static void batchReportReturnCode(String funcName, long rtt, int code) {
        HiidoReporter.batchReportReturnCode(funcName, rtt, code);
    }

    private static String getDeviceName() {
        final String manufacturer = Build.MANUFACTURER;
        final String model = Build.MODEL;
        return model.startsWith(manufacturer) ? capitalizedPhrase(model) :
                capitalizedPhrase(manufacturer) + " " + model;
    }

    private static String capitalizedPhrase(String s) {
        if (s == null || s.length() == 0) {
            return s;
        } else {
            StringBuilder phrase = new StringBuilder();
            boolean next = true;
            for (char c : s.toCharArray()) {
                if (next && Character.isLetter(c) || Character.isWhitespace(c)) {
                    next = Character.isWhitespace(c = Character.toUpperCase(c));
                }

                phrase.append(c);
            }
            return phrase.toString();
        }
    }

    @SuppressWarnings("WeakerAccess")
    public final static class Codes {
        /* 1 到 4 为已经废弃的错误类型，不再上报。但是为了前向兼容，不再使用它们 */

        public static final Integer PullMsgNull = 1;
        public static final Integer LocalMaxRemote = 2;
        public static final Integer ContinuePullErr = 3;
        public static final Integer ParseMsgErr = 4;

        public static final Integer ExceptionalDispatch = 5;
        public static final Integer NotResetIsDraining = 6;
        public static final Integer ImpossibleScene = 7;

        public static final Integer UndefinedError = 999;
    }

    public final static class ReliableType {
        /*消息是否为可靠消息, 0为非可靠，1为可靠*/

        public static final Integer DoNotReport = -1;
        public static final Integer UnReliable = 0;
        public static final Integer Reliable = 1;
    }

    public static SparseIntArray Frequencies = new SparseIntArray();
    private static SparseIntArray RequestedCounts = new SparseIntArray();
    private final static String TAG = "StatisticsReporter";

    public static class Fields {
        public int errCode;
        public Long mLogId;
        public Long localSeqId;
        public Long remoteSeqId;
        public Long maxSeqId;
        public String errInfo;

        public Fields(int errCode) {
            this.errCode = errCode;
        }

        public Fields() {

        }

        public void setLocalSeqId(Long localSeqId) {
            this.localSeqId = localSeqId;
        }

        public void setRemoteSeqId(Long remoteSeqId) {
            this.remoteSeqId = remoteSeqId;
        }

        public void setMaxSeqId(Long maxSeqId) {
            this.maxSeqId = maxSeqId;
        }

        public void setErrInfo(String errInfo) {
            this.errInfo = errInfo;
        }
    }
}

