package com.hummer._internals.mq;

import android.content.SharedPreferences;

import com.hummer.Error;
import com.hummer.ErrorEnum;
import com.hummer.HMR;
import com.hummer._internals.channel.Channel;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.proto.Im;
import com.hummer._internals.proto.Push;
import com.hummer._internals.report.HiidoReporter;
import com.hummer._internals.report.StatisticsReporter;
import com.hummer._internals.utility.CompletionUtils;
import com.hummer._internals.utility.HMRContext;
import com.hummer._internals.utility.HMRContext.Region;
import com.hummer._internals.utility.Objects;
import com.hummer._internals.utility.PrefStorage;
import com.hummer._internals.utility.RichCompletionArg;
import com.hummer.model.completion.OnFailure;
import com.hummer.model.completion.OnSuccessArg;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

public final class Source implements MQService.Source {

    private final Mode mode;
    private Long seqId;
    private Long lastPullAt;
    private boolean isDraining;
    private boolean sourceChanged;

    public Source(Mode mode) {
        this.mode = mode;
    }

    public Mode getMode() {
        return mode;
    }

    @Override
    public void onTimerPulse(final MQService.MessagesDispatcher dispatcher) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (Source.this.isOverdue(TimeoutDuration)) {
                    if (isDraining) {
                        StatisticsReporter.report(StatisticsReporter.Codes.NotResetIsDraining,
                                new StatisticsReporter.Fields());
                    }

                    // 进一步预防 http://ylbug.yypm.com:8081/browse/ANDYN-10515 问题可能带来的影响
                    // 防止rpc过程异常导致没有正确重置isDraining标识
                    isDraining = false;
                }

                Source.this.drainMessagesIfNeeded(dispatcher);
            }
        });
    }

    @Override
    public void onManualPullingRequest(final MQService.MessagesDispatcher dispatcher) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                Source.this.drainMessagesIfNeeded(dispatcher);
            }
        });
    }

    @Override
    public void onNetworkReconnected(final MQService.MessagesDispatcher dispatcher) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                // 网络断连时间无法确定，如果不重置lastDrainedAt的话会影响后台的消息拉取时延统计结果
                lastPullAt = null;
                Source.this.drainMessagesIfNeeded(dispatcher);
            }
        });
    }

    @Override
    public void start(final MQService.MessagesDispatcher dispatcher) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                Source.this.performStarting(dispatcher);
            }
        });
    }

    @Override
    public void stop() {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                performStopping();
            }
        });
    }

    @Override
    public String toString() {
        return mode.sourceName();
    }

    /**
     * 消息连续性检查，在通知业务的地方，通过检查MQ队列消息的连续性保证消息输入输出的正确性
     *
     * @param msg 消息
     */
    public void checkIntegrity(final Im.Msg msg) {
        HiidoReporter.reportQueue.async(new Runnable() {
            @Override
            public void run() {
                if (!mode.isCheckIntegrity()) {
                    return;
                }

                if (mode.getCheckTermId() != null
                        && mode.getCheckIntegritySeqId() != null
                        && mode.getCheckTermId().compareTo(msg.getTerm()) == 0) {
                    // 无本地信息，或者termId不一致，无需检查连续性
                    if (mode.getCheckIntegritySeqId() + 1 == msg.getIntegrityCheckSeq()) {
                        // 连续性检查通过
                    } else {
                        // 判断region不为空，避免region为空导致空指针
                        if (HMRContext.region != null) {
                            mode.checkIntegrityRequest(msg);
                        } else {
                            Log.w(mode.sourceName(), Trace.method("checkIntegrity")
                                    .msg("did not call checkIntegrityRequest，becz region is null"));
                        }
                    }
                }

                mode.updateParam(msg);
            }
        });
    }

    private void performStarting(final MQService.MessagesDispatcher dispatcher) {
        Log.i(mode.sourceName(), Trace.method("performStarting"));

        isDraining = false;
        lastPullAt = null;

        mode.start(new Mode.NotifyHandler() {
            @Override
            public void onNotify(final long prevSeqId, final Im.Msg msg, final long seqId) {
                HMRContext.work.async(new Runnable() {
                    @Override
                    public void run() {
                        if (msg == null) {
                            sourceChanged = true;
                            Source.this.drainMessagesIfNeeded(dispatcher);
                            return;
                        }

                        Log.i(mode.sourceName(), Trace.method("onNotify")
                                .info("prevSeqId", prevSeqId)
                                .info("seqId", seqId)
                                .info("draining", isDraining)
                                .info("localSeqId", Source.this.seqId));
                        // 如果带上消息的话有两种情况
                        // 1. 本地的 seqId 还没收到的情况下不知道从什么地方啦，也不知道该不该接收
                        // 2. 正在拉消息，可能拉倒的消息了包括该消息
                        // 3. 如果 prevSeqId 为 0 的话，则直接忽略该消息通知
                        // 4. 当通知的 prevSeqId 小于本地的 seqId, 则直接忽略
                        // 所以这两种情况下忽略这条通知
                        if (Source.this.seqId == null
                                || Source.this.isDraining
                                || prevSeqId == 0
                                || prevSeqId < Source.this.seqId) {
                            return;
                        }

                        // 如果本地的 seqId 与通知的之前的一条 seqId 相同
                        // 并且没有上述的情况， 则直接 dispatch 这条消息出去
                        // 同时更改本地的 seqId
                        if (prevSeqId == Source.this.seqId) {
                            Log.i(mode.sourceName(), Trace.method("onNotify: Accept msg"));
                            List<Im.Msg> msgs = new ArrayList<>();
                            msgs.add(msg);
                            dispatcher.dispatch(msgs, Source.this);
                            Source.this.lastPullAt = System.currentTimeMillis();
                            Source.this.setSeqId(seqId, true);
                        } else {
                            Log.i(mode.sourceName(), Trace.method("onNotify: Pull missing msgs"));
                            sourceChanged = true;
                            Source.this.drainMessagesIfNeeded(dispatcher);
                        }
                    }
                });
            }
        });

        mode.loadSeqId(getStrategy(mode), new RichCompletionArg<Long>()
                .onSuccess(new OnSuccessArg<Long>() {
                    @Override
                    public void onSuccess(final Long loadedSeqId) {
                        if (loadedSeqId == null) {
                            Log.e("Source", Trace.method("performStarting").msg("loadedSeqId is <null>"));
                        }

                        HMRContext.work.async(new Runnable() {
                            @Override
                            public void run() {
                                // 当本地没有seqId（例如重装app）时，应强制写入本地，避免下次启动再去拉取
                                Source.this.setSeqId(loadedSeqId, Source.this.seqId == null);
                                Source.this.drainMessagesIfNeeded(dispatcher);
                            }
                        });
                    }
                })
                .onFailure(new OnFailure() {
                    @Override
                    public void onFailure(Error err) {
                        HMRContext.work.async(new Runnable() {
                            @Override
                            public void run() {
                                // 如果失败，则无论如何都从0开始，避免因为该流程而导致始终无法拉取消息
                                // 缺点是，有可能增加会拉取到冗余、重复消息的概率
                                Source.this.setSeqId(0L, false);
                                Source.this.drainMessagesIfNeeded(dispatcher);
                            }
                        });
                    }
                }));
    }

    private MQService.FetchStrategy getStrategy(Mode mode) {
        if (mode.getStrategy() != null) {
            return mode.getStrategy();
        }

        if (HMR.getService(MQService.class).getFetchStrategy() != null) {
            return HMR.getService(MQService.class).getFetchStrategy();
        }

        return MQService.FetchStrategy.Continuously;

    }

    private void performStopping() {
        Log.i(mode.sourceName(), Trace.method("performStopping"));

        mode.stop();

        isDraining = false;
        lastPullAt = null;
        seqId = null;
    }

    private static final long TimeoutDuration = 60 * 1000;
    private static final long DEFAULT_PULLING_PERIOD = 15 * 60 * 1000; // 测试暂定1分钟

    private boolean isOverdue(long period) {
        long tolerance = period / 30;
        long now = System.currentTimeMillis();
        return (lastPullAt == null || (now - lastPullAt) >= (period - tolerance));
    }

    private void drainMessagesIfNeeded(final MQService.MessagesDispatcher dispatcher) {
        @SuppressWarnings("PointlessBooleanExpression")
        boolean shouldIgnore =
                // 当使用SkipFetched模式启动时，seqId可能存在短暂窗口处于不确定的状态。在这个窗口期间，如果因为超期
                // 检测而触发了drainMessagesIfNeeded调用，应直接忽略掉。
                seqId == null

                        // 另一个消息拉取尝试正在进行，忽略掉冗余拉取请求
                        || isDraining

                        // 既没有更多消息，也没有过期，忽略拉取请求。主要是因为时间脉冲触发的拉取尝试导致的。
                        || !(sourceChanged || isOverdue(mode.getPullingPeriod()));

        if (shouldIgnore) {
            return;
        }

        Log.i(mode.sourceName(), Trace.method("drainMessagesIfNeeded")
                .info("seqId", seqId)
                .info("changed", sourceChanged)
                .info("draining", isDraining)
                .info("overdue", isOverdue(mode.getPullingPeriod())));

        sourceChanged = false; // Reset dirty flag before draining
        isDraining = true;
        drainMessages(dispatcher, seqId, lastPullAt == null, new Runnable() {
            @Override
            public void run() {
                isDraining = false;
            }
        });
    }

    private void drainMessages(final MQService.MessagesDispatcher dispatcher,
                               final long fromSeqId,
                               final boolean isFirstDrain,
                               final Runnable completion) {
        Log.i(mode.sourceName(), Trace.method("drainMessages")
                .info("fromSeqId", seqId)
                .info("isFirstDrain", isFirstDrain));

        HMR.getService(Channel.class).run(mode.createPullingRequest(fromSeqId, isFirstDrain,
                new RichCompletionArg<RPCPullingResponse>()
                        .onSuccess(new OnSuccessArg<RPCPullingResponse>() {
                            @Override
                            public void onSuccess(RPCPullingResponse resp) {
                                // seqId 为null的情况是会存在的。
                                // 发起拉消息，就等着service的回包，在回包之前，用户发起了退房间
                                // 则会调用removeSource，那么执行到performStopping时，会将seqId设置为null
                                // 此时service回包，这种情况seqId就是null的。丢弃这次回包
                                if (Source.this.seqId == null) {
                                    Log.i(mode.sourceName(), Trace.method("drainMessages")
                                            .msg("seqId is null, return"));
                                    return;
                                }

                                // 由于各种原因，对同一个起始seqid有可能触发了多次请求，这样就导致会有重复消息的情况
                                // MQ队列里的消息，后台有保证是单调递增，在此基础上，就使用seqid进行去重
                                // 当发现队列消息里面有比本地seqid还小的，则判定为重复拉取
                                List<Im.Msg> validMsgs = new ArrayList<>();
                                for (Im.Msg msg : resp.messages) {
                                    if (msg.getSeqId() <= Source.this.seqId) {
                                        Log.i(mode.sourceName(), Trace.method("handleDrainingSuccess")
                                                .msg("BUGGY!! abandon message")
                                                .info("lastPullAt -> ", lastPullAt)
                                                .info("localSeqid", Source.this.seqId)
                                                .info("msgSeqid", msg.getSeqId()));
                                    } else {
                                        validMsgs.add(msg);
                                    }
                                }

                                if (resp.messages.size() > 0 && validMsgs.size() <= 0) {
                                    // 无有效信息，则认为这次请求结果为重复请求
                                    // 作整个请求的抛弃处理
                                    Log.i(mode.sourceName(), Trace.method("handleDrainingSuccess")
                                            .msg("BUGGY!! abandon messages")
                                            .info("lastPullAt -> ", lastPullAt)
                                            .info("localSeqid", Source.this.seqId)
                                            .info("maxSeqid", resp.maxSeqId)
                                            .info("msgSize", resp.messages.size()));
                                    completion.run();
                                    return;
                                }

                                if (validMsgs.size() > 0) {
                                    try { // 加上try/catch, 避免外部消息分派处理崩溃导致drain流程被异常中断
                                        dispatcher.dispatch(resp.messages, Source.this);
                                    } catch (final Throwable t) {
                                        Log.e(mode.sourceName(), Trace.method("handleDrainingSuccess")
                                                .msg("Exception while dispatching messages")
                                                .info("exception", t.getLocalizedMessage()));

                                        // 外部异常不应该影响内部逻辑的继续执行，因此，不应该return！

                                        StatisticsReporter.report(StatisticsReporter.Codes.ExceptionalDispatch,
                                                new StatisticsReporter.Fields() {{
                                                    StringWriter sw = new StringWriter();
                                                    t.printStackTrace(new PrintWriter(sw));

                                                    errInfo = sw.toString();
                                                }});
                                    }
                                }

                                // 仅在拉取成功时才可以更新lastPullAt数据，这里的设置不会影响isInitialDrain参数，因为它是在发起drain请求
                                // 之前就已经决议的
                                lastPullAt = System.currentTimeMillis();
                                Log.i(mode.sourceName(), Trace.method("handleDrainingSuccess")
                                        .msg("lastPullAt -> " + lastPullAt));

                                if (resp.maxSeqId != null) {
                                    // 每次有效的pull操作，都应该尽可能更新seqId
                                    // 调整 seqid 和 isdraining 更新顺序，避免推拉结合机制下，会收到重复消息的问题
                                    // 更新本地seqid 的操作，应该要在isdraining标志更新之前
                                    // 否则当消息推送的时候，本地seqid还未更新造成误判而进行分发消息
                                    Source.this.setSeqId(resp.maxSeqId, true);

                                    if (resp.hasMore) {
                                        // 场景：消费了消息，且还有更多后续消息
                                        // 应对：继续拉取，根据maxSeqId触发下一次拉取
                                        Source.this.drainMessages(dispatcher, resp.maxSeqId, isFirstDrain, completion);
                                    } else {
                                        // 场景：已消费完所有消息 - EOF
                                        // 应对：停止拉取，下次拉取应从maxSeqId开始
                                        HMRContext.work.async(completion);
                                    }

                                } else { // maxSeqId == null
                                    if (!resp.hasMore) {
                                        // 场景：进行了一次无效拉取，没有消费任何消息
                                        // 应对：停止拉取，下次拉取还是从fromSeqId开始
                                        HMRContext.work.async(completion);
                                    } else {
                                        // 该场景理论上不可能出现，但确实偶发会遇到，为了避免客户端崩溃，这里只是进行了日志记录，并尝试中断
                                        // 拉取流程。并'寄望于'下次拉取可以正常恢复
                                        Log.e(mode.sourceName(), Trace.method("handleDrainingSuccess")
                                                .msg("BUGGY!! 未知场景，hasMore为True, 但maxSeqId为null，无法确定下次拉取从哪开始"));

                                        HMRContext.work.async(completion);

                                        StatisticsReporter.report(StatisticsReporter.Codes.ImpossibleScene,
                                                new StatisticsReporter.Fields() {
                                                    {
                                                        errInfo = String.format(Locale.US,
                                                                "hasMore为True, 但maxSeqId为null，"
                                                                        + "无法确定下次拉取从哪开始。"
                                                                        + "fromSeqId: %d", fromSeqId);
                                                    }
                                                });
                                    }
                                }
                            }
                        })
                        .onFailure(new OnFailure() {
                            @Override
                            public void onFailure(Error err) {
                                HMRContext.work.async(completion);
                            }
                        })
        ));
    }

    private void setSeqId(long seqId, boolean shouldPersistent) {
        if (this.seqId != null && this.seqId == seqId) {
            // Ignore nonsense assignment
            return;
        }

        Log.i(mode.sourceName(), Trace.method("setSeqId")
                .msg("%d -> %d", this.seqId, seqId));

        if (shouldPersistent) {
            this.mode.storeSeqId(seqId);
        }

        this.seqId = seqId;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        Source source = (Source) o;
        return mode.equals(source.mode);
    }

    @Override
    public int hashCode() {
        return mode.hashCode();
    }

    public interface Mode {
        interface NotifyHandler {
            void onNotify(long previous, Im.Msg msg, long seqId);
        }

        String sourceName();

        String topicName();

        MQService.FetchStrategy getStrategy();

        long getPullingPeriod();

        void loadSeqId(MQService.FetchStrategy strategy, RichCompletionArg<Long> completion);

        void storeSeqId(long seqId);

        Channel.RPC createPullingRequest(long seqId,
                                         boolean isFirstDrain,
                                         RichCompletionArg<RPCPullingResponse> completion);

        void start(NotifyHandler notifyHandler);

        void stop();

        String getArea();

        void checkIntegrityRequest(Im.Msg msg);

        Long getCheckTermId();

        Integer getCheckIntegritySeqId();

        boolean isCheckIntegrity();

        void updateParam(Im.Msg msg);

        void setCheckIntegrity(boolean checkIntegrity);
    }

    public static class Private implements Mode {

        private static final String StorageKeySeqIDPrefix = "local_sequence_id";
        private final int queueId;
        private final String topic;
        private Channel.NotificationHandler changeHandler;
        private MQService.FetchStrategy strategy;
        private long pullingPeriod;

        private Long checkTermId; // 消息连续性检查比对
        private Integer checkIntegritySeqId; // 消息连续性检查比对
        private long checkLastSeqId;
        private boolean isCheckIntegrity = false;

        public Private(String topic) {
            this(topic, null);
        }

        public Private(String topic, MQService.FetchStrategy strategy) {
            this(0, topic, strategy, DEFAULT_PULLING_PERIOD);
        }

        public Private(String topic, MQService.FetchStrategy strategy, long pullingPeriod) {
            this(0, topic, strategy, pullingPeriod);
        }

        public Private(int queueId,
                       String topic,
                       MQService.FetchStrategy strategy,
                       long pullingPeriod) {
            this.queueId = queueId;
            this.topic = (topic == null) ? "" : topic;
            this.strategy = strategy;
            this.pullingPeriod = pullingPeriod < 1L ? DEFAULT_PULLING_PERIOD : pullingPeriod;
        }

        @Override
        public Long getCheckTermId() {
            return checkTermId;
        }

        @Override
        public Integer getCheckIntegritySeqId() {
            return checkIntegritySeqId;
        }

        @Override
        public boolean isCheckIntegrity() {
            return isCheckIntegrity;
        }

        @Override
        public void updateParam(Im.Msg msg) {
            checkTermId = msg.getTerm();
            checkIntegritySeqId = msg.getIntegrityCheckSeq();
            checkLastSeqId = msg.getSeqId();
        }

        @Override
        public void setCheckIntegrity(boolean checkIntegrity) {
            isCheckIntegrity = checkIntegrity;
        }

        @Override
        public void checkIntegrityRequest(Im.Msg msg) {
            HMR.getService(Channel.class).run(new RPCCheckMsgIntegrity(topic,
                    checkLastSeqId,
                    msg.getSeqId(),
                    0,
                    null));
        }

        @Override
        public MQService.FetchStrategy getStrategy() {
            return this.strategy;
        }

        @Override
        public long getPullingPeriod() {
            return this.pullingPeriod;
        }

        public int getQueueId() {
            return queueId;
        }

        public String getTopic() {
            return topic;
        }

        @Override
        public String sourceName() {
            return "PrivateSource(" + queueId + ", " + topic + ")";
        }

        @Override
        public String topicName() {
            return topic;
        }

        @Override
        public void loadSeqId(MQService.FetchStrategy strategy, final RichCompletionArg<Long> completion) {
            Long seqId = null;

            if (strategy == MQService.FetchStrategy.Continuously) {
                seqId = PrefStorage.storage().execute(new PrefStorage.Query<Long>() {
                    @Override
                    public Long run(SharedPreferences pref) {
                        if (pref == null) {
                            return null;
                        }

                        long seq = pref.getLong(Private.this.prefKey(), -1);
                        if (seq == -1) {
                            // null 才是正确的"不确定、不存在、没有"语义，-1是错误的语义表达方式
                            return null;
                        } else {
                            return seq;
                        }
                    }
                });
            } else if (strategy == MQService.FetchStrategy.ReloadHistories) {
                seqId = 0L;
            }

            Log.i(sourceName(), Trace.method("loadSeqId").info("seqId", seqId));

            if (seqId == null) {
                if (strategy == MQService.FetchStrategy.IgnoreBefore) {
                    fetchMaxSeqId(3, queueId, topic, completion);
                } else {
                    // 如果无法确定seqId，则应尝试从服务器进行查询
                    // 1. 本地重装App
                    // 2. 强行使用SkipFetched模式
                    fetchSeqId(3, queueId, topic, completion);
                }
            } else {
                CompletionUtils.dispatchSuccess(completion, seqId);
            }
        }

        @Override
        public void storeSeqId(final long seqId) {
            PrefStorage.storage().execute(new PrefStorage.Edit() {
                @Override
                public void run(SharedPreferences.Editor editor) {
                    editor.putLong(Private.this.prefKey(), seqId);
                }
            });

            // 客户端重装时，可以进行用户上下文恢复，因为该请求为非关键请求，所以对completion不关心
            HMR.getService(Channel.class).run(new RPCReportPrivateSeqId(queueId, topic, seqId, null));
        }

        @Override
        public Channel.RPC createPullingRequest(long seqId,
                                                boolean isFirstDrain,
                                                RichCompletionArg<RPCPullingResponse> completion) {
            return new RPCPullPrivateMessages(queueId, topic, isFirstDrain, seqId, 200, null, completion);
        }

        @Override
        public void start(final NotifyHandler notifyHandler11) {
            Log.i(sourceName(), Trace.method("start"));

            // 服务器在个人队列新增了消息时会下发单播通知，该通知是被动的，必须进行通知注册才能进行响应
            changeHandler = new Channel.NotificationHandler() {
                @Override
                public void onNotify(String serviceName, String functionName, final byte[] data) {
                    String wantedUri = "service_api_gateway/cim.proto.PushService.IMPushMsg";
                    String actualUri = serviceName + '/' + functionName;
                    if (!wantedUri.equals(actualUri)) {
                        return;
                    }

                    HMRContext.work.async(new Runnable() {
                        @Override
                        public void run() {
                            handleNotify(data);
                        }
                    });
                }

                @Override
                public String toString() {
                    return sourceName();
                }

                private void handleNotify(byte[] data) {
                    try {
                        final Push.IMPushMsgRequest change = Push.IMPushMsgRequest
                                .newBuilder()
                                .mergeFrom(data)
                                .build();

                        if (change.getEnvName() != null && !change.getEnvName().isEmpty()) {
                            if (HMRContext.region == null) {
                                Log.w(sourceName(), Trace.method("onNotify")
                                        .msg("localEnv is null"));
                                return;
                            }
                            Region region = HMRContext.region;
                            String localEnv = String.format(Locale.US, "%s_%s_%s",
                                    region.type,
                                    region.name,
                                    region.area);

                            String remoteEnv = String.format(Locale.US, "%s_%s_%s",
                                    change.getEnvType(),
                                    change.getEnvName(),
                                    Region.getArea(change.getRegion()));

                            if (!Objects.equals(remoteEnv, localEnv)) {
                                // 收到不属于当前接入区域的消息，直接忽略

                                Log.i(sourceName(), Trace.method("onNotify")
                                        .msg("Ignored becz of different env")
                                        .info("localEnv", localEnv)
                                        .info("notifEnv", remoteEnv));

                                return;
                            }
                        }

                        if (Objects.equals(topic, change.getTopic())
                                && Objects.equals(queueId, change.getQueueId())) {
                            Log.i(sourceName(), Trace.method("onPrivateSourceChanged")
                                    .info("topic", change.getTopic())
                                    .info("seqId", change.getSeqId())
                                    .info("queueId", change.getQueueId()));

                            // Notify when private source changed
                            notifyHandler11.onNotify(0, null, change.getSeqId());
                        }
                    } catch (Throwable t) {
                        Log.e(sourceName(), Trace.method("handleNotify")
                                .msg("Failed parsing IMPushMsgRequest")
                                .info("Exception", t));
                    }
                }
            };
            HMR.getService(Channel.class).addNotificationHandler(changeHandler);
        }

        @Override
        public void stop() {
            Log.i(sourceName(), Trace.method("stop"));
            HMR.getService(Channel.class).removeNotificationHandler(changeHandler);
        }

        @Override
        public String getArea() {
            return null;
        }

        private void fetchSeqId(final int remainTries,
                                final int queueId,
                                final String topic,
                                final RichCompletionArg<Long> completion) {
            Log.i(sourceName(), Trace.method("fetchSeqId"));
            if (remainTries <= 0) {
                // 超过重试次数时，将seqId设置为0，这是为了保证在弱网情况下fetchReportedSequenceId无限重试，进而
                // 使得始终无法拉取消息的问题。提高了消息拉取逻辑的整体稳定性

                CompletionUtils.dispatchFailure(completion,
                        new Error(ErrorEnum.OPERATION_TIMEOUT, "Failed fetching seqId: " + topic));

                return;
            }

            // See <RichCompletion Token> Pattern
            HMR.getService(Channel.class).run(new RPCFetchPrivateSeqId(queueId, topic,
                    new RichCompletionArg<Long>()
                            .onSuccess(new OnSuccessArg<Long>() {
                                @Override
                                public void onSuccess(Long seqId) {
                                    CompletionUtils.dispatchSuccess(completion, seqId);
                                }
                            })
                            .onFailure(new OnFailure() {
                                @Override
                                public void onFailure(Error error) {
                                    if (error != null && error.code != ErrorEnum.OPERATION_TIMEOUT.getCode()) {
                                        CompletionUtils.dispatchFailure(completion, error);
                                    } else {
                                        // 网络超时属于可恢复错误，可以再尝试几次
                                        HMRContext.work.asyncAfter(3 * 1000, new Runnable() {
                                            @Override
                                            public void run() {
                                                Private.this.fetchSeqId(queueId, remainTries - 1, topic, completion);
                                            }
                                        });
                                    }
                                }
                            })
            ));
        }

        private void fetchMaxSeqId(final int remainTries,
                                   final int queueId,
                                   final String topic,
                                   final RichCompletionArg<Long> completion) {
            Log.i(sourceName(), Trace.method("fetchMaxSeqId"));
            if (remainTries <= 0) {
                CompletionUtils.dispatchFailure(completion,
                        new Error(ErrorEnum.OPERATION_TIMEOUT, "Failed fetching seqId: " + topic));

                return;
            }

            // See <RichCompletion Token> Pattern
            HMR.getService(Channel.class).run(new RPCFetchPrivateMaxSeqId(queueId, topic,
                    new RichCompletionArg<Long>()
                            .onSuccess(new OnSuccessArg<Long>() {
                                @Override
                                public void onSuccess(Long seqId) {
                                    CompletionUtils.dispatchSuccess(completion, seqId);
                                }
                            })
                            .onFailure(new OnFailure() {
                                @Override
                                public void onFailure(Error error) {
                                    if (error != null && error.code != ErrorEnum.OPERATION_TIMEOUT.getCode()) {
                                        CompletionUtils.dispatchFailure(completion, error);
                                    } else {
                                        // 网络超时属于可恢复错误，可以再尝试几次
                                        HMRContext.work.asyncAfter(3 * 1000, new Runnable() {
                                            @Override
                                            public void run() {
                                                Private.this.fetchMaxSeqId(remainTries - 1,
                                                        queueId, topic, completion);
                                            }
                                        });
                                    }
                                }
                            })
            ));
        }

        private String prefKey() {
            String prefKey = StorageKeySeqIDPrefix;
            if (queueId > 0) {
                prefKey = prefKey + "_" + queueId;
            }

            if (!topic.isEmpty()) {
                prefKey = prefKey + "_" + topic;
            }

            return prefKey;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            Private p = (Private) o;

            if (queueId != p.queueId) {
                return false;
            }
            return topic.equals(p.topic);
        }

        @Override
        public int hashCode() {
            int result = queueId;
            result = 31 * result + (topic != null ? topic.hashCode() : 0);
            return result;
        }
    }

    public static class Shared implements Mode {

        private static final String PrefKeySeqID = "_group_sys_seqid";
        private final int queueId;
        private final long groupId;
        private final String topic;
        private final String area;
        private Channel.NotificationHandler sharedHandler;
        private MQService.FetchStrategy strategy;
        private long pullingPeriod;

        private Long checkTermId; // 消息连续性检查比对
        private Integer checkIntegritySeqId; // 消息连续性检查比对
        private long checkLastSeqId;
        private boolean isCheckIntegrity = false;

        public Shared(long groupId, String topic, String area) {
            this(groupId, topic, area, null);
        }

        public Shared(long groupId, String topic, String area, MQService.FetchStrategy strategy) {
            this(0, groupId, topic, area, strategy, DEFAULT_PULLING_PERIOD);
        }

        public Shared(int queueId,
                      long groupId,
                      String topic,
                      String area,
                      MQService.FetchStrategy strategy,
                      long pullingPeriod) {
            this.groupId = groupId;
            this.topic = (topic == null) ? "" : topic;
            this.area = area;
            this.strategy = strategy;
            this.pullingPeriod = pullingPeriod < 1L ? DEFAULT_PULLING_PERIOD : pullingPeriod;
            this.queueId = queueId;
        }

        @Override
        public Long getCheckTermId() {
            return checkTermId;
        }

        @Override
        public Integer getCheckIntegritySeqId() {
            return checkIntegritySeqId;
        }

        @Override
        public boolean isCheckIntegrity() {
            return isCheckIntegrity;
        }

        @Override
        public void updateParam(Im.Msg msg) {
            checkTermId = msg.getTerm();
            checkIntegritySeqId = msg.getIntegrityCheckSeq();
            checkLastSeqId = msg.getSeqId();
        }

        @Override
        public void setCheckIntegrity(boolean checkIntegrity) {
            isCheckIntegrity = checkIntegrity;
        }

        @Override
        public void checkIntegrityRequest(Im.Msg msg) {
            HMR.getService(Channel.class).run(new RPCCheckGrpSysMsgIntegrity(groupId,
                    topic,
                    checkLastSeqId,
                    msg.getSeqId(),
                    0,
                    null));
        }

        @Override
        public MQService.FetchStrategy getStrategy() {
            return this.strategy;
        }

        public int getQueueId() {
            return queueId;
        }

        public long getGroupId() {
            return groupId;
        }

        public String getTopic() {
            return topic;
        }

        @Override
        public String getArea() {
            return area;
        }

        @Override
        public long getPullingPeriod() {
            return this.pullingPeriod;
        }

        @Override
        public String sourceName() {
            return "SharedSource(" + queueId + ", " + groupId + ", " + topic + ")";
        }

        @Override
        public String topicName() {
            return topic;
        }

        @Override
        public void loadSeqId(MQService.FetchStrategy strategy, RichCompletionArg<Long> completion) {
            Long seqId = null;

            if (strategy == MQService.FetchStrategy.Continuously) {
                seqId = PrefStorage.storage().execute(new PrefStorage.Query<Long>() {
                    @Override
                    public Long run(SharedPreferences pref) {
                        if (pref == null) {
                            return null;
                        }

                        long seq = pref.getLong(Shared.this.prefKey(), -1);
                        if (seq == -1) {
                            // null 才是正确的"不确定、不存在、没有"语义，-1是错误的语义表达方式
                            return null;
                        } else {
                            return seq;
                        }
                    }
                });
            } else if (strategy == MQService.FetchStrategy.ReloadHistories) {
                seqId = 0L;
            }

            Log.i(sourceName(), Trace.method("loadSeqId").info("seqId", seqId));

            if (seqId == null) {
                if (strategy == MQService.FetchStrategy.IgnoreBefore) {
                    fetchMaxSeqId(3, queueId, topic, completion);
                } else {
                    // 如果无法确定seqId，则应尝试从服务器进行查询
                    // 1. 本地重装App
                    // 2. 强行使用SkipFetched模式
                    fetchSeqId(3, queueId, topic, completion);
                }
            } else {
                CompletionUtils.dispatchSuccess(completion, seqId);
            }
        }

        private void fetchSeqId(final int remainTries,
                                final int queueId,
                                final String topic,
                                final RichCompletionArg<Long> completion) {
            Log.i(sourceName(), Trace.method("Shared fetchSeqId"));
            if (remainTries <= 0) {
                // 超过重试次数时，将seqId设置为0，这是为了保证在弱网情况下fetchReportedSequenceId无限重试，进而
                // 使得始终无法拉取消息的问题。提高了消息拉取逻辑的整体稳定性

                CompletionUtils.dispatchFailure(completion,
                        new Error(ErrorEnum.OPERATION_TIMEOUT, "Shared Failed fetching seqId: " + topic));

                return;
            }

            HMR.getService(Channel.class).run(new RPCFetchSharedSeqId(queueId, topic, this.area,
                    groupId,
                    new RichCompletionArg<Long>()
                            .onSuccess(new OnSuccessArg<Long>() {
                                @Override
                                public void onSuccess(Long seqId) {
                                    CompletionUtils.dispatchSuccess(completion, seqId);
                                }
                            })
                            .onFailure(new OnFailure() {
                                @Override
                                public void onFailure(Error error) {
                                    if (error != null && error.code != ErrorEnum.OPERATION_TIMEOUT.getCode()) {
                                        CompletionUtils.dispatchFailure(completion, error);
                                    } else {
                                        // 网络超时属于可恢复错误，可以再尝试几次
                                        HMRContext.work.asyncAfter(3 * 1000,
                                                new Runnable() {
                                                    @Override
                                                    public void run() {
                                                        Shared.this.fetchSeqId(remainTries - 1,
                                                                queueId, topic, completion);
                                                    }
                                                });
                                    }
                                }
                            })));
        }

        private void fetchMaxSeqId(final int remainTries,
                                   final int queueId,
                                   final String topic,
                                   final RichCompletionArg<Long> completion) {
            Log.i(sourceName(), Trace.method("Shared fetchMaxSeqId"));
            if (remainTries <= 0) {
                // 超过重试次数时，将seqId设置为0，这是为了保证在弱网情况下fetchReportedSequenceId无限重试，进而
                // 使得始终无法拉取消息的问题。提高了消息拉取逻辑的整体稳定性

                CompletionUtils.dispatchFailure(completion,
                        new Error(ErrorEnum.OPERATION_TIMEOUT, "Shared Failed fetching maxSeqId: " + topic));

                return;
            }

            HMR.getService(Channel.class).run(new RPCFetchSharedMaxSeqId(queueId, topic,
                    groupId,
                    area,
                    new RichCompletionArg<Long>()
                            .onSuccess(new OnSuccessArg<Long>() {
                                @Override
                                public void onSuccess(Long seqId) {
                                    CompletionUtils.dispatchSuccess(completion, seqId);
                                }
                            })
                            .onFailure(new OnFailure() {
                                @Override
                                public void onFailure(Error error) {
                                    if (error != null && error.code != ErrorEnum.OPERATION_TIMEOUT.getCode()) {
                                        CompletionUtils.dispatchFailure(completion, error);
                                    } else {
                                        // 网络超时属于可恢复错误，可以再尝试几次
                                        HMRContext.work.asyncAfter(3 * 1000,
                                                new Runnable() {
                                                    @Override
                                                    public void run() {
                                                        Shared.this.fetchMaxSeqId(remainTries - 1,
                                                                queueId, topic, completion);

                                                    }
                                                });
                                    }
                                }
                            })));
        }

        @Override
        public void storeSeqId(final long seqId) {
            PrefStorage.storage().execute(new PrefStorage.Edit() {
                @Override
                public void run(SharedPreferences.Editor editor) {
                    editor.putLong(Shared.this.prefKey(), seqId);
                }
            });

            // 客户端重装时，可以进行用户上下文恢复，因为该请求为非关键请求，所以对completion不关心
            HMR.getService(Channel.class)
                    .run(new RPCReportSharedSeqId(queueId, topic, this.area, groupId, seqId, null));
        }

        @Override
        public Channel.RPC createPullingRequest(long seqId,
                                                boolean isFirstDrain,
                                                RichCompletionArg<RPCPullingResponse> completion) {
            return new RPCPullSharedMessages(queueId, groupId, topic, area, seqId, 200, isFirstDrain, completion);
        }

        @Override
        public void start(final NotifyHandler notifyHandler) {
            Log.i(sourceName(), Trace.method("start"));

            sharedHandler = new Channel.NotificationHandler() {
                @Override
                public void onNotify(String serviceName, String functionName, final byte[] data) {
                    String wantedUri = "service_api_gateway/cim.proto.PushService.IMPushGroupSysMsg";
                    String actualUri = serviceName + '/' + functionName;
                    if (!wantedUri.equals(actualUri)) {
                        return;
                    }
                    HMRContext.work.async(new Runnable() {
                        @Override
                        public void run() {
                            handleNotify(data);
                        }
                    });
                }

                @Override
                public String toString() {
                    return sourceName();
                }

                private void handleNotify(byte[] data) {
                    final Push.IMPushGroupSysMsgRequest request;
                    try {
                        request = Push.IMPushGroupSysMsgRequest
                                .newBuilder()
                                .mergeFrom(data)
                                .build();
                    } catch (Throwable t) {
                        Log.e(sourceName(), Trace.method("handleNotify")
                                .msg("Failed parsing IMPushGroupSysMsgRequest")
                                .info("Exception", t));
                        return;
                    }

                    if (request.getEnvName() != null && !request.getEnvName().isEmpty()) {
                        if (HMRContext.region == null) {
                            Log.w(sourceName(), Trace.method("onNotify")
                                    .msg("localEnv is null"));
                            return;
                        }
                        Region region = HMRContext.region;
                        String localEnv = String.format(Locale.US, "%s_%s_%s",
                                region.type,
                                region.name,
                                Shared.this.area);

                        String remoteEnv = String.format(Locale.US, "%s_%s_%s",
                                request.getEnvType(),
                                request.getEnvName(),
                                Region.getArea(request.getRegion()));

                        if (!Objects.equals(remoteEnv, localEnv)) {
                            // 收到不属于当前接入区域的消息，直接忽略
                            Log.i(sourceName(), Trace.method("onNotify")
                                    .msg("Ignored becz of different env")
                                    .info("localEnv", localEnv)
                                    .info("notifyEnv", remoteEnv));
                            return;
                        }
                    }

                    if (Objects.equals(topic, request.getTopic())
                            && Objects.equals(groupId, request.getGroupId())
                            && Objects.equals(queueId, request.getQueueId())) {

                        Log.i(sourceName(), Trace.method("onSharedSourceChanged")
                                .info("topic", request.getTopic())
                                .info("groupId", request.getGroupId())
                                .info("seqId", request.getSeqId())
                                .info("queueId", request.getQueueId()));

                        // Notify when share source changed
                        notifyHandler.onNotify(request.getPrevSeqId(), request.getMsg(), request.getSeqId());
                    }
                }
            };

            HMR.getService(Channel.class).addNotificationHandler(sharedHandler);
            HMR.getService(Channel.class).subscribeGroupcast(getGroup(), null);
        }

        @Override
        public void stop() {
            Log.i(sourceName(), Trace.method("stop"));

            HMR.getService(Channel.class).removeNotificationHandler(sharedHandler);
            HMR.getService(Channel.class).unSubscribeGroupcast(getGroup(), null);
        }

        private String getGroup() {
            return String.format(Locale.US, "hummer:%d:%s:%d", HMRContext.getAppId(), topic, groupId);
        }

        private String prefKey() {
            String prefKey;
            if (topic.isEmpty()) {
                prefKey = groupId + PrefKeySeqID;
            } else {
                prefKey = groupId + "_" + topic + PrefKeySeqID;
            }

            if (queueId > 0) {
                prefKey = queueId + "_" + prefKey;
            }

            return prefKey;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }

            Shared shared = (Shared) o;

            if (queueId != shared.queueId) {
                return false;
            }
            if (groupId != shared.groupId) {
                return false;
            }
            if (topic != null ? !topic.equals(shared.topic) : shared.topic != null) {
                return false;
            }
            return area != null ? area.equals(shared.area) : shared.area == null;
        }

        @Override
        public int hashCode() {
            int result = queueId;
            result = 31 * result + (int) (groupId ^ (groupId >>> 32));
            result = 31 * result + (topic != null ? topic.hashCode() : 0);
            result = 31 * result + (area != null ? area.hashCode() : 0);
            return result;
        }
    }
}

