package com.hummer._internals.mq;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.Error;
import com.hummer._internals.utility.IMRPC;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.proto.Pull;
import com.hummer._internals.shared.StringChain;
import com.hummer._internals.utility.RichCompletionArg;
import com.hummer._internals.utility.CompletionUtils;

public class RPCPullPrivateMessages extends IMRPC<Pull.PullMsgRequest,
        Pull.PullMsgRequest.Builder, Pull.PullMsgResponse> {

    public static final String TAG = "RPCPullPrivateMessages";
    private final boolean isFirstDrain;
    private final int count;
    private final long fromSeqId;
    private final String topic;
    private final int queueId;
    private final RichCompletionArg<RPCPullingResponse> completion;
    private Trace.Flow flow;

    public RPCPullPrivateMessages(int queueId,
                                  String topic,
                                  boolean firstDrain,
                                  long fromSeqId,
                                  int count,
                                  Trace.Flow flow,
                                  RichCompletionArg<RPCPullingResponse> completion) {
        this.queueId = queueId;
        this.topic = topic;
        this.count = count;
        this.fromSeqId = fromSeqId;
        this.completion = completion;
        this.isFirstDrain = firstDrain;
        this.flow = flow != null ? flow : new Trace.Flow();
    }

    @Override
    public long getLogId() {
        return flow.logId;
    }

    @Override
    public String getHummerFunction() {
        return "PullMsg";
    }

    @Override
    public void buildHummerRequest(@NonNull Pull.PullMsgRequest.Builder builder) throws Throwable {
        builder.setExclusiveStartSeqId(fromSeqId)
                .setLimit(count)
                .setTopic(topic)
                .setStatisDeliveryDelay(!isFirstDrain)
                .setQueueId(queueId)
                .build().toByteArray();
    }

    @Override
    public void handleHummerSuccess(@NonNull Pull.PullMsgResponse res) throws Throwable {
        Long maxSeqId = res.getMaxSeqId();
        if (maxSeqId == 0) {
            maxSeqId = null;
        }
        CompletionUtils.dispatchSuccess(completion,
                new RPCPullingResponse(res.getMsgsList(), res.getHasMore(), maxSeqId));
    }

    @Override
    public String describeHummerResponse(@NonNull Pull.PullMsgResponse res) {
        return new StringChain().acceptNullElements()
                .add("size", res.getMsgsCount())
                .add("hasMore", res.getHasMore())
                .add("maxSeqId", res.getMaxSeqId())
                .toString();
    }

    @Override
    public void handleHummerError(@Nullable Pull.PullMsgResponse res, @NonNull Error error) {
        CompletionUtils.dispatchFailure(completion, error);
    }

}

