package com.hummer._internals.mq;

import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer.Error;
import com.hummer.ErrorEnum;
import com.hummer._internals.utility.IMRPC;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.proto.User;
import com.hummer._internals.shared.StringChain;
import com.hummer._internals.utility.RichCompletionArg;
import com.hummer._internals.utility.CompletionUtils;

public class RPCFetchPrivateSeqId extends IMRPC<User.GetMaxAcquiredSeqIDRequest,
        User.GetMaxAcquiredSeqIDRequest.Builder, User.GetMaxAcquiredSeqIDResponse> {
    public static final String TAG = "RPCFetchPrivateSeqId";
    private final int queueId;
    private final String topic;
    private final RichCompletionArg<Long> completion;
    private final Trace.Flow flow = new Trace.Flow();

    public RPCFetchPrivateSeqId(int queueId, String topic, RichCompletionArg<Long> completion) {
        this.queueId = queueId;
        this.topic = topic;
        this.completion = completion;
    }

    @Override
    public long getLogId() {
        return flow.logId;
    }

    @Override
    public String getHummerFunction() {
        return "GetMaxAcquiredSeqID";
    }

    @Override
    public String describeHummerResponse(@NonNull User.GetMaxAcquiredSeqIDResponse res) {
        return new StringChain().acceptNullElements()
                .add("srvCode", res.getCode())
                .add("srvDesc", res.getMsg())
                .add("seqId", res.getSeqId())
                .toString();
    }

    @Override
    public void buildHummerRequest(@NonNull User.GetMaxAcquiredSeqIDRequest.Builder builder) throws Throwable {
        builder.setQueueId(queueId).setTopic(topic).build().toByteArray();
    }

    @Override
    public void handleHummerSuccess(@NonNull User.GetMaxAcquiredSeqIDResponse res) throws Throwable {
        CompletionUtils.dispatchSuccess(completion, res.getSeqId());
    }

    @Override
    public void handleHummerError(@Nullable User.GetMaxAcquiredSeqIDResponse res, @NonNull Error error) {
        if (error.code == ErrorEnum.RESOURCE_NOT_FOUND_EXCEPTION.getCode()) {
            // 遇到服务器返回结果为2004时，代表此账号未设置过maxacuiredseqid，当strategy为Continuously时，
            // 会因为查询失败而不作seqid的存储，这样会造成此种情况下加载不出seqid而造成很多无效请求
            // 所以碰到这种情况，以seqid为0返回成功结果
            CompletionUtils.dispatchSuccess(completion, 0L);
        } else {
            CompletionUtils.dispatchFailure(completion, error);
        }
    }
}

