package com.hummer._internals.mq;

import android.support.annotation.NonNull;

import com.hummer.Error;
import com.hummer.HMR;
import com.hummer._internals.channel.Channel;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.proto.Im;
import com.hummer._internals.report.StatisticsReporter;
import com.hummer._internals.user.UserService;
import com.hummer._internals.utility.CompletionUtils;
import com.hummer._internals.utility.HMRContext;
import com.hummer._internals.utility.Objects;
import com.hummer._internals.utility.PrefStorage;
import com.hummer._internals.utility.RichCompletion;
import com.hummer._internals.utility.RichCompletionArg;
import com.hummer._internals.utility.ServiceProvider;
import com.hummer.model.completion.OnFailure;
import com.hummer.model.completion.OnSuccess;
import com.hummer.model.completion.OnSuccessArg;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CopyOnWriteArraySet;

public final class MQServiceImpl implements MQService,
        MQService.MessagesDispatcher,
        ServiceProvider.Service {

    private static final String TAG = "MQService";
    // Stable properties
    private final CopyOnWriteArraySet<Source> sources = new CopyOnWriteArraySet<>();
    // Unstable properties
    private String runningToken;
    private boolean isConnected;
    /**
     * 队列总数
     */
    private Integer queueCount;
    private Channel.StateChangedListener channelStateListener;
    private FetchStrategy mFetchStrategy = FetchStrategy.Continuously;

    @Override
    public Class[] staticDependencies() {
        return new Class[]{Channel.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return null;
    }

    @Override
    public void initService() {
    }

    @Override
    public void deInitService() {
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class, PrefStorage.class};
    }

    @Override
    public void openService(@NonNull final RichCompletion completion) {
        if (runningToken != null) {
            Log.w(TAG, Trace.method("openService").msg("在服务已运行状态下调用了openService方法"));
            return;
        }

        runningToken = UUID.randomUUID().toString();

        prepareChannel();

        if (HMRContext.region != null) {
            HMRContext.work.async(new Runnable() {
                @Override
                public void run() {
                    reportUserRegion();
                }
            });
        }

        startSource();
        getQueueConfig();

        CompletionUtils.dispatchSuccess(completion);
    }

    private void getQueueConfig() {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                HMR.getService(Channel.class).run(new RPCGetQueueConfig(new RichCompletionArg<Integer>()
                        .onSuccess(new OnSuccessArg<Integer>() {
                            @Override
                            public void onSuccess(Integer result) {
                                queueCount = result;
                                if (queueCount != null && queueCount > 1) {
                                    for (Source source : sources) {
                                        addQueueSource(source);
                                    }
                                }
                            }
                        }).onFailure(new OnFailure() {
                            @Override
                            public void onFailure(Error error) {
                                queueCount = null;
                            }
                        })));
            }
        });
    }

    private void reportUserRegion() {
        /*
        region 为啥不缓存起来？
        用户在设备A 登录选择中国(chn)区域， 然后在设备B 登录选择印尼(idn)区域，再用设备A 登录
        如果缓存起来，那么再次在设备A 登录，发现缓存了chn，那么就不会去上报，然而用户最新上报的是idn区域。
        这样会导致用户收不到消息
        所以简单粗暴，每次open都上报。当然如果只需要单端登录，有互踢的通知将缓存清除掉也是可以的。
         */
        Log.i(TAG, Trace.method("reportUserRegion").msg("start"));

        HMR.getService(Channel.class).run(new RPCReportUserRegion(new RichCompletion()
                .onSuccess(new OnSuccess() {
                    @Override
                    public void onSuccess() {
                        Log.i(TAG, Trace.method("reportUserRegion")
                                .msg("success")
                                .info("user", HMR.getMe())
                                .info("region", HMRContext.region == null ? null : HMRContext.region.area));
                    }
                })
                .onFailure(new OnFailure() {
                    @Override
                    public void onFailure(Error error) {
                        Log.e(TAG, Trace.method("reportUserRegion")
                                .msg("failed")
                                .info("user", HMR.getMe())
                                .info("region", HMRContext.region == null ? null : HMRContext.region.area)
                                .info("error", error));

                        HMRContext.work.asyncAfter(5 * 60 * 1000, new Runnable() {
                            @Override
                            public void run() {
                                reportUserRegion();
                            }
                        });
                    }
                })));
    }

    private void startSource() {
        for (Source s : sources) {
            s.start(MQServiceImpl.this);
        }

        scheduleTimerPulse();
    }

    @Override
    public void closeService() {
        if (runningToken == null) {
            Log.w(TAG, Trace.method("closeService").msg("在服务已关闭状态下调用了closeService方法"));
            return;
        }

        for (Source s : sources) {
            s.stop();
        }

        teardownChannel();
        runningToken = null;
    }

    @Override
    public FetchStrategy getFetchStrategy() {
        return mFetchStrategy;
    }

    @Override
    public void setFetchStrategy(FetchStrategy strategy) {
        mFetchStrategy = strategy;
    }

    @Override
    public void addSource(final Source source) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                doAddSource(source);

                if (queueCount != null && queueCount > 1) {
                    addQueueSource(source);
                }
            }
        });
    }

    private void addQueueSource(final Source source) {
        for (int i = 1; i < queueCount; i++) {
            if (source instanceof com.hummer._internals.mq.Source) {
                Source queueSource = assembleSource(source, i);
                if (queueSource != null) {
                    doAddSource(queueSource);
                }
            }
        }
    }

    private void doAddSource(final Source source) {
        Log.i(TAG, Trace.method("addSource")
                .info("source", source));

        if (sources.contains(source)) {
            Log.i(TAG, Trace.method("addSource")
                    .info("Duplicate source", source));
            return;
        }

        sources.add(source);

        // Draining标记未重置的上报频率应该和sources的数量是相关的，否则同一个问题就会被上报多次
        StatisticsReporter.Frequencies.put(StatisticsReporter.Codes.NotResetIsDraining, sources.size());

        // 应立即启动CIM服务运行时过程添加的源
        if (MQServiceImpl.this.isRunning()) {
            source.start(MQServiceImpl.this);
        }
    }

    @Override
    public void removeSource(final Source source) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                doRemoveSource(source);

                if (queueCount != null && queueCount > 1) {
                    for (int i = 1; i < queueCount; i++) {
                        if (source instanceof com.hummer._internals.mq.Source) {
                            Source queueSource = assembleSource(source, i);
                            if (queueSource != null) {
                                doRemoveSource(queueSource);
                            }
                        }
                    }
                }
            }
        });
    }

    private void doRemoveSource(final Source source) {
        Log.i(TAG, Trace.method("removeSource")
                .info("source", source));

        Source operateSource = getSource(source);

        // 应立即关闭CIM服务运行时过程添加的源，否则该源的一些相关资源可能无法正确释放
        if (MQServiceImpl.this.isRunning()) {
            operateSource.stop();
        }

        sources.remove(operateSource);
    }

    private Source assembleSource(Source source, int i) {
        Source queueSource = null;
        if (((com.hummer._internals.mq.Source) source).getMode()
                instanceof com.hummer._internals.mq.Source.Private) {
            com.hummer._internals.mq.Source.Private mode
                    = (com.hummer._internals.mq.Source.Private)
                    ((com.hummer._internals.mq.Source) source).getMode();
            queueSource = new com.hummer._internals.mq.Source(
                    new com.hummer._internals.mq.Source.Private(i,
                            mode.getTopic(),
                            mode.getStrategy(),
                            mode.getPullingPeriod()));

        } else if (((com.hummer._internals.mq.Source) source).getMode()
                instanceof com.hummer._internals.mq.Source.Shared) {
            com.hummer._internals.mq.Source.Shared mode
                    = (com.hummer._internals.mq.Source.Shared)
                    ((com.hummer._internals.mq.Source) source).getMode();
            queueSource = new com.hummer._internals.mq.Source(
                    new com.hummer._internals.mq.Source.Shared(i,
                            mode.getGroupId(),
                            mode.getTopic(),
                            mode.getArea(),
                            mode.getStrategy(),
                            mode.getPullingPeriod()));
        }

        return queueSource;
    }

    @Override
    public void pullManually() {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.method("pullManually"));

                MQServiceImpl.this.notifyPullManually();
            }
        });
    }

    @Override
    public void dispatch(final List<Im.Msg> msgs, com.hummer._internals.mq.Source fromSource) {
        for (Im.Msg msg : msgs) {
            parseMsg(msg, fromSource);
        }
    }

    private Source getSource(Source source) {
        for (Source s : sources) {
            if (s.equals(source)) {
                return s;
            }
        }

        return source;
    }

    private void prepareChannel() {
        if (channelStateListener != null) {
            Log.e("MQServiceImpl", Trace.method("prepareChannel")
                    .info("channelStateListener exception", null));
            return;
        }

        isConnected = true;

        channelStateListener = new Channel.StateChangedListener() {
            @Override
            public void onChannelConnecting() {

            }

            @Override
            public void onChannelConnected() {

            }

            @Override
            public void onChannelBinded() {
                HMRContext.work.async(new Runnable() {
                    @Override
                    public void run() {
                        Log.i(TAG, Trace.method("onChannelBinded"));
                        isConnected = true;
                        notifyNetworkReconnected();
                    }
                });
            }

            @Override
            public void onChannelDisconnected() {
                HMRContext.work.async(new Runnable() {
                    @Override
                    public void run() {
                        Log.i(TAG, Trace.method("onChannelDisconnected"));
                        isConnected = false;
                    }
                });
            }
        };
        HMR.getService(Channel.class).addStateListener(channelStateListener);
    }

    private void teardownChannel() {
        HMR.getService(Channel.class).removeStateListener(channelStateListener);
        channelStateListener = null;

        isConnected = false;
    }

    private void scheduleTimerPulse() {
        final String token = runningToken;

        final int periodMillis = 60 * 1000; // 激发时间脉冲事件的间隔，单位为毫秒
        final Runnable inspector = new Runnable() {
            @Override
            public void run() {
                // 对于长延时任务，必须进行token校验，否则可能引发非当前用户上下文s的操作
                if (!Objects.equals(token, runningToken)) {
                    // Timer的终止条件是调用了closeService方法，进而清除/改变了runningToken
                    // 而一旦timer的工作上下文发生了变化，则应停止timer，直到用户重新调用了openService，会再次在另一个
                    // 独立的timer配合其上下文进行工作。
                    return;
                }

                if (!isConnected) {
                    // 通道断连时，不应该发起周期请求，例如延时消息拉取。否则会导致大量无效请求
                    return;
                }

                for (Source s : sources) {
                    s.onTimerPulse(MQServiceImpl.this);
                }

                HMRContext.work.asyncAfter(periodMillis, this);
            }
        };

        HMRContext.work.asyncAfter(periodMillis, inspector);
    }

    private boolean isRunning() {
        return runningToken != null;
    }

    private void notifyNetworkReconnected() {
        for (Source s : sources) {
            s.onNetworkReconnected(this);
        }
    }

    private void notifyPullManually() {
        for (Source s : sources) {
            s.onManualPullingRequest(this);
        }
    }

    @Override
    public void registerMsgParser(MsgParser parser) {
        mMsgParsers.add(parser);
    }

    @Override
    public void unRegisterMsgParser(MsgParser parser) {
        mMsgParsers.remove(parser);
    }

    private void parseMsg(Im.Msg msg, com.hummer._internals.mq.Source fromSoure) {
        for (MsgParser parser : mMsgParsers) {
            try {
                parser.parse(msg, fromSoure);
            } catch (Exception e) {
                Log.e(TAG, Trace.method("parseMsg")
                        .msg("Parsing failed")
                        .info("msg", msg)
                        .info("exception", e));
            }
        }
    }

    private final List<MsgParser> mMsgParsers = new ArrayList<>();
}

