package com.hummer._internals.mq;

import android.support.annotation.NonNull;

import com.hummer._internals.proto.Im;
import com.hummer._internals.utility.RichCompletion;

import java.util.List;

public interface MQService {

    FetchStrategy getFetchStrategy();
    void setFetchStrategy(FetchStrategy strategy);

    /**
     * 新增消息来源
     * <p>
     * 该操作既可以在用户上下文启用前调用，也可以在之后调用。但一般来说，建议在尽可能早的时机进行决策。
     * 例如通过和业务服务器协商，使用固定的来源，以便降低实现成本。
     */
    void addSource(Source source);

    /**
     * 移除消息来源
     * <p>
     * 该操作既可以在用户上下文启用前调用，也可以在之后调用。但一般来说，动态源管理会极大增加业务
     * 实现成本，因此不建议在用户登录后进行消息源管理。
     */
    void removeSource(Source source);

    /**
     * 强行拉取消息数据
     */
    void pullManually();

    void openService(@NonNull RichCompletion completion);

    void closeService();

    void registerMsgParser(MsgParser parser);

    void unRegisterMsgParser(MsgParser parser);

    /**
     * 离线消息拉取的策略类型
     * CIM支持离线消息能力，因此在用户上下文启用时，需要通过不同的策略来决定如何消费离线消息
     */
    enum FetchStrategy {
        /**
         * 接着此前消费的最后一条消息开始持续地获取消息，以尽可能保证消息的连续性。该策略可以满足绝大
         * 部分应用场景。首次激活时不会获取已经被它端消费过的消息。
         */
        Continuously,

        /**
         * 跳过已被消费的消息
         * <p>
         * 因为某些特殊原因，例如业务服务器强行指定消费起始点，部分消息可能会被标记为已消费。那么
         * 后续处理应该跳过它们。该策略仅在非常少的应用场景中会被使用，目前可能包含：
         * > 用户在多个设备中使用同一账号进行登录，但是业务希望已经被其中一台设备获取过的消息，
         * 在另一端的应用启动时，不再重复获取
         * > 存在两套同时工作，但是数据互通的IM SDK，在它们之间进行切换时，被其中一个SDK所消
         * 费过的消息，不应该再被另一个SDK所获取
         */
        SkipFetched,

        /**
         * 强行拉取服务器留存的所有消息
         */
        ReloadHistories,

        /**
         * 从服务器获取当时最新的seqId，然后从该位置开始拉取
         */
        IgnoreBefore,
    }

    interface MessagesDispatcher {
        void dispatch(List<Im.Msg> messages, com.hummer._internals.mq.Source fromSource);
    }

    /**
     * Source 定义了消息来源抽象接口，具体来源必定实现该接口
     * <p>
     * 消息的来源是多样化的，例如：
     * > 单聊消息，来源于私有消息源
     * > 服务号消息，来源于全员共享源
     * > 房间消息，来源于私有消息源，或组播通知
     * <p>
     * 目前而言，业务只能增加私有消息源来实现消息源扩展。具体请参见PrivateSource定义
     */
    interface Source {
        void onTimerPulse(MessagesDispatcher dispatcher);

        void onManualPullingRequest(MessagesDispatcher dispatcher);

        void onNetworkReconnected(MessagesDispatcher dispatcher);

        void start(MessagesDispatcher dispatcher);

        void stop();
    }

    interface MsgParser {
        void parse(Im.Msg msg, com.hummer._internals.mq.Source fromSource) throws Exception;
    }
}
