package com.hummer._internals.log;

import android.support.annotation.IntDef;
import android.support.annotation.NonNull;

import com.hummer.Error;
import com.hummer.HMR;
import com.hummer._internals.report.StatisticsReporter;

import java.lang.annotation.Retention;
import java.lang.annotation.RetentionPolicy;
import java.util.Locale;

import static android.util.Log.DEBUG;
import static android.util.Log.ERROR;
import static android.util.Log.INFO;
import static android.util.Log.WARN;
import static android.util.Log.println;

/**
 * CIM内部完全依赖Log模块进行日志记录处理，提供了简单、多级的日志记录能力。
 */
public final class Log {

    /**
     * CIM内部完全依赖Log模块进行日志记录处理，提供了简单、多级的日志记录能力。接入业务应在{@link HMR#init}中提供
     * 具体日志记录器的实例。否则只能在本地的Logcat中进行日志输出。
     */
    public interface Logger {

        /**
         * 该注解复用了Android本身的日志分类方式，帮助{@link #log}方法进行level的传值约束
         */
        @IntDef({DEBUG, INFO, WARN, ERROR})
        @Retention(RetentionPolicy.SOURCE)
        @interface Level {
        }

        /**
         * 进行日志记录
         *
         * @param lv  日志等级，{@link Level}，明确的日志等级对快速日志定位与分类有很大帮助
         * @param msg 日志详情
         */
        void log(@Level int lv, String msg);
    }

    // 在记录日志时，如果每次都用Log.log(Level.Info, "TAGS", "message");
    // 但日志记录属于高频调用，因此下面的工具方法提供了更简单的写法。对上面这条日志来说，可写为：
    // Log.i("TAGS", "message");
    // 不同的日志等级，本别对应不同的工具方法：
    // d: debug, i: info, w: warning, e: err

    public static void d(String tag, Object msg) {
        write(DEBUG, tag, msg.toString());
    }

    public static void i(String tag, Object msg) {
        write(INFO, tag, msg.toString());
    }

    public static void w(String tag, Object msg) {
        write(WARN, tag, msg.toString());
    }

    public static void e(String tag, Object msg) {
        write(ERROR, tag, msg.toString());
    }

    public static void e(final String tag, @NonNull final Error error, final Object msg) {
        write(ERROR, tag, msg.toString() + " >> " + error.toString());

        StatisticsReporter.report(new StatisticsReporter.Fields(error.code) {{
            setErrInfo("[" + tag + "] " + msg.toString() + " >> " + error.desc);
        }});
    }

    public static void setLogger(@NonNull final Logger logger) {
        Log.logger = logger;
    }

    public static Logger getLogger() {
        return Log.logger;
    }

    private static void write(@Logger.Level int level, final String tag, final String message) {
        if (logger != null) {
            logger.log(level, stringFrom(tag, message));
        } else {
            println(level, "", stringFrom(tag, message));
        }
    }

    private static String stringFrom(String tag, String message) {
        Thread currentThread = Thread.currentThread();
        String threadInfo = currentThread.getName();
        int pid = android.os.Process.myPid();

        if (tag == null) {
            return String.format(Locale.US, "(%d-%-12s)%s", pid, threadInfo, message, pid);
        } else {
            return String.format(Locale.US, "(%d-%-12s)[%s] %s", pid, threadInfo, tag, message);
        }
    }

    private static Logger logger;
}
