package com.hummer._internals.channel;

import android.os.Handler;
import android.support.annotation.NonNull;
import android.util.SparseIntArray;

import com.google.protobuf.InvalidProtocolBufferException;
import com.hummer.Error;
import com.hummer.ErrorEnum;
import com.hummer.HMR;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.report.StatisticsReporter;
import com.hummer._internals.user.UserService;
import com.hummer._internals.utility.CompletionUtils;
import com.hummer._internals.utility.HMRContext;
import com.hummer._internals.utility.HummerException;
import com.hummer._internals.utility.IMRPC;
import com.hummer._internals.utility.RichCompletion;
import com.hummer.model.completion.OnFailure;
import com.hummer.model.completion.OnSuccess;
import com.yy.platform.baseservice.ConstCode;
import com.yy.platform.baseservice.IChannelListener;
import com.yy.platform.baseservice.IRPCChannel;
import com.yy.platform.baseservice.YYServiceCore;
import com.yy.platform.baseservice.profile.ChannelProfile;
import com.yy.platform.baseservice.profile.LogProfile;
import com.yy.platform.baseservice.profile.ServiceProfileFactory;
import com.yy.platform.baseservice.task.BindTask;
import com.yy.platform.baseservice.task.BroadSubOrUnSubTaskV2;
import com.yy.platform.baseservice.task.RPCTask;
import com.yy.platform.baseservice.task.UnBindTask;
import com.yy.platform.baseservice.utils.UserGroupTypeString;
import com.yy.spidercrab.SCLog;

import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 用于实现Hummer内部通信的具体通道
 */
public class ServiceChannel implements Channel {

    @Override
    public Class[] staticDependencies() {
        return null;
    }

    @Override
    public Class[] inherentDynamicDependencies() {
        return new Class[]{UserService.class};
    }

    @Override
    public Class[] plantingDynamicDependencies() {
        return null;
    }

    @Override
    public void initService() {
        if (mode instanceof AutonomousMode) {
            SCLog.addLogger("Service");

            //如果未有中间件或其他模块初始化ServiceSDK，则由本模块启动初始化ServiceSDK并上报数据
            YYServiceCore.initWithGSLB(HMRContext.getAppContext(), HMRContext.appId,
                    "", new ServiceProfileFactory() {
                        @Override
                        public LogProfile logProfile() {
                            return new LogProfile() {
                                @Override
                                public ILog getLog() {
                                    return new ILog() {
                                        @Override
                                        public void outputLog(String log) {
                                            SCLog.i("Service", log);
                                        }
                                    };
                                }

                                @Override
                                public String logPath() {
                                    return null;
                                }

                                @Override
                                public boolean isLogCat() {
                                    return false;
                                }
                            };
                        }

                        @Override
                        public ChannelProfile channelProfile() {
                            return null;
                        }
                    }, null);
        }
    }

    @Override
    public void deInitService() {
        if (mode instanceof AutonomousMode) {
            YYServiceCore.deInit();
        }
    }

    public interface Mode {
        void performOpening(@NonNull Long me, @NonNull RichCompletion completion);

        void performClosing();

        void refreshToken(@NonNull final String token, RichCompletion completion);
    }

    /**
     * 该模式下，Hummer只是简单地使用通道进行通信，不会进行鉴权处理。如果通同一个App内有多个子系统都接入了Service通道，则鉴权处理应该在更高层的地方进行处理，并且使用该模式进行工作
     */
    @SuppressWarnings("unused")
    public static final class DelegateMode implements Mode {
        @Override
        public void performOpening(@NonNull Long me, @NonNull RichCompletion completion) {
            CompletionUtils.dispatchSuccess(completion);
        }

        @Override
        public void performClosing() {
        }

        @Override
        public void refreshToken(@NonNull String token, RichCompletion completion) {

        }
    }

    /**
     * 该模式下，Hummer会拥有通道的完整控制权，包括如何进行用户鉴权等。如果整个App只接入了一个使用Service通道的SDK，可以使用该种模式
     */
    @SuppressWarnings("unused")
    public static final class AutonomousMode implements Mode {
        /**
         * 国际版UDB的service token，使用用yyloginlitesdk 的getServiceToken 接口得到token
         */
        public final static int NEW_UDB_TOKEN = 0;
        /**
         * 第三方产的service token
         */
        public final static int THIRD_USER_TOKEN = 1;
        /**
         * YY体系（国内UDB）下的linkd token
         */
        public final static int YY_UDB_TOKEN = 2;

        public AutonomousMode(int tokenType) {
            this.tokenType = tokenType;
        }

        public AutonomousMode(int tokenType, String token) {
            this.token = token;
            this.tokenType = tokenType;
        }

        @Override
        public void performOpening(@NonNull Long me, @NonNull final RichCompletion completion) {
            Log.i(TAG, Trace.method("performOpening").msg("bind service start"));
            hasBindService = true;
            bindService(me, completion);
        }

        private void bindService(Long me, @NonNull final RichCompletion completion) {
            YYServiceCore.getInstance().bind(
                    me,
                    tokenType,
                    new IChannelListener.ITokenProvider() {
                        @Override
                        public byte[] getToken(long uid) {
                            if (token == null) {
                                Log.i(TAG, Trace.method("getToken").msg("token: null"));
                                return "".getBytes();
                            }

                            Log.i(TAG, Trace.method("getToken").msg("token: " + token.length()));
                            return token.getBytes();
                        }
                    },
                    new IRPCChannel.RPCCallbackRespHeaders<BindTask.ResponseParam>() {
                        @Override
                        public void onSuccess(int requestId, BindTask.ResponseParam response) {
                            Log.i(TAG, Trace.method("bindService").msg("bind service done"));
                            CompletionUtils.dispatchSuccess(completion);
                        }

                        @Override
                        public void onFail(int requestId,
                                           int sdkResCode,
                                           int srvResCode,
                                           Map<String, String> srvHeaders,
                                           Exception e) {
                            Log.i(TAG, Trace.method("bindService")
                                    .msg("bind service fail")
                                    .info("reqId", requestId)
                                    .info("sdkResCode", sdkResCode)
                                    .info("srvResCode", srvResCode)
                                    .info("srvHeaders", srvHeaders));

                            // 先判断sdkResCode是否等于1。
                            // 不等于1表示sdk执行有误，解析相应错误；等于1表示sdk执行正常，解析srvResCode
                            // 如果srvResCode == 403, 需要判断header 是否存在，存在则需要解析header
                            Error error = getError(sdkResCode, srvResCode, srvHeaders);

                            CompletionUtils.dispatchFailure(completion, error);
                        }
                    });
        }

        @Override
        public void performClosing() {
            // 发现callback不能设置为null否则会崩溃
            Log.i(TAG, Trace.method("performClosing").msg("unbind service start"));
            hasBindService = false;
            YYServiceCore.getInstance()
                    .unBind(new IRPCChannel.RPCCallback<UnBindTask.ResponseParam>() {
                        @Override
                        public void onSuccess(int requestId, UnBindTask.ResponseParam response) {
                            Log.i(TAG, Trace.method("performClosing").msg("unbind service done"));
                        }

                        @Override
                        public void onFail(int requestId, int sdkResCode, int srvResCode,
                                           Exception e) {
                            Log.i(TAG, Trace.method("performClosing").msg("unbind service fail"));
                        }
                    });
        }

        @Override
        public void refreshToken(@NonNull String token, final RichCompletion completion) {
            Log.i(TAG, Trace.method("refreshToken").msg("token.size: " + (token == null ? null : token.length())));

            this.token = token;

            if (HMR.getState() != HMR.State.Connected || !hasBindService) {
                Error error = new Error(ErrorEnum.UNINITIALIZED_EXCEPTION, "hmr state has not connected");
                Log.e(TAG, Trace.method("refreshToken")
                        .msg("error: %s | ", error.toString())
                        .info("state", HMR.getState()));
                CompletionUtils.dispatchFailure(completion, error);
                return;
            }

            int channelStatus = YYServiceCore.getInstance().getChannelStatus();

            if (HMR.getMe() == null) {
                Error error = new Error(ErrorEnum.BAD_USER_ERROR, "user is null");
                Log.e(TAG, Trace.method("refreshToken")
                        .msg("error: %s", error.toString()));
                CompletionUtils.dispatchFailure(completion, error);
                return;
            }

            Log.i(TAG, Trace.method("refreshToken")
                    .info("status", channelStatus));

            bindService(HMR.getMe(), new RichCompletion()
                    .onSuccess(new OnSuccess() {
                        @Override
                        public void onSuccess() {
                            // 绑定成功, 那就打个日志喽
                            Log.i(TAG, Trace.method("refreshToken").msg("refresh token success"));
                            CompletionUtils.dispatchSuccess(completion);
                        }
                    }).onFailure(new OnFailure() {
                        @Override
                        public void onFailure(Error error) {
                            Log.e(TAG, Trace.method("refreshToken")
                                    .msg("refresh token failed")
                                    .info("error", error.toString()));

                            // 失败了，回调绑定失败的通知，上层做修改状态处理
                            channelBindResultHandler.onChannelBindResult(error);

                            // 同步结果给调用方
                            CompletionUtils.dispatchFailure(completion, error);
                        }
                    }));
        }

        private final int tokenType;
        private String token;
        private boolean hasBindService = false;
    }

    public ServiceChannel(Mode mode) {
        if (mode == null) {
            Log.e("ServiceChannel", Trace.method("ServiceChannel")
                    .info("mode", "null"));
        }

        this.mode = mode;
    }

    @Override
    public void refreshToken(final String token, RichCompletion completion) {
        this.mode.refreshToken(token, completion);
    }

    @Override
    public void openService(@NonNull final RichCompletion completion) {
        YYServiceCore.getInstance().registUnicastListener(unicastNotifyHandler);
        YYServiceCore.getInstance().registBroadcastListener(bcNotifyHandler);
        YYServiceCore.getInstance().registBroadcastListener(strGroupBroadcastNotifyHandler);
        YYServiceCore.getInstance().setForceUnBindListener(mIforceUnBindNotify);
        YYServiceCore.getInstance().registBindVerifyErrNotify(bindVerifyErrNotify, new Handler());

        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                mode.performOpening(HMR.getMe(), new RichCompletion()
                        .onSuccess(new OnSuccess() {
                            @Override
                            public void onSuccess() {
                                CompletionUtils.dispatchSuccess(completion);
                            }
                        })
                        .onFailure(new OnFailure() {
                            @Override
                            public void onFailure(Error error) {
                                YYServiceCore.getInstance().unregistBroadcastListener(bcNotifyHandler);
                                YYServiceCore.getInstance().unregistUnicastListener(unicastNotifyHandler);
                                YYServiceCore.getInstance().unregistBroadcastListener(strGroupBroadcastNotifyHandler);
                                YYServiceCore.getInstance().setForceUnBindListener(null);
                                YYServiceCore.getInstance().unregistBindVerifyErrNotify(bindVerifyErrNotify);
                                CompletionUtils.dispatchFailure(completion, error);
                            }
                        }));
            }
        });
    }

    @Override
    public void closeService() {
        YYServiceCore.getInstance().unregistBroadcastListener(bcNotifyHandler);
        YYServiceCore.getInstance().unregistUnicastListener(unicastNotifyHandler);
        YYServiceCore.getInstance().unregistBroadcastListener(strGroupBroadcastNotifyHandler);
        YYServiceCore.getInstance().setForceUnBindListener(null);
        YYServiceCore.getInstance().unregistBindVerifyErrNotify(bindVerifyErrNotify);

        this.mode.performClosing();
    }

    @Override
    public void run(final Channel.RPC rpc) {
        Error error = null;

        final Long currentMe = HMR.getMe();
        if (currentMe == null) {
            error = new Error(ErrorEnum.INVALID_PARAMETER, "Missing login user!");
        }

        byte[] requestPayLoad = null;
        try {
            requestPayLoad = rpc.getRequestBytes();
        } catch (Throwable t) {
            // 异常时 requestPayload是null，交给下面的统一判断处理
            Log.e(TAG, Trace.method("run").info("requestBytes error: ", t.getMessage()));
        }

        if (requestPayLoad == null) {
            error = new Error(ErrorEnum.INVALID_PARAMETER, "Failed parsing data from request");
        }

        if (error != null) {
            Log.e(TAG, Trace.method("runRPC").msg(error.toString()));
            rpc.handleError(error);
            return;
        }

        Map<String, String> clientHeaders = new HashMap<>(1, 1);
        clientHeaders.put("SCKDeviceOS", "android");
        clientHeaders.put("APIVersion", "v2");

        RPCTask.RequestParam requestParam = new RPCTask.RequestParam(
                "",
                rpc.serviceName(),
                rpc.getFunctionName(),
                requestPayLoad,
                rpc.protoType(),
                null,
                clientHeaders,
                null);

        final long startTs = System.currentTimeMillis();
        final String shortFunctionName = shortFunctionName(rpc);
        Log.i(TAG, Trace.method("rpcCall")
                .msg("start call rpc timestamp:%s, shortFunctionName:%s, SCKDeviceOS:%s",
                        startTs, shortFunctionName, clientHeaders.get("SCKDeviceOS")));

        final int rpcRequestId = YYServiceCore.getInstance().rpcCall(requestParam,
                null,
                new IRPCChannel.RPCCallback<RPCTask.ResponseParam>() {
                    @Override
                    public void onSuccess(final int requestId, final RPCTask.ResponseParam response) {
                        HMRContext.work.async(new Runnable() {
                            @Override
                            public void run() {

                                Log.i(TAG, Trace.method("rpcCall response")
                                        .msg("Success")
                                        .info("rpcRequestId", requestId)
                                        .info("traceId", response.mTraceId)
                                        .info("funcName", response.mFuncName));

                                Error error = null;

                                try {
                                    if (!HMR.isMe(currentMe)) {
                                        error = new Error(ErrorEnum.BAD_USER_ERROR, "RPC overdue!");
                                    } else {
                                        rpc.handleResponse(response.mResponseData);
                                    }

                                } catch (InvalidProtocolBufferException e) {
                                    error = new Error(ErrorEnum.INTERNAL_SERVER_ERROR,
                                            "Protocol exceptions: " + e.getMessage());
                                } catch (HummerException e) {
                                    error = e.error;
                                } catch (Throwable t) {
                                    error = new Error(ErrorEnum.UNKNOWN_ERROR,
                                            "Undefined exceptions: " + t.getMessage());
                                    t.printStackTrace();
                                }

                                if (error != null) {
                                    rpc.handleError(error);

                                    StatisticsReporter.batchReportReturnCode(shortFunctionName, 0, error.code);
                                }
                            }
                        });
                    }

                    @Override
                    public void onFail(final int requestId, final int sdkResCode, final int srvResCode, Exception e) {
                        HMRContext.work.async(new Runnable() {
                            @Override
                            public void run() {
                                Log.i(TAG, Trace.method("rpcCall response")
                                        .msg("Fail")
                                        .info("rpcRequestId", requestId)
                                        .info("sdkResCode", sdkResCode)
                                        .info("srvResCode", srvResCode));

                                if (!HMR.isMe(currentMe)) {
                                    Error err = new Error(ErrorEnum.BAD_USER_ERROR, "RPC overdue!");

                                    Log.w(TAG, err.desc);
                                    rpc.handleError(err);
                                    return;
                                }

                                Error error = getError(sdkResCode, srvResCode, null);

                                rpc.handleError(error);
                                StatisticsReporter.batchReportReturnCode(shortFunctionName, 0, error.code);
                            }
                        });
                    }
                });

        Log.i(TAG, Trace.method("run")
                .msg("after call rpc timestamp:%s, shortFunctionName:%s, requestId:%d", startTs, shortFunctionName,
                        rpcRequestId));
    }

    @Override
    public void addNotificationHandler(final Channel.NotificationHandler handler) {
        notifyHandlers.add(handler);
    }

    @Override
    public void removeNotificationHandler(Channel.NotificationHandler handler) {
        notifyHandlers.remove(handler);
    }

    @Override
    public void subscribeGroupcast(String group, final RichCompletion completion) {
        Set<UserGroupTypeString> grps = new HashSet<>();
        grps.add(new UserGroupTypeString(group));
        YYServiceCore.getInstance().subscribeStrBroadcast(grps,
                new IRPCChannel.RPCCallback<BroadSubOrUnSubTaskV2.ResponseParam>() {
                    @Override
                    public void onSuccess(int requestId, BroadSubOrUnSubTaskV2.ResponseParam response) {
                        CompletionUtils.dispatchSuccess(completion);
                    }

                    @Override
                    public void onFail(int requestId, int sdkResCode, int srvResCode, Exception e) {
                        CompletionUtils.dispatchFailure(completion,
                                new Error(sdkResCode, e.getLocalizedMessage()));
                    }
                });
    }

    @Override
    public void unSubscribeGroupcast(String group, final RichCompletion completion) {
        Set<UserGroupTypeString> grps = new HashSet<>();
        grps.add(new UserGroupTypeString(group));
        YYServiceCore.getInstance().unSubscribeStrBroadcast(grps,
                new IRPCChannel.RPCCallback<BroadSubOrUnSubTaskV2.ResponseParam>() {
                    @Override
                    public void onSuccess(int requestId, BroadSubOrUnSubTaskV2.ResponseParam response) {
                        CompletionUtils.dispatchSuccess(completion);
                    }

                    @Override
                    public void onFail(int requestId, int sdkResCode, int srvResCode, Exception e) {
                        CompletionUtils.dispatchFailure(completion,
                                new Error(sdkResCode, e.getLocalizedMessage()));
                    }
                });
    }

    @Override
    public void addStateListener(final StateChangedListener listener) {
        IChannelListener.IChannelStatusNotify stateHandler = new IChannelListener.IChannelStatusNotify() {
            @Override
            public void onStatus(int status) {
                if (status == ConstCode.Status.Binded) {
                    listener.onChannelBinded();
                } else if (status == ConstCode.Status.OnConnected) {
                    listener.onChannelConnected();
                } else if (status == ConstCode.Status.OnConnecting) {
                    listener.onChannelConnecting();
                } else {
                    listener.onChannelDisconnected();
                }
            }
        };

        stateListeners.put(listener, stateHandler);

        YYServiceCore.getInstance().registChannelStatusListener(stateHandler, new Handler());
    }

    @Override
    public void removeStateListener(StateChangedListener listener) {
        IChannelListener.IChannelStatusNotify stateHandler = stateListeners.get(listener);
        if (stateHandler == null) {
            return;
        }

        YYServiceCore.getInstance().unregistChannelStatusListener(stateHandler);
        stateListeners.remove(listener);
    }

    @Override
    public void setForceUnbindHandler(ForceUnbindHandler handler) {
        forceUnbindHandler = handler;
    }

    @Override
    public void setChannelBindResultHandler(ChannelBindResultHandler handler) {
        channelBindResultHandler = handler;
    }

    @Override
    public void addTokenInvalidHandler(final TokenInvalidHandler handler) {
        tokenInvalidHandlers.add(handler);
    }

    @Override
    public void removeTokenInvalidHandler(TokenInvalidHandler handler) {
        tokenInvalidHandlers.remove(handler);
    }

    @Override
    public long getLastSyncServerTs() {
        return YYServiceCore.getInstance().getLastSyncServerTS();
    }

    @Override
    public long getAlignmentServerTs() {
        return YYServiceCore.getInstance().getAlignmentServerTS();
    }

    private static String shortFunctionName(Channel.RPC rpc) {
        return (rpc instanceof IMRPC) ? ((IMRPC) rpc).getHummerFunction() : rpc.getFunctionName();
    }

    private void onNotify(final String serviceName, final String functionName, final byte[] data) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                for (NotificationHandler handler : notifyHandlers) {
                    handler.onNotify(serviceName, functionName, data);
                }
            }
        });
    }

    private void notifyTokenVerifyResult(final long uid, final Error error) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                for (TokenInvalidHandler handler : tokenInvalidHandlers) {
                    handler.onTokenVerifyResult(uid, error);
                }
            }
        });
    }

    private static Error getError(int sdkResCode, int srvResCode, Map<String, String> srvHeaders) {
        Error error;
        if (sdkResCode != ConstCode.SdkResCode.RECV_RESPONSE) {
            error = errorFromServiceFailure(true, sdkResCode);
        } else {
            error = getErrorByParseHeader(srvResCode, srvHeaders);
        }
        return error;
    }

    private static Error errorFromServiceFailure(boolean fromSdk, int code) {
        String message;
        if (fromSdk) {
            SparseIntArray codeMapping = new SparseIntArray() {{
                put(ConstCode.SdkResCode.NO_CONNECTED, ErrorEnum.CONNECTION_EXCEPTION.getCode());
                put(ConstCode.SdkResCode.NEED_BIND, ErrorEnum.UNAUTHORIZED_EXCEPTION.getCode());
                put(ConstCode.SdkResCode.TIMEOUT, ErrorEnum.OPERATION_TIMEOUT.getCode());
                put(ConstCode.SdkResCode.TIMEOUT_SEND, ErrorEnum.OPERATION_TIMEOUT.getCode());
            }};

            message = ConstCode.SdkResCode.desc(code);
            code = codeMapping.get(code, ErrorEnum.UNKNOWN_ERROR.getCode());
        } else {
            message = ConstCode.SrvResCode.desc(code);
            code = ErrorEnum.INTERNAL_SERVER_ERROR.getCode();
        }

        return new Error(code, message);
    }

    private static final String TAG = "ServiceChannel";

    private final IChannelListener.IServiceBroadcastNotify bcNotifyHandler =
            new IChannelListener.IServiceBroadcastNotify() {
                @Override
                public void onBroadCast(long uid, long grpType,
                                        long grpId, String serviceName,
                                        String functionName, String protoType, byte[] data) {
                    ServiceChannel.this.onNotify(serviceName, functionName, data);
                }
            };

    private final IChannelListener.IServiceUnicastNotify unicastNotifyHandler =
            new IChannelListener.IServiceUnicastNotify() {
                @Override
                public void onUnicast(long uid, String serviceName,
                                      String functionName, String protoType, byte[] data) {
                    ServiceChannel.this.onNotify(serviceName, functionName, data);
                }
            };

    private final IChannelListener.IServiceStrGroupBroadcastNotify strGroupBroadcastNotifyHandler =
            new IChannelListener.IServiceStrGroupBroadcastNotify() {
                @Override
                public void onBroadCastFromStrGroup(long uid, String grpStr, String serviceName, String functionName,
                                                    String protoType, byte[] data) {
                    ServiceChannel.this.onNotify(serviceName, functionName, data);
                }
            };

    private final IChannelListener.IForceUnBindNotify mIforceUnBindNotify =
            new IChannelListener.IForceUnBindNotify() {
                @Override
                public void onForceOut(long uid, int code, String desc) {
                    if (ServiceChannel.forceUnbindHandler == null) {
                        Log.i(TAG, Trace.method("onForceOut")
                                .info("uid", uid)
                                .info("code", code)
                                .info("desc", desc));
                        return;
                    }

                    ServiceChannel.forceUnbindHandler.onForceOut(uid, Error.ofError(code, desc));
                }
            };

    private final IChannelListener.IBindVerifyErrNotify bindVerifyErrNotify
            = new IChannelListener.IBindVerifyErrNotify() {
        @Override
        public void onBindVerifyError(long uid, int resCode, Map<String, String> srvHeaders) {
            Log.i(TAG, Trace.method("onBindVerifyError")
                    .info("uid", uid)
                    .info("resCode", resCode)
                    .info("srvHeaders", srvHeaders));

            Error error = getErrorByParseHeader(resCode, srvHeaders);
            if (error != null) {
                notifyTokenVerifyResult(uid, error);
            }

        }
    };

    private static final String TOKEN_SERVICE_ERROR_CODE = "tokenservice-errorCode";
    private static final String TOKEN_SERVICE_ERROR_MSG = "tokenservice-errorMsg";

    private static Error getErrorByParseHeader(int srvResCode, Map<String, String> header) {
        Error error;
        if (header != null && header.size() > 0) {
            int code = srvResCode;
            if (header.containsKey(TOKEN_SERVICE_ERROR_CODE)) {
                code = Integer.parseInt(header.get(TOKEN_SERVICE_ERROR_CODE));
            }
            String msg = "";
            if (header.containsKey(TOKEN_SERVICE_ERROR_MSG)) {
                msg = header.get(TOKEN_SERVICE_ERROR_MSG);
            }
            error = new Error(code, msg);
        } else {
            error = errorFromServiceFailure(false, srvResCode);
        }

        return error;
    }

    private static Map<StateChangedListener, IChannelListener.IChannelStatusNotify> stateListeners =
            new HashMap<>();

    private static List<NotificationHandler> notifyHandlers = new LinkedList<>();

    private static ForceUnbindHandler forceUnbindHandler;
    private static ChannelBindResultHandler channelBindResultHandler;

    private static List<TokenInvalidHandler> tokenInvalidHandlers = new LinkedList<>();

    private Mode mode;
}
