package com.hummer._internals.channel;

import android.support.annotation.NonNull;

import com.hummer.Error;
import com.hummer._internals.utility.ServiceProvider;
import com.hummer._internals.utility.RichCompletion;

/**
 * Channel的接口定义，描述了抽象的服务器通信通道行为
 */
public interface Channel extends ServiceProvider.Service {

    /**
     * RPC接口为Channel对应的RPC请求抽象类型，所有与后台通信的具体RPC，都应该实现该接口。
     */
    interface RPC {
        /**
         * 具体的RPC类型应在getNames方法中回对应请求的服务名称信息
         *
         * @return 该RPC对应的服务名
         */
        String getFunctionName();

        /**
         * 具体的RPC类型应在getNames方法中回对应请求的服务名称信息
         *
         * @return 该RPC对应的服务名
         */
        String serviceName();

        /**
         * 具体 RPC 所使用的协议类型，
         * pb 的话应该是 yy-rp-pb
         * yyp 的话应该暂时为 null
         *
         * @return 协议类型
         */
        String protoType();

        /**
         * 生成具体RPC请求的二进制数据
         *
         * @return 具体RPC请求的二进制数据
         */
        byte[] getRequestBytes() throws Throwable;

        /**
         * 当通道成功处理了具体的RPC请求，并返回了响应数据时，会调用onSuccess回调方法进行响应数据处理
         *
         * @param responseBytes 通道返回的响应二进制数据
         */
        void handleResponse(@NonNull byte[] responseBytes) throws Throwable;

        /**
         * 当通道返回了错误信息，例如鉴权失败等，都会通过onError回调方法进行回调
         *
         * @param err 包含错误的具体详情Error类型
         */
        void handleError(@NonNull Error err);
    }

    interface NotificationHandler {
        void onNotify(String serviceName, String functionName, byte[] data);
    }

    interface ForceUnbindHandler {
        void onForceOut(long kickUid, Error error);
    }

    interface TokenInvalidHandler {
        /**
         * token校验结果通知
         *
         * @param userId 用户id
         * @param error  错误结果
         */
        void onTokenVerifyResult(long userId, Error error);
    }

    interface ChannelBindResultHandler {
        void onChannelBindResult(Error error);
    }

    /**
     * Run.
     *
     * @param rpc the rpc
     */
    void run(RPC rpc);

    void addNotificationHandler(NotificationHandler handler);

    void removeNotificationHandler(NotificationHandler handler);

    void subscribeGroupcast(String group, RichCompletion completion);

    void unSubscribeGroupcast(String group, RichCompletion completion);

    interface StateChangedListener {
        void onChannelConnecting();

        void onChannelConnected();

        void onChannelBinded();

        void onChannelDisconnected();
    }

    void addStateListener(StateChangedListener listener);

    void removeStateListener(StateChangedListener listener);

    void setForceUnbindHandler(ForceUnbindHandler handler);

    void setChannelBindResultHandler(ChannelBindResultHandler handler);

    void addTokenInvalidHandler(TokenInvalidHandler handler);

    void removeTokenInvalidHandler(TokenInvalidHandler handler);

    void refreshToken(final String token, RichCompletion completion);

    long getLastSyncServerTs();

    long getAlignmentServerTs();
}
