package com.hummer;

import android.content.Context;
import android.content.pm.ApplicationInfo;
import android.support.annotation.NonNull;
import android.support.annotation.Nullable;

import com.hummer._internals.channel.Channel;
import com.hummer._internals.channel.ServiceChannel;
import com.hummer._internals.log.HummerLogger;
import com.hummer._internals.log.Log;
import com.hummer._internals.log.trace.Trace;
import com.hummer._internals.report.HiidoReporter;
import com.hummer._internals.report.statis.TextUtils;
import com.hummer._internals.utility.CompletionUtils;
import com.hummer._internals.utility.DispatchQueue;
import com.hummer._internals.utility.HMRContext;
import com.hummer._internals.utility.Objects;
import com.hummer._internals.utility.RequestIdBuilder;
import com.hummer._internals.utility.RichCompletion;
import com.hummer._internals.utility.ServiceProvider;
import com.hummer.model.RequestId;
import com.yy.spidercrab.SCLog;

import java.lang.ref.WeakReference;

import static com.hummer._internals.utility.HMRContext.Region.AREA_CN;

/**
 * Hummer类型是服务的总入口，主要包含了SDK初始化，以及用户上下文切换的功能
 * <br>实际业务接入时，必须遵循下述原则：
 * <br>1. 必须在App初始化时，依赖Hummer的模块初始化之前，调用Hummer.init方法，进行初始化
 * <br>2. 在用户身份确定（成功登录）后，调用{@link HMR#login}启用IM用户上下文
 * <br>3. 在用户身份失效（退出登录）后，调用{@link HMR#logout}注销IM用户上下文
 */
public final class HMR {

    private HMR() {
    }

    /**
     * Hummer SDK的初始化方法，关于运行环境的参数，欢迎垂询
     *
     * @param appContext Android应用Context实例对象
     * @param appId      应用appId，区分不同业务的重要标识。目前需要和Hummer服务提供方通过人工申请
     * @param appVersion 应用版本号，主要用于数据统计和应用查障
     * @param listener   事件监听器
     */
    public static void init(@NonNull final Context appContext,
                            final long appId,
                            final String appVersion,
                            @NonNull final HummerEventListener listener) {
        HMRContext.work.sync(new Runnable() {
            @Override
            public void run() {
                if (isInitialized()) {
                    Log.e(TAG, Trace.method("init").msg("Hummer already been initialized!"));
                    return;
                }

                setupLogger(appContext);

                HMR.listener = listener;
                HMRContext.appContext = new WeakReference<>(appContext.getApplicationContext());
                HMRContext.appId = appId;

                /* 如果没有设置，则表示是自治模式。新建一个 */
                if (getService(Channel.class) == null) {
                    Channel channel = new ServiceChannel(new ServiceChannel.AutonomousMode(
                            ServiceChannel.AutonomousMode.THIRD_USER_TOKEN
                    ));

                    ServiceProvider.register(Channel.class, channel);
                }

                Log.i(TAG, "init | appVersion: " + appVersion);
                ServiceProvider.loadServicesIfNeeded(appContext, "com.hummer");
            }
        });
    }

    private static void setupLogger(Context context) {
        if (Log.getLogger() == null) {
            if (isApkInDebug(context)) {
                SCLog.enableConsoleLogger(true);
            }
            SCLog.init(context);
            Log.setLogger(new HummerLogger("HMR"));
        }
    }

    private static boolean isApkInDebug(Context context) {
        try {
            ApplicationInfo info = context.getApplicationInfo();
            return (info.flags & ApplicationInfo.FLAG_DEBUGGABLE) != 0;
        } catch (Exception e) {
            return false;
        }
    }

    /**
     * 切换至指定用户的工作上下文，该方法应该在上层业务的用户登录操作完成之后执行
     *
     * @param uid        用户uid，uid不能为0
     * @param region     需要连接服务器所处的环境参数，详情咨询相关SDK开发人员
     * @param token      用户凭证，用于服务鉴权
     * @param completion 异步操作回调，可为null，会在操作结束时通过其不同的回调方法返回操作结果。
     *
     * @return requestId 请求的唯一标识
     */
    public static RequestId login(final long uid,
                                  @NonNull final String region,
                                  @NonNull final String token,
                                  @Nullable final HMR.Completion completion) {

        final RequestId hummerRequestId = new RequestId(RequestIdBuilder.generateRequestId());
        Log.i(TAG, Trace.method("login").info("hummerRequestId", hummerRequestId));

        final RichCompletion richCompletion = new RichCompletion(hummerRequestId, completion);
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {

                Error error = getLoginError(uid, region);
                if (error != null) {
                    Log.e(TAG, "login | " + error);
                    CompletionUtils.dispatchFailure(richCompletion, error);
                    return;
                }

                HMR.getService(Channel.class).addStateListener(HMR.channelStateListener);
                HMR.getService(Channel.class).addTokenInvalidHandler(TOKEN_INVALID_HANDLER);

                setHummerState(State.Connecting, "Logging");
                refreshToken(token, null);
                initForceKickHandler(uid);
                initChannelBindResultHandler();

                Log.i(TAG, Trace.method("login")
                        .msg("login sdk")
                        .info("uid", uid));

                HMR.me = uid;
                HMRContext.region = HMRContext.Region.make(region);

                setReportRegion();

                ServiceProvider.openServices(richCompletion
                        .beforeFailure(new Runnable() {
                            @Override
                            public void run() {
                                HMRContext.region = null;
                                HMR.me = null;
                                setHummerState(State.Disconnected, "Login failure");
                            }
                        })
                        .beforeSuccess(new Runnable() {
                            @Override
                            public void run() {
                                setHummerState(State.Connected, "Login success");
                                Log.i(TAG, Trace.method("login").msg("finish"));
                            }
                        })
                );
            }
        });

        return hummerRequestId;
    }

    private static Error getLoginError(long uid, String region) {
        if (uid < 0) {
            return new Error(ErrorEnum.INVALID_PARAMETER, "uid cannot be negative");
        }

        if (TextUtils.isBlank(region)) {
            return new Error(ErrorEnum.INVALID_PARAMETER, "region can not been null");
        }

        // 需要init hummer
        if (!isInitialized()) {
            return new Error(ErrorEnum.UNINITIALIZED_EXCEPTION, "hummer has not been init");
        }

        // service需要初始化
        if (getService(Channel.class) == null) {
            return new Error(ErrorEnum.UNAUTHORIZED_EXCEPTION, "hummer channel has not been init");
        }

        // 需要未登录
        if (HMR.state != State.Disconnected) {
            return new Error(ErrorEnum.BAD_USER_ERROR, "Replicated HMR.login");
        }

        return null;
    }

    private static void setReportRegion() {
        // 上报区域设置
        if (AREA_CN.equals(HMRContext.region.area)) {
            HiidoReporter.setRegion(HiidoReporter.Region.China);
        } else {
            HiidoReporter.setRegion(HiidoReporter.Region.Overseas);
        }
    }

    /**
     * 获取服务实例
     *
     * <br> Hummer包含了许多服务接口，例如PeerService, RoomService，所有服务的实例都必须通过getService方法来获取
     *
     * @return 如果正确加载、或者确实可以提供某个Service实例，则返回对应的实现实例，否则返回null
     */
    public static <Service> Service getService(@NonNull Class<Service> serviceClass) {
        // ！！该方法是提供给业务调用的，Hummer内部应直接使用{@link ServiceProvider#get}来获取service实例
        return ServiceProvider.get(serviceClass);
    }

    /**
     * 关闭Hummer的功能，该操作应该在业务进行实际的用户注销（退出登录）之前执行
     *
     * @return requestId 请求的唯一标识
     */
    public static RequestId logout() {
        final RequestId hummerRequestId = new RequestId(RequestIdBuilder.generateRequestId());
        Log.i(TAG, Trace.method("logout").info("hummerRequestId", hummerRequestId));

        logout("Logout");

        return hummerRequestId;
    }

    private static void logout(final String reason) {
        HMRContext.work.async(new Runnable() {
            @Override
            public void run() {
                if (HMR.state == State.Disconnected) {
                    Log.w(TAG, "logout | hummer state is Disconnected");
                    return;
                }

                Log.i(TAG, Trace.method("logout"));
                HMR.getService(Channel.class).removeStateListener(HMR.channelStateListener);
                HMR.getService(Channel.class).removeTokenInvalidHandler(TOKEN_INVALID_HANDLER);

                ServiceProvider.closeServices();

                HMR.me = null;
                HMRContext.region = null;

                setHummerState(State.Disconnected, reason);
                Log.i(TAG, Trace.method("logout").msg("finish"));
            }
        });
    }

    public enum State {
        /**
         * 未连接
         */
        Disconnected,
        /**
         * 连接中
         */
        Connecting,
        /**
         * 重连中
         */
        Reconnecting,
        /**
         * 已连接
         */
        Connected
    }

    public interface HummerEventListener {
        void onHummerStateChanged(State fromState, State toState, String reason);

        void onHummerKicked(int code, String description);

        void onHummerPreviousTokenExpired();
    }

    /**
     * 获取 SDK 当前状态
     *
     * @return SDK当前状态
     */
    public static State getState() {
        return HMR.state;
    }

    /**
     * 获取 SDK 版本号
     *
     * @return SDK版本号
     */
    public static String getVersion() {
        return BuildConfig.BUILD_VERSION;
    }

    /**
     * 获取当前Hummer工作上下文的用户对象
     *
     * @return 如果已经成功通过HMR.open方法启用了用户上下文，则返回对应的用户实例，否则返回null。可以通过返回值来判断Hummer是否已经
     * 正确开启了用户上下文。
     */
    public static Long getMe() {
        return me;
    }

    /**
     * 判断某个标识对象是否当前Hummer的工作用户
     *
     * @param id 欲比对的标识对象
     *
     * @return 如果identity对象实际User类型，且其id值与当前Hummer用户的id值一致，则返回true，否则返回false
     */
    public static boolean isMe(Long id) {
        return Objects.equals(id, me);
    }

    /**
     * 设置日志的保存路径
     *
     * <br> 请不要使用根目录当日志路径
     * <br> 最好在调用{@link HMR#init}之前设置
     * <br> 请尽量保持设置的路径是不变的
     *
     * @param loggerPath 日志路径
     *
     * @return true: 设置成功
     */
    public static boolean setLoggerFilePath(@NonNull String loggerPath) {
        if (android.text.TextUtils.isEmpty(loggerPath)) {
            return false;
        }
        SCLog.setDefaultFilePath(loggerPath);
        return true;
    }

    public interface Completion {
        void onSuccess(RequestId requestId);

        void onFailed(RequestId requestId, Error err);
    }

    public interface CompletionArg<Argument> {
        void onSuccess(RequestId requestId, Argument arg);

        void onFailed(RequestId requestId, Error err);
    }

    /**
     * 刷新用户凭证
     *
     * @param token      待刷新的用户凭证
     * @param completion 异步回调
     *
     * @return 递增的一个请求标识，主要用于查障。
     */
    public static RequestId refreshToken(@NonNull final String token,
                                         @Nullable final HMR.Completion completion) {
        final RequestId hummerRequestId = new RequestId(RequestIdBuilder.generateRequestId());
        Log.i(TAG, Trace.method("refreshToken").info("hummerRequestId", hummerRequestId));

        final RichCompletion richCompletion = new RichCompletion(hummerRequestId, completion);

        if (getService(Channel.class) == null) {
            Error error = new Error(ErrorEnum.UNINITIALIZED_EXCEPTION, "channel service not ready");
            Log.e(TAG, "refreshToken | " + error);
            CompletionUtils.dispatchFailure(richCompletion, error);
            return hummerRequestId;
        }

        getService(Channel.class).refreshToken(token, richCompletion);
        return hummerRequestId;
    }

    private static void initForceKickHandler(final long uid) {
        HMR.getService(Channel.class).setForceUnbindHandler(new Channel.ForceUnbindHandler() {
            @Override
            public void onForceOut(long kickUid, Error error) {
                if (kickUid != uid) {
                    return;
                }

                HMR.logout("Force kicked");
                notifyForceKicked(error.code, error.desc);
            }
        });
    }

    private static void initChannelBindResultHandler() {
        HMR.getService(Channel.class).setChannelBindResultHandler(
                new Channel.ChannelBindResultHandler() {
                    @Override
                    public void onChannelBindResult(Error error) {
                        logout("Login failure");
                    }
                });
    }

    private static void setHummerState(@Nullable final HMR.State state, String reason) {
        if (state == HMR.state) {
            return;
        }

        HMR.State fromState = HMR.state;
        if (state != null) {
            HMR.state = state;
        }

        notifyUpdateHummerState(fromState, HMR.state, reason);
    }

    private static void notifyUpdateHummerState(final HMR.State fromState,
                                                final HMR.State toState,
                                                final String reason) {
        DispatchQueue.main.sync(new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.method("onHummerStateChanged")
                        .info("fromState", fromState)
                        .info("toState", toState)
                        .info("reason", reason));

                HMR.listener.onHummerStateChanged(fromState, toState, reason);
            }
        });
    }

    private static void notifyForceKicked(final int code, final String desc) {
        DispatchQueue.main.sync(new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.method("notifyForceKicked")
                        .info("code", code)
                        .info("desc", desc));
                HMR.listener.onHummerKicked(code, desc);
            }
        });
    }

    private static void notifyTokenInvalid(final long notifyUid, final Error error) {
        if (HMR.getMe() == null || HMR.getMe() != notifyUid) {
            Log.i(TAG, Trace.method("notifyTokenInvalid")
                    .msg("user not login or notify uid is not current user")
                    .info("uid", HMR.getMe())
                    .info("notifyUid", notifyUid));
            return;
        }

        DispatchQueue.main.sync(new Runnable() {
            @Override
            public void run() {
                Log.i(TAG, Trace.method("notifyTokenInvalid")
                        .info("uid", notifyUid)
                        .info("error", error));

                /// 对于主动调用接口(登录、刷新Token)的场景，错误通过接口的completion返回
                /// 这里只需要处理通道层通知的Token过期事件
                if (error.code == ErrorEnum.EXPIRED_TOKEN_EXCEPTION.getCode()) {
                    HMR.listener.onHummerPreviousTokenExpired();
                }
            }
        });
    }

    private static boolean isInitialized() {
        return HMRContext.appId != null;
    }

    private static HummerEventListener listener;
    private static final String TAG = "HMR";
    private static State state = State.Disconnected;
    private static Long me;

    private final static Channel.StateChangedListener channelStateListener = new Channel.StateChangedListener() {
        @Override
        public void onChannelConnecting() {
            // 只能由Connected 到 Reconnecting
            // 如果断网情况下登录，那状态会由 Disconnected -> Connecting
            // 但是由于断网，触发了service断网的通知，就可能造成 Connecting -> Reconnecting的状态变化
            // 根据需求，Reconnecting 只有和 Connected 这两个状态之间切换
            if (getState() == State.Connected) {
                setHummerState(State.Reconnecting, "Network interruption");
            }
        }

        @Override
        public void onChannelConnected() {
            // 账号互踢，也会通知，故去掉
//            setHummerState(State.Reconnecting, "网络中断");
        }

        @Override
        public void onChannelBinded() {
            if (getState() == State.Reconnecting) {
                setHummerState(State.Connected, "Reconnect success");
            }
        }

        @Override
        public void onChannelDisconnected() {
            if (getState() == State.Connected) {
                setHummerState(State.Reconnecting, "Network interruption");
            }
        }
    };

    private final static Channel.TokenInvalidHandler TOKEN_INVALID_HANDLER
            = new Channel.TokenInvalidHandler() {
        @Override
        public void onTokenVerifyResult(long userId, Error error) {
            notifyTokenInvalid(userId, error);
        }
    };
}
