/*
 * Copyright (C) 2013 Leszek Mzyk
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.edu24ol.newclass.widget.viewpager;

import android.content.Context;
import android.database.DataSetObserver;
import android.os.Handler;
import android.os.Message;
import android.util.AttributeSet;
import android.util.Log;
import android.view.MotionEvent;

import androidx.appcompat.app.AppCompatActivity;
import androidx.viewpager.widget.PagerAdapter;
import androidx.viewpager.widget.ViewPager;

import com.sensorsdata.analytics.android.sdk.SensorsDataAPI;
import com.yy.android.educommon.log.YLog;

import java.lang.ref.WeakReference;

/**
 * A ViewPager subclass enabling infinte scrolling of the viewPager elements
 * When used for paginating views (in opposite to fragments), no code changes
 * should be needed only change xml's from <android.support.v4.view.ViewPager>
 * to <com.imbryk.viewPager.LoopViewPager> If "blinking" can be seen when
 * paginating to first or last view, simply call seBoundaryCaching( true ), or
 * change DEFAULT_BOUNDARY_CASHING to true When using a FragmentPagerAdapter or
 * FragmentStatePagerAdapter, additional changes in the adapter must be done.
 * The adapter must be prepared to create 2 extra items e.g.: The original
 * adapter creates 4 items: [0,1,2,3] The modified adapter will have to create 6
 * items [0,1,2,3,4,5] with mapping realPosition=(position-1)%count [0->3, 1->0,
 * 2->1, 3->2, 4->3, 5->0]
 */
public class LoopViewPager extends ViewPager {

    private static final String TAG = "LoopViewPager";
//    private static final boolean DEBUG = BuildConfig.DEBUG;
    private static final boolean DEBUG = false;

    private OnPageChangeListener mOuterPageChangeListener;
    private LoopPagerAdapter mAdapter;
    private AppCompatActivity mActivity;
    private MyLifeCycleObserver mMyLifeCycleObserver;

    /**
     * helper function which may be used when implementing FragmentPagerAdapter
     *
     * @param position
     * @param count
     * @return (position - 1)%count
     */
    public static int toRealPosition(int position, int count) {
        position = position - 1;
        if (position < 0) {
            position += count;
        } else {
            position = position % count;
        }
        return position;
    }

    @Override
    public void setAdapter(PagerAdapter adapter) {
        PagerAdapter adapterTmp = getAdapter();
        if (adapterTmp != null) {
            adapterTmp.unregisterDataSetObserver(dataSetObserver);
        }
        if (adapter instanceof LoopPagerAdapter) {
            mAdapter = (LoopPagerAdapter) adapter;
        } else {
            throw new IllegalArgumentException("the adapter must be LoopPagerAdapter");
        }
        super.setAdapter(mAdapter);
        setCurrentItem(0, false);
        if (mAdapter != null) {
            adapter.registerDataSetObserver(dataSetObserver);
        }
        onAdapterChanged();
    }

    @Override
    protected void onFinishInflate() {
        super.onFinishInflate();
        SensorsDataAPI.sharedInstance().ignoreView(this);
    }

    @Override
    public PagerAdapter getAdapter() {
        return mAdapter;
    }

    @Override
    public int getCurrentItem() {
        return mAdapter != null ? mAdapter.toRealPosition(super.getCurrentItem()) : 0;
    }

    public int getSuperCurrentItem() {
        return mAdapter != null ? super.getCurrentItem() : 0;
    }

    public void setCurrentItem(int item, boolean smoothScroll) {
        int realItem = mAdapter.toInnerPosition(item);
        super.setCurrentItem(realItem, smoothScroll);
    }

    @Override
    public void setCurrentItem(int item) {
        if (getCurrentItem() != item) {
            setCurrentItem(item, true);
        }
    }

    @Override
    public void setOnPageChangeListener(OnPageChangeListener listener) {
        mOuterPageChangeListener = listener;
    }

    public LoopViewPager(Context context) {
        this(context, null);
    }

    public LoopViewPager(Context context, AttributeSet attrs) {
        super(context, attrs);
        if (context != null && context instanceof AppCompatActivity) {
            mActivity = (AppCompatActivity) context;
        }
        init();
    }

    private void init() {
        mHandler = new UIHandler(this);
        super.setOnPageChangeListener(onPageChangeListener);
    }

    @Override
    protected void onPageScrolled(int position, float offset, int offsetPixels) {
        int realPosition = position;
        if (mAdapter != null) {
            realPosition = mAdapter.toRealPosition(position);
        }
        super.onPageScrolled(realPosition, offset, offsetPixels);
    }

    private OnPageChangeListener onPageChangeListener = new OnPageChangeListener() {
        private float mPreviousOffset = -1;
        private float mPreviousPosition = -1;

        @Override
        public void onPageSelected(int position) {
            int realPosition = mAdapter.toRealPosition(position);
            if (DEBUG) {
                Log.i(TAG, "onPageSelected: " + realPosition + " position=" + position);
            }
            mCurrentPageIndex = realPosition;
            if (mPreviousPosition != realPosition) {
                mPreviousPosition = realPosition;
                if (mOuterPageChangeListener != null) {
                    mOuterPageChangeListener.onPageSelected(realPosition);
                }
            }
        }

        @Override
        public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {
            int realPosition = position;
            if (DEBUG) {
                Log.i(TAG, "onPageScrolled: " + realPosition);
            }

            // if (mAdapter != null) {
            //     realPosition = mAdapter.toRealPosition(position);
            // }

            mPreviousOffset = positionOffset;
            if (mOuterPageChangeListener != null) {
                if (realPosition != mAdapter.getRealCount() - 1) {
                    mOuterPageChangeListener.onPageScrolled(realPosition, positionOffset,
                            positionOffsetPixels);
                } else {
                    if (positionOffset > .5) {
                        mOuterPageChangeListener.onPageScrolled(0, 0, 0);
                    } else {
                        mOuterPageChangeListener.onPageScrolled(realPosition, 0, 0);
                    }
                }
            }
        }

        @Override
        public void onPageScrollStateChanged(int state) {
            if (mAdapter != null) {
                int position = LoopViewPager.super.getCurrentItem();
                int realPosition = mAdapter.toRealPosition(position);
                if (DEBUG) {
                    Log.i(TAG,
                            "onPageScrollStateChanged: " + position + "/" + realPosition + "/" +
                                    state);
                }
                if (state == ViewPager.SCROLL_STATE_IDLE
                        && (position == 0 || position == mAdapter.getCount() - 1)) {
                    if (DEBUG) {
                        Log.i(TAG, "onPageScrollStateChanged: setCurrentItem: " + realPosition);
                    }
                    setCurrentItem(realPosition, false);
                }
            }

            switch (state) {
                case SCROLL_STATE_IDLE:
                    if (mDraged) {
                        YLog.debug(this, "SCROLL_STATE_IDLE send msg");
                        mHandler.removeMessages(MSG_CHANGE);
                        mHandler.sendEmptyMessageDelayed(MSG_CHANGE, mChangeDelay);
                        mDraged = false;
                    }
                    break;
                case SCROLL_STATE_SETTLING:
                    if (mDraged) {
                        YLog.debug(this, "SCROLL_STATE_SETTLING remove msg");
                        mHandler.removeMessages(MSG_CHANGE);
                    }
                    break;
                case SCROLL_STATE_DRAGGING:
                    // 手动滚动了
                    mDraged = true;
                    YLog.debug(this, "SCROLL_STATE_DRAGGING");
                    break;
                default:
                    break;
            }

            if (mOuterPageChangeListener != null) {
                mOuterPageChangeListener.onPageScrollStateChanged(state);
            }
        }
    };

    private static final int MSG_CHANGE = 1;

    private int mChangeDelay = 5000;// 5s

    private int mCurrentPageIndex = 0;
    private int mMaxPages = 0;

    private boolean mDraged = false;

    private UIHandler mHandler;

    /**
     * 设置轮播时间间隔
     *
     * @param changeDelay 时间间隔，ms
     */
    public void setChangeDelay(int changeDelay) {
        mChangeDelay = changeDelay;
    }

    private static class UIHandler extends Handler {
        private WeakReference<LoopViewPager> mReference;

        public UIHandler(LoopViewPager loopViewPager) {
            mReference = new WeakReference<LoopViewPager>(loopViewPager);
        }

        @Override
        public void handleMessage(Message msg) {
            LoopViewPager loopViewPager = mReference.get();
            if (loopViewPager == null) {
                return;
            }
            switch (msg.what) {
                case MSG_CHANGE:
                    if (loopViewPager.mMaxPages > 0 &&
                            loopViewPager.mCurrentPageIndex < loopViewPager.mMaxPages) {
                        try {
                            loopViewPager.setCurrentItem(loopViewPager.mCurrentPageIndex);
                        } catch (Exception e) {
                            YLog.error(this, e);
                        }
                        loopViewPager.mCurrentPageIndex++;
                        if (loopViewPager.mCurrentPageIndex < 0 ||
                                loopViewPager.mCurrentPageIndex >= loopViewPager.mMaxPages) {
                            loopViewPager.mCurrentPageIndex = 0;
                        }
                        removeMessages(MSG_CHANGE);
                        sendEmptyMessageDelayed(MSG_CHANGE, loopViewPager.mChangeDelay);
                    }
                    break;
                default:
                    super.handleMessage(msg);
                    break;
            }
        }
    }

    private void onAdapterChanged() {
        mHandler.removeMessages(MSG_CHANGE);
        if (mAdapter != null && mAdapter.getCount() > 0) {
            mCurrentPageIndex = 0;
            mMaxPages = mAdapter.getRealCount();
            if (mMaxPages > 1) {
                mHandler.removeMessages(MSG_CHANGE);
                mHandler.sendEmptyMessageDelayed(MSG_CHANGE, mChangeDelay);
            }
        } else {
            mCurrentPageIndex = 0;
            mMaxPages = 0;
        }
    }

    public void startRoll(boolean isStartRoll) {
        if (mAdapter != null && mAdapter.getCount() > 0) {
            mMaxPages = mAdapter.getRealCount();
            if (mMaxPages > 1 && mHandler != null) {
                mHandler.removeMessages(MSG_CHANGE);
                if (isStartRoll) {
                    mHandler.sendEmptyMessageDelayed(MSG_CHANGE, mChangeDelay);
                }
            }
        }
    }

    private DataSetObserver dataSetObserver = new DataSetObserver() {
        @Override
        public void onChanged() {
            super.onChanged();
            onAdapterChanged();
        }

        @Override
        public void onInvalidated() {
            super.onInvalidated();
        }
    };

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        YLog.debug(this, "onDetachedFromWindow");
        mHandler.removeMessages(MSG_CHANGE);
        if (mActivity != null && mMyLifeCycleObserver != null) {
            mActivity.getLifecycle().removeObserver(mMyLifeCycleObserver);
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        YLog.debug(this, "onAttachedToWindow");
        mHandler.removeMessages(MSG_CHANGE);
        mHandler.sendEmptyMessageDelayed(MSG_CHANGE, mChangeDelay);
        if (mActivity != null) {
            mMyLifeCycleObserver = new MyLifeCycleObserver() {
                @Override
                public void onActivityResume() {
                    startRoll(true);
                }

                @Override
                public void onActivityPause() {
                    startRoll(false);
                }
            };
            mActivity.getLifecycle().addObserver(mMyLifeCycleObserver);
        }
    }

    @Override
    public boolean onInterceptHoverEvent(MotionEvent event) {
        try {
            return super.onInterceptHoverEvent(event);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }
}
