/*
 * Copyright (C) 2013 Leszek Mzyk
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.edu24ol.newclass.widget.viewpager;

import android.content.Context;
import android.view.View;
import android.view.ViewGroup;

import androidx.viewpager.widget.PagerAdapter;

import java.util.ArrayList;
import java.util.List;

/**
 * A PagerAdapter wrapper responsible for providing a proper page to
 * LoopViewPager;
 */
public abstract class LoopPagerAdapter<T> extends PagerAdapter {

    private List<T> redirectItems = new ArrayList<>();
    private Context mContext;
    private View.OnClickListener mClickListener;

    public LoopPagerAdapter(Context context,
                            View.OnClickListener clickListener) {
        mContext = context;
        mClickListener = clickListener;
    }

    public LoopPagerAdapter(Context context, List<T> itemList,
                            View.OnClickListener clickListener) {
        mContext = context;
        this.redirectItems.addAll(itemList);
        mClickListener = clickListener;
    }

    public LoopPagerAdapter() {
    }

    public int toRealPosition(int position) {
        int realCount = getRealCount();
        if (realCount == 0) {
            return 0;
        }
        int realPosition = (position - 1) % realCount;
        if (realPosition < 0) {
            realPosition += realCount;
        }

        return realPosition;
    }

    public void addData(List<T> redirectItems) {
        this.redirectItems.addAll(redirectItems);
    }

    public void setData(List<T> redirectItems) {
        this.redirectItems.clear();
        addData(redirectItems);
    }

    public int toInnerPosition(int realPosition) {
        int position = (realPosition + 1);
        return position;
    }

    private int getRealFirstPosition() {
        return 1;
    }

    private int getRealLastPosition() {
        return getRealFirstPosition() + getRealCount() - 1;
    }

    @Override
    public int getCount() {
        return redirectItems == null ? 0 : redirectItems.size();
    }

    public int getRealCount() {
        return getCount() - 2;
    }

    @Override
    public Object instantiateItem(ViewGroup container, int position) {
        T t = redirectItems.get(position);
        return onInstantiateItem(container, position, t);
    }

    public T getDataFromPosition(int position) {
        if (redirectItems != null && position < redirectItems.size()) {
            return redirectItems.get(position);
        }
        return null;
    }

    protected abstract Object onInstantiateItem(ViewGroup container, int position, T t);

    @Override
    public void destroyItem(ViewGroup container, int position, Object object) {
        View view = (View) object;
        if (view != null && view.getParent() != null) {
            container.removeView(view);
        }
    }

    @Override
    public boolean isViewFromObject(View view, Object o) {
        return view == o;
    }

    @Override
    public int getItemPosition(Object object) {
        return POSITION_NONE;
    }
}
