package com.edu24ol.newclass.widget.viewpager;

/**
 * Created by zhanghong on 2019/3/15.
 */

import android.content.Context;
import android.database.DataSetObserver;
import android.util.AttributeSet;
import android.view.MotionEvent;

import androidx.viewpager.widget.PagerAdapter;
import androidx.viewpager.widget.ViewPager;

import com.yy.android.educommon.log.YLog;

/**
 * A ViewPager subclass enabling infinte scrolling of the viewPager elements
 * When used for paginating views (in opposite to fragments), no code changes
 * should be needed only change xml's from <android.support.v4.view.ViewPager>
 * to <com.imbryk.viewPager.LoopViewPager> If "blinking" can be seen when
 * paginating to first or last view, simply call seBoundaryCaching( true ), or
 * change DEFAULT_BOUNDARY_CASHING to true When using a FragmentPagerAdapter or
 * FragmentStatePagerAdapter, additional changes in the adapter must be done.
 * The adapter must be prepared to create 2 extra items e.g.: The original
 * adapter creates 4 items: [0,1,2,3] The modified adapter will have to create 6
 * items [0,1,2,3,4,5] with mapping realPosition=(position-1)%count [0->3, 1->0,
 * 2->1, 3->2, 4->3, 5->0]
 */
public class DragLoopViewPager extends ViewPager {

    private OnPageChangeListener mOuterPageChangeListener;
    private LoopPagerAdapter mAdapter;

    /**
     * helper function which may be used when implementing FragmentPagerAdapter
     *
     * @param position
     * @param count
     * @return (position - 1)%count
     */
    public static int toRealPosition(int position, int count) {
        position = position - 1;
        if (position < 0) {
            position += count;
        } else {
            position = position % count;
        }
        return position;
    }

    @Override
    public void setAdapter(PagerAdapter adapter) {
        PagerAdapter adapterTmp = getAdapter();
        if (adapterTmp != null) {
            adapterTmp.unregisterDataSetObserver(dataSetObserver);
        }
        if (adapter instanceof LoopPagerAdapter) {
            mAdapter = (LoopPagerAdapter) adapter;
        } else {
            throw new IllegalArgumentException("the adapter must be LoopPagerAdapter");
        }
        super.setAdapter(mAdapter);
        setCurrentItem(0, false);
        if (mAdapter != null) {
            adapter.registerDataSetObserver(dataSetObserver);
        }
        onAdapterChanged();
    }

    @Override
    public PagerAdapter getAdapter() {
        return mAdapter;
    }

    @Override
    public int getCurrentItem() {
        return mAdapter != null ? mAdapter.toRealPosition(super.getCurrentItem()) : 0;
    }

    public void setCurrentItem(int item, boolean smoothScroll) {
        int realItem = mAdapter.toInnerPosition(item);
        super.setCurrentItem(realItem, smoothScroll);
    }

    @Override
    public void setCurrentItem(int item) {
        if (getCurrentItem() != item) {
            setCurrentItem(item, true);
        }
    }

    @Override
    public void setOnPageChangeListener(OnPageChangeListener listener) {
        mOuterPageChangeListener = listener;
    }

    public DragLoopViewPager(Context context) {
        this(context, null);
    }

    public DragLoopViewPager(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    private void init() {
        super.setOnPageChangeListener(onPageChangeListener);
    }

    private OnPageChangeListener onPageChangeListener = new OnPageChangeListener() {
        private float mPreviousOffset = -1;
        private float mPreviousPosition = -1;

        @Override
        public void onPageSelected(int position) {
            int realPosition = mAdapter.toRealPosition(position);
            mCurrentPageIndex = realPosition;
            if (mPreviousPosition != realPosition) {
                mPreviousPosition = realPosition;
                if (mOuterPageChangeListener != null) {
                    mOuterPageChangeListener.onPageSelected(realPosition);
                }
            }
        }

        @Override
        public void onPageScrolled(int position, float positionOffset, int positionOffsetPixels) {
            int realPosition = position;
            if (mAdapter != null) {
                realPosition = mAdapter.toRealPosition(position);

                // if (positionOffset == 0 && mPreviousOffset == 0
                // && (position == 0 || position == mAdapter.getCount() - 1)) {
                // setCurrentItem(realPosition, false);
                // }
            }

            mPreviousOffset = positionOffset;
            if (mOuterPageChangeListener != null) {
                if (realPosition != mAdapter.getRealCount() - 1) {
                    mOuterPageChangeListener.onPageScrolled(realPosition, positionOffset,
                            positionOffsetPixels);
                } else {
                    if (positionOffset > .5) {
                        mOuterPageChangeListener.onPageScrolled(0, 0, 0);
                    } else {
                        mOuterPageChangeListener.onPageScrolled(realPosition, 0, 0);
                    }
                }
            }
        }

        @Override
        public void onPageScrollStateChanged(int state) {
            if (mAdapter != null) {
                int position = DragLoopViewPager.super.getCurrentItem();
                int realPosition = mAdapter.toRealPosition(position);
                if (state == ViewPager.SCROLL_STATE_IDLE
                        && (position == 0 || position == mAdapter.getCount() - 1)) {
                    setCurrentItem(realPosition, false);
                }
            }

            switch (state) {
                case SCROLL_STATE_IDLE:
                    if (mDraged) {
                        YLog.debug(this, "SCROLL_STATE_IDLE send msg");
                        mDraged = false;
                    }
                    break;
                case SCROLL_STATE_SETTLING:
                    if (mDraged) {
                        YLog.debug(this, "SCROLL_STATE_SETTLING remove msg");
                    }
                    break;
                case SCROLL_STATE_DRAGGING:
                    // 手动滚动了
                    mDraged = true;
                    YLog.debug(this, "SCROLL_STATE_DRAGGING");
                    break;
                default:
                    break;
            }

            if (mOuterPageChangeListener != null) {
                mOuterPageChangeListener.onPageScrollStateChanged(state);
            }
        }
    };

    private int mCurrentPageIndex = 0;
    private int mMaxPages = 0;

    private boolean mDraged = false;

    private void onAdapterChanged() {
        if (mAdapter != null && mAdapter.getCount() > 0) {
            mCurrentPageIndex = 0;
            mMaxPages = mAdapter.getRealCount();
        } else {
            mCurrentPageIndex = 0;
            mMaxPages = 0;
        }
    }

    private DataSetObserver dataSetObserver = new DataSetObserver() {
        @Override
        public void onChanged() {
            super.onChanged();
            onAdapterChanged();
        }

        @Override
        public void onInvalidated() {
            super.onInvalidated();
        }
    };

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();
        YLog.debug(this, "onDetachedFromWindow");
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();
        YLog.debug(this, "onAttachedToWindow");
    }

    @Override
    public boolean onInterceptHoverEvent(MotionEvent event) {
        try {
            return super.onInterceptHoverEvent(event);
        } catch (Exception e) {
            e.printStackTrace();
            return false;
        }
    }
}
