package com.edu24ol.newclass.discover.widget.edittext;

import android.content.Context;
import android.text.Editable;
import android.text.Spannable;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.LongSparseArray;
import android.view.KeyEvent;
import android.view.inputmethod.EditorInfo;
import android.view.inputmethod.InputConnection;
import android.view.inputmethod.InputConnectionWrapper;
import android.widget.EditText;

import com.edu24.data.server.discover.entity.DiscoverDynamicElement;
import com.edu24.data.server.discover.entity.DiscoverSimpleTopic;
import com.edu24.data.server.discover.entity.DiscoverUser;
import com.edu24.data.server.discover.entity.Range;
import com.edu24ol.newclass.discover.text.AtUserClickableSpan;
import com.edu24ol.newclass.discover.text.TopicClickableSpan;
import com.hqwx.android.platform.utils.ToastUtil;
import com.hqwx.android.platform.widgets.text.editText.MaxCountLimitEditText;

import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 发布动态界面的EditText，有插入和删除话题、用户的功能
 */
public class DynamicEditText extends MaxCountLimitEditText {

    private static class DynamicElementsInfo {
        LongSparseArray<Pattern> mPattern = new LongSparseArray<>();
        LongSparseArray<Range> mRangeList = new LongSparseArray<>();
        LongSparseArray<DiscoverDynamicElement> mElements = new LongSparseArray();

        public void removeElement(long elementId) {
            mRangeList.remove(elementId);
            mPattern.remove(elementId);
            mElements.remove(elementId);
        }
    }

    private DynamicElementsInfo mTopicElementsInfo = new DynamicElementsInfo();
    private DynamicElementsInfo mUserElementsInfo = new DynamicElementsInfo();

    private boolean mIsSelected;
    private Range mLastSelectedRange;




    public DynamicEditText(Context context) {
        this(context, null);
    }

    public DynamicEditText(Context context, AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public DynamicEditText(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
    }

    public void setText(final String text, List<DiscoverSimpleTopic> topics,
                        List<DiscoverUser> atUsers) {
        if (topics != null && topics.size() > 0) {
            for (DiscoverSimpleTopic topic : topics) {
                addToElementsList(text, topic, mTopicElementsInfo);
            }
        }
        if (atUsers != null && atUsers.size() > 0) {
            for (DiscoverUser atUser : atUsers) {
                addToElementsList(text, atUser, mUserElementsInfo);
            }
        }
        setText(text);
    }

    private Range getElementRangeInText(String text, Pattern elementPattern) {
        Matcher matcher = elementPattern.matcher(text);
        if (matcher.find()) {
            String mentionText = matcher.group();
            int start = text.indexOf(mentionText);
            int end = start + mentionText.length();
            Range range = new Range(start, end);
            return range;
        }
        return null;
    }

    public void addTopics(List<DiscoverSimpleTopic> topics) {
        for (DiscoverSimpleTopic topic : topics) {
            addTopic(topic);
        }
    }

    /**
     * 把编辑元素添加到相应的元素集合中（话题、用户、图片均为内容的一个元素）
     *
     * @param element
     * @param targetDynamicElementsInfo
     */
    public void addElement(DiscoverDynamicElement element,
                           DynamicElementsInfo targetDynamicElementsInfo) {
        String insertText = element.getInsertText();
        if (getText().toString().length() - 1 + insertText.length() > getMaxLength()) {
            ToastUtil.showMessage(getContext(), "字数超出，添加失败!");
            return;
        }
        addToElementsList(element, targetDynamicElementsInfo);
        addElementText(element, targetDynamicElementsInfo);
    }

    private void addElementText(DiscoverDynamicElement element,
                                DynamicElementsInfo targetDynamicElementsInfo) {
        String insertText = element.getInsertText();
        int selectionStart = getSelectionStart();
        String preTextBeforeSelectionPosition = selectionStart > 0 ?
                getText().toString().substring(selectionStart - 1, selectionStart) : "";
        int start, end;
        if (selectionStart > 0 &&
                preTextBeforeSelectionPosition.equals(element.getBeginPatternSign())) {
            start = selectionStart - 1;
            end = start + insertText.length();
            //should add Range before text changed, replace method will call textChange method
            Range range = new Range(start, end);
            targetDynamicElementsInfo.mRangeList.put(element.getId(), range);
            getText().replace(selectionStart - 1, selectionStart, insertText);
        } else {
            start = selectionStart;
            end = start + insertText.length();
            Range range = new Range(start, end);
            targetDynamicElementsInfo.mRangeList.put(element.getId(), range);
            getText().insert(selectionStart, insertText);
        }
    }

    private void addToElementsList(String text, DiscoverDynamicElement element,
                                   DynamicElementsInfo dynamicElementsInfo) {
        addToElementsList(element, dynamicElementsInfo);
        Range range = getElementRangeInText(text,
                dynamicElementsInfo.mPattern.get(element.getId()));
        if (range != null) {
            dynamicElementsInfo.mRangeList.put(element.getId(), range);
        }
    }

    private void addToElementsList(DiscoverDynamicElement element,
                                   DynamicElementsInfo dynamicElementsInfo) {
        Pattern pattern = Pattern.compile(element.getPatternString());
        dynamicElementsInfo.mElements.put(element.getId(), element);
        dynamicElementsInfo.mPattern.put(element.getId(), pattern);
    }

    public void addTopic(DiscoverSimpleTopic topic) {
        addElement(topic, mTopicElementsInfo);
    }

    public void addATUsers(List<DiscoverUser> atUsers) {
        for (DiscoverUser ATUser : atUsers) {
            addATUser(ATUser);
        }
    }

    public void addATUser(DiscoverUser atUser) {
        addElement(atUser, mUserElementsInfo);
    }

    public LongSparseArray<? extends DiscoverDynamicElement> getTopics() {
        return mTopicElementsInfo.mElements;
    }

    public LongSparseArray<? extends DiscoverDynamicElement> getATUsers() {
        return mUserElementsInfo.mElements;
    }

    @Override
    protected void onSelectionChanged(int selStart, int selEnd) {
        super.onSelectionChanged(selStart, selEnd);
        if (mLastSelectedRange != null && mLastSelectedRange.isEqual(selStart, selEnd)) {
            return;
        }
        //if user cancel a selection of mention string, reset the state of 'mIsSelected'
        Range closestRange = getRangeOfClosestMentionString(selStart, selEnd);
        if (closestRange != null && closestRange.to == selEnd) {
            mIsSelected = false;
        }

        Range nearbyRange = getRangeOfNearbyMentionString(selStart, selEnd);
        //if there is no mention string nearby the cursor, just skip
        if (nearbyRange == null) {
            return;
        }
        //forbid cursor located in the mention string.
        if (selStart == selEnd) {
            setSelection(Math.min(getText().length(), nearbyRange.getAnchorPosition(selStart)));
        } else {
            if (selEnd < nearbyRange.to) {
                setSelection(selStart, nearbyRange.to);
            }
            if (selStart > nearbyRange.from) {
                setSelection(nearbyRange.from, selEnd);
            }
        }
    }

    @Override
    public void onTextChanged(CharSequence charSequence, int start, int before, int count) {
        setupTopicAndUserHighlight();
        if (count == 1 && !TextUtils.isEmpty(charSequence)) {
            char mentionChar = charSequence.toString().charAt(start);
            if ("@".equals(String.valueOf(mentionChar))) {
                if (mEventListener != null) {
                    mEventListener.onTextInputATUser();
                }
            } else if ("#".equals(String.valueOf(mentionChar))) {
                if (mEventListener != null) {
                    mEventListener.onTextInputTopic();
                }
            }
        }
    }

    private void setupTopicAndUserHighlight() {
        if (mTopicElementsInfo == null || mUserElementsInfo == null) {
            return;
        }
        //reset state
        mIsSelected = false;
        Editable spannableText = getText();
        if (spannableText == null || TextUtils.isEmpty(spannableText.toString())) {
            return;
        }

        //remove previous spans(清除掉所有的span)
        TopicClickableSpan[] oldSpans =
                spannableText.getSpans(0, spannableText.length(), TopicClickableSpan.class);
        for (TopicClickableSpan oldSpan : oldSpans) {
            spannableText.removeSpan(oldSpan);
        }
        AtUserClickableSpan[] atUserClickableSpans =
                spannableText.getSpans(0, spannableText.length(), AtUserClickableSpan.class);
        for (AtUserClickableSpan oldSpan : atUserClickableSpans) {
            spannableText.removeSpan(oldSpan);
        }

        //find mention string and color it (找到高亮的地方，高亮并记录它的位置)
        String text = spannableText.toString();
        mTopicElementsInfo.mRangeList.clear();
        for (int i = 0; i < mTopicElementsInfo.mPattern.size(); i++) {
            long topicId = mTopicElementsInfo.mPattern.keyAt(i);
            Pattern pattern = mTopicElementsInfo.mPattern.valueAt(i);
            Matcher matcher = pattern.matcher(text);
            int lastMentionIndex = -1;
            while (matcher.find()) {
                String mentionText = matcher.group();
                int start;
                if (lastMentionIndex != -1) {
                    start = text.indexOf(mentionText, lastMentionIndex);
                } else {
                    start = text.indexOf(mentionText);
                }
                int end = start + mentionText.length();
                spannableText.setSpan(new TopicClickableSpan(topicId), start, end,
                        Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
                lastMentionIndex = end;
                //record all mention-string's position
                mTopicElementsInfo.mRangeList.put(topicId, new Range(start, end));
            }
        }

        for (int i = 0; i < mUserElementsInfo.mPattern.size(); i++) {
            long userId = mUserElementsInfo.mPattern.keyAt(i);
            Pattern pattern = mUserElementsInfo.mPattern.valueAt(i);
            Matcher matcher = pattern.matcher(text);
            int lastMentionIndex = -1;
            while (matcher.find()) {
                String mentionText = matcher.group();
                int start;
                if (lastMentionIndex != -1) {
                    start = text.indexOf(mentionText, lastMentionIndex);
                } else {
                    start = text.indexOf(mentionText);
                }
                int end = start + mentionText.length();
                spannableText.setSpan(new AtUserClickableSpan(userId), start, end,
                        Spannable.SPAN_EXCLUSIVE_EXCLUSIVE);
                lastMentionIndex = end;
                //record all mention-string's position
                mUserElementsInfo.mRangeList.put(userId, new Range(start, end));
            }
        }
    }

    @Override
    public InputConnection onCreateInputConnection(EditorInfo outAttrs) {
        mInputConnection = new HackInputConnection(super.onCreateInputConnection(outAttrs), true,
                this);
        return mInputConnection;
    }

    private InputConnectionWrapper mInputConnection;

    public InputConnectionWrapper getInputConnection() {
        return mInputConnection;
    }

    private class HackInputConnection extends InputConnectionWrapper {
        private EditText editText;

        HackInputConnection(InputConnection target, boolean mutable, DynamicEditText editText) {
            super(target, mutable);
            this.editText = editText;
        }

        @Override
        public boolean sendKeyEvent(KeyEvent event) {
            if (event.getAction() == KeyEvent.ACTION_DOWN &&
                    event.getKeyCode() == KeyEvent.KEYCODE_DEL) {
                int selectionStart = editText.getSelectionStart();
                int selectionEnd = editText.getSelectionEnd();
                Range closestRange = getRangeOfClosestMentionString(selectionStart, selectionEnd);
                if (closestRange == null) {
                    mIsSelected = false;
                    //delete selected topic or user
                    return super.sendKeyEvent(event);
                }
                //if mention string has been selected or the cursor is at the beginning of mention string, just use default action(delete)
                if (mIsSelected || selectionStart == closestRange.from) {
                    mIsSelected = false;
                    removeElementByRange(closestRange);
                    return super.sendKeyEvent(event);
                } else {
                    //select the mention string
                    mIsSelected = true;
                    mLastSelectedRange = closestRange;
                    setSelection(closestRange.to, closestRange.from);
                }
                return true;
            }
            return super.sendKeyEvent(event);
        }

        @Override
        public boolean deleteSurroundingText(int beforeLength, int afterLength) {
            if (beforeLength == 1 && afterLength == 0) {
                return sendKeyEvent(new KeyEvent(KeyEvent.ACTION_DOWN, KeyEvent.KEYCODE_DEL))
                        && sendKeyEvent(new KeyEvent(KeyEvent.ACTION_UP, KeyEvent.KEYCODE_DEL));
            }
            return super.deleteSurroundingText(beforeLength, afterLength);
        }
    }

    private void removeElementByRange(Range target) {
        if ((mTopicElementsInfo == null ||
                mTopicElementsInfo.mRangeList.size() == 0) && (mUserElementsInfo == null ||
                mUserElementsInfo.mRangeList.size() == 0)) {
            return;
        }

        for (int i = 0; i < mTopicElementsInfo.mRangeList.size(); i++) {
            Range range = mTopicElementsInfo.mRangeList.valueAt(i);
            long topicId = mTopicElementsInfo.mRangeList.keyAt(i);
            if (range == target) {
                mTopicElementsInfo.removeElement(topicId);
                return;
            }
        }
        for (int i = 0; i < mUserElementsInfo.mRangeList.size(); i++) {
            Range range = mUserElementsInfo.mRangeList.valueAt(i);
            long userId = mUserElementsInfo.mRangeList.keyAt(i);
            if (range == target) {
                mUserElementsInfo.removeElement(userId);
                return;
            }
        }
    }

    private Range getRangeOfClosestMentionString(int selStart, int selEnd) {
        if ((mTopicElementsInfo == null ||
                mTopicElementsInfo.mRangeList.size() == 0) && (mUserElementsInfo == null ||
                mUserElementsInfo.mRangeList.size() == 0)) {
            return null;
        }
        int start = Math.min(selStart, selEnd);
        int end = Math.max(selStart, selEnd);
        for (int i = 0; i < mTopicElementsInfo.mRangeList.size(); i++) {
            Range range = mTopicElementsInfo.mRangeList.valueAt(i);
            if (range.contains(start, end)) {
                return range;
            }
        }

        for (int i = 0; i < mUserElementsInfo.mRangeList.size(); i++) {
            Range range = mUserElementsInfo.mRangeList.valueAt(i);
            if (range.contains(start, end)) {
                return range;
            }
        }
        return null;
    }

    private Range getRangeOfNearbyMentionString(int selStart, int selEnd) {
        if ((mTopicElementsInfo == null ||
                mTopicElementsInfo.mRangeList.size() == 0) && (mUserElementsInfo == null ||
                mUserElementsInfo.mRangeList.size() == 0)) {
            return null;
        }
        int start = Math.min(selStart, selEnd);
        int end = Math.max(selStart, selEnd);
        for (int i = 0; i < mTopicElementsInfo.mRangeList.size(); i++) {
            Range range = mTopicElementsInfo.mRangeList.valueAt(i);
            if (range.isWrappedBy(start, end)) {
                return range;
            }
        }
        for (int i = 0; i < mUserElementsInfo.mRangeList.size(); i++) {
            Range range = mUserElementsInfo.mRangeList.valueAt(i);
            if (range.isWrappedBy(start, end)) {
                return range;
            }
        }
        return null;
    }

    private EventListener mEventListener;

    public void setEventListener(EventListener eventListener) {
        mEventListener = eventListener;
    }

    public interface EventListener {
        /**
         * 输入@
         */
        void onTextInputATUser();

        /**
         * 输入#
         */
        void onTextInputTopic();
    }
}
