package com.edu24ol.newclass.discover.widget.article;

import android.content.Context;
import android.text.TextUtils;
import android.util.AttributeSet;
import android.util.TypedValue;
import android.view.LayoutInflater;
import android.view.View;
import android.widget.CheckBox;
import android.widget.ImageView;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.constraintlayout.widget.ConstraintLayout;

import com.bumptech.glide.Glide;
import com.edu24.data.server.discover.entity.ArticleInfo;
import com.hqwx.android.discover.common.R;
import com.hqwx.android.platform.utils.DisplayUtils;
import com.hqwx.android.platform.widgets.CircleImageView;
import com.hqwx.android.service.difference.DifferenceConfig;
import com.yy.android.educommon.utils.TimeUtils;

import org.jetbrains.annotations.NotNull;

/**
 * Time:2022/3/17 11:46
 * Author:
 * Description: 文章头部View
 */
public class DiscoverArticleHeaderView extends ConstraintLayout {

    protected View mBaseHeaderRootLayoutView;
    protected CircleImageView mBaseHeaderAuthorHeaderView;
    protected TextView mBaseHeaderAuthorHeaderNameView;
    protected TextView mBaseHeaderAuthorTimeDescView;
    protected ImageView mVIcon;

    protected CheckBox mAttentionCb;
    protected ImageView mMoreIcon;

    private boolean isShowAttentionView = true;
    private boolean enableCancelAttention = false;

    public void setEnableCancelAttention(boolean enableCancelAttention) {
        this.enableCancelAttention = enableCancelAttention;
    }

    public DiscoverArticleHeaderView(
            @NonNull @NotNull Context context) {
        this(context, null);
    }

    public DiscoverArticleHeaderView(@NonNull @NotNull Context context,
                                     @Nullable @org.jetbrains.annotations.Nullable
                                             AttributeSet attrs) {
        this(context, attrs, 0);
    }

    public DiscoverArticleHeaderView(@NonNull @NotNull Context context,
                                     @Nullable @org.jetbrains.annotations.Nullable
                                             AttributeSet attrs,
                                     int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        initView();
    }

    protected void initView() {
        View itemView =
                LayoutInflater.from(getContext())
                        .inflate(R.layout.discover_item_common_header_layout, this, true);
        mBaseHeaderRootLayoutView = itemView;
        mBaseHeaderAuthorHeaderView =
                itemView.findViewById(R.id.header_discover_publish_header_view);
        mBaseHeaderAuthorHeaderNameView =
                itemView.findViewById(R.id.header_discover_publish_name_view);
        mBaseHeaderAuthorTimeDescView =
                itemView.findViewById(R.id.header_discover_publish_time_desc_view);
        mMoreIcon = itemView.findViewById(R.id.icon_more);
        mVIcon = itemView.findViewById(R.id.icon_v);

        setClipToPadding(false);
        setClipChildren(false);
        mAttentionCb = itemView.findViewById(R.id.cb_header_discover_attention);
        mAttentionCb.setOnClickListener(v -> {
            if (mEventClickListener != null) {
                mEventClickListener
                        .onAttentionClick(mAttentionCb);
            }
        });

        mMoreIcon.setOnClickListener(v -> {
            if (mEventClickListener != null) {
                mEventClickListener.onMoreClick(v);
            }
        });
        setViewParams();
        if(DifferenceConfig.discoverDifferenceService.isHideDiscoverCellInteractive()){
            mMoreIcon.setVisibility(GONE);
            mAttentionCb.setVisibility(GONE);
        }
    }

    protected void setViewParams() {
        int paddingLeft = getContext().getResources()
                .getDimensionPixelSize(R.dimen.platform_common_margin_size);
        int paddingTop = DisplayUtils.dip2px(getContext(), 16);
        int paddingBottom = DisplayUtils.dip2px(getContext(), 8);
        setPadding(paddingLeft, paddingTop, paddingLeft, paddingBottom);
    }

    private EventClickListener mEventClickListener;

    public void setEventClickListener(
            EventClickListener eventClickListener) {
        mEventClickListener = eventClickListener;
    }

    public interface EventClickListener {

        void onMoreClick(View moreView);

        void onAttentionClick(CheckBox checkBox);

    }

    public void refresh(ArticleInfo articleInfo) {
        if (articleInfo.author != null) {
            Glide.with(getContext()).load(articleInfo.author.pic)
                    .placeholder(R.mipmap.default_ic_avatar).dontAnimate()
                    .centerCrop().into(mBaseHeaderAuthorHeaderView);
            mBaseHeaderAuthorHeaderNameView.setText(articleInfo.author.name);
            if (articleInfo.author.isV()) {
                mVIcon.setVisibility(View.VISIBLE);
            } else {
                mVIcon.setVisibility(View.GONE);
            }
            mBaseHeaderAuthorTimeDescView.setMaxWidth(
                    DisplayUtils.getScreenWidth(getContext()) -
                            DisplayUtils.dip2px(getContext(), 60));
            String description = "";
            if (!TextUtils.isEmpty(articleInfo.author.description)) {
                description = " · " + articleInfo.author.description;
            }
            mBaseHeaderAuthorTimeDescView
                    .setText(TimeUtils.transformArticleTime(articleInfo.publishTime) + description);
        } else {
            mBaseHeaderAuthorTimeDescView
                    .setText(TimeUtils.transformArticleTime(articleInfo.publishTime));
            mBaseHeaderAuthorHeaderView.setImageResource(R.mipmap.default_ic_avatar);
            mBaseHeaderAuthorHeaderNameView.setText("");
        }
        refreshAttentionState(articleInfo);
    }

    public void refreshAttentionState(ArticleInfo articleInfo) {
        if(DifferenceConfig.discoverDifferenceService.isHideDiscoverCellInteractive()){
            isShowAttentionView = false;
        }
        if (isShowAttentionView) {
            if (articleInfo.author != null) {
                if (articleInfo.author.isAttendAuthor()) {
                    mAttentionCb.setText("已关注");
                    mAttentionCb.setSelected(true);
                    if (!enableCancelAttention) {
                        mAttentionCb.setVisibility(View.INVISIBLE);
                    }
                } else {
                    mAttentionCb.setText("+ 关注");
                    mAttentionCb
                            .setSelected(false);
                    mAttentionCb.setVisibility(View.VISIBLE);
                }
                mAttentionCb.setTag(articleInfo.author);
            }
        }
    }

    public void showAttentionView(boolean isShow) {
        isShowAttentionView = isShow;
        mAttentionCb.setVisibility(isShow ? View.VISIBLE : View.GONE);
    }

    public void showMoreView(boolean isShow) {
        if(DifferenceConfig.discoverDifferenceService.isHideDiscoverCellInteractive()){
            isShow = false;
        }

        mMoreIcon.setVisibility(isShow ? View.VISIBLE : View.GONE);
        int paddingLeft = getContext().getResources()
            .getDimensionPixelSize(R.dimen.platform_common_margin_size);
        if (isShow) {
            setPadding(paddingLeft, getPaddingTop(), paddingLeft, getPaddingBottom());
        } else {
            setPadding(paddingLeft, getPaddingTop(), DisplayUtils.dip2px(getContext(), 10),
                getPaddingBottom());
        }
    }

    /**
     * 作者名称大小
     */
    public void onSetAuthorNameTextSize(float size) {
        if (mBaseHeaderAuthorHeaderNameView != null) {
            mBaseHeaderAuthorHeaderNameView.setTextSize(TypedValue.COMPLEX_UNIT_SP, size);
        }
    }

    /**
     * 作者名称是否加粗
     */
    public void onSetAuthorNameTextStyle(boolean fakeBoldText) {
        if (mBaseHeaderAuthorHeaderNameView != null) {
            mBaseHeaderAuthorHeaderNameView.getPaint().setFakeBoldText(fakeBoldText);
        }
    }

    /**
     * 作者名称颜色
     */
    public void onSetAuthorNameTextColor(int color) {
        if (mBaseHeaderAuthorHeaderNameView != null) {
            mBaseHeaderAuthorHeaderNameView.setTextColor(color);
        }
    }

}
