package com.edu24ol.newclass.discover.util;

import android.content.Context;
import android.graphics.Rect;
import android.text.TextUtils;
import android.view.View;

import com.edu24.data.server.discover.entity.ArticleInfo;
import com.hqwx.android.discover.common.R;
import com.hqwx.android.platform.utils.DisplayUtils;
import com.hqwx.android.platform.utils.Network;
import com.hqwx.android.playercontroller.ListVideoItemView;
import com.hqwx.android.service.ServiceFactory;
import com.yy.android.educommon.log.YLog;

import org.jetbrains.annotations.NotNull;

import java.util.ArrayList;
import java.util.HashMap;

import androidx.constraintlayout.widget.Group;
import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;
import base.ListVideoPlayItem;
import base.VideoDefinition;

/**
 * 计算滑动，自动播放
 */

public class ScrollCalculatorHelper {

    private Context mContext;

    private ListVideoItemView mLastVideoItemView;
    private HashMap<Long, Long> videoPlayPositionCache = new HashMap<>();

    public ScrollCalculatorHelper(Context context) {
        mContext = context;
    }

    public void reset() {

    }

    public void onScrollStateChanged(RecyclerView view, int scrollState) {
        switch (scrollState) {
            case RecyclerView.SCROLL_STATE_IDLE:
                onScrollPlayVideo(view);
                break;
            default:
                break;
        }
    }

    public void onScrolled(RecyclerView view, int dx, int dy) {

    }

    /**
     * 一个屏幕内的视频item
     */
    ArrayList<ListVideoItemView> visibleVideoItemViewList = new ArrayList<>();

    public void onScrollPlayVideo(RecyclerView view) {
        if (view == null) {
            return;
        }
        visibleVideoItemViewList.clear();
        LinearLayoutManager layoutManager = (LinearLayoutManager) view.getLayoutManager();
        // int firstVisibleItem = layoutManager.findFirstVisibleItemPosition();
        // int lastVisibleItem = layoutManager.findLastVisibleItemPosition();
        // int visibleCount = lastVisibleItem - firstVisibleItem;
        //R.id.item_video_view
        ListVideoItemView videoItemView = null;
        ListVideoItemView lastVideoItemView = mLastVideoItemView;

        //layoutManager.getChildAt(i) 为屏幕可见的第i个Item
        int childCount = layoutManager.getChildCount();
        for (int i = 0; i < childCount; i++) {
            int playId = R.id.item_video_view;
            if (layoutManager.getChildAt(i) != null &&
                    layoutManager.getChildAt(i).findViewById(playId) != null) {
                ListVideoItemView currentVideoItemView =
                        layoutManager.getChildAt(i).findViewById(playId);
                //视频区域的View，如果在播放的话，是VideoView，如果没播放是默认图片
                View rangeTargetView;
                if (currentVideoItemView.getVisibility() == View.VISIBLE) {
                    rangeTargetView = currentVideoItemView;
                } else {
                    rangeTargetView = layoutManager.getChildAt(i)
                            .findViewById(R.id.item_discover_recommend_img);
                }
                //找出一个屏幕内可播放的视频item
                if (isViewInPlayRange(rangeTargetView)) {
                    visibleVideoItemViewList.add(currentVideoItemView);
                }
            }
        }

        if (visibleVideoItemViewList.size() > 1) {
            //列表中有多个视频，取接近屏幕中间的视频
            videoItemView = getCloseAtScreenCenterView(visibleVideoItemViewList);
        } else if (visibleVideoItemViewList.size() == 1) {
            //只有一个可播放的视频
            videoItemView = visibleVideoItemViewList.get(0);
        }

        if (videoItemView == null) {
            //当前屏幕没有视频可播放
            if (lastVideoItemView != null) {
                //需要暂停播放上一个播放的视频
                pauseVideo(lastVideoItemView);
                mLastVideoItemView = null;
            }
        } else {
            Network.Type networkType = Network.getNetworkType(mContext);
            if (networkType == Network.Type.WIFI && ServiceFactory.getAppService().isAutoPlayListVideo()) {
                playVideo(videoItemView);
                mLastVideoItemView = videoItemView;
                if (lastVideoItemView != null &&
                        lastVideoItemView.getPlayListItem() != videoItemView.getPlayListItem()) {
                    pauseVideo(lastVideoItemView);
                }
            }
        }
    }

    // public boolean isViewInPlayRange(View targetView) {
    //     //如果未播放，需要播放
    //     int[] screenPosition = new int[2];
    //     targetView.getLocationOnScreen(screenPosition);
    //     int halfHeight = targetView.getHeight() / 2;
    //     int rangePosition = screenPosition[1] + halfHeight;
    //     //中心点在播放区域内
    //     if (rangePosition >= rangeTop && rangePosition <= rangeBottom) {
    //         return true;
    //     } else {
    //         return false;
    //     }
    // }

    /**
     * 判断View是否在可自动播放区域内（视频一半内容以上可见）
     *
     * @param targetView
     * @return
     */
    public boolean isViewInPlayRange(View targetView) {
        int viewHeight = targetView.getHeight();
        if (viewHeight <= 0) {
            return false;
        }
        Rect rect = new Rect();
        targetView.getLocalVisibleRect(rect);
        int outOfHeight = targetView.getHeight() / 2;
        int height = targetView.getHeight();
        if (rect.top == 0 && rect.bottom >= outOfHeight) {
            return true;
        } else if (rect.top <= outOfHeight && rect.bottom == height) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 获取距离屏幕中心距离最近的视频View
     *
     * @param list
     * @param <T>
     * @return
     */
    public <T extends View> T getCloseAtScreenCenterView(ArrayList<T> list) {
        int screenCenterY = DisplayUtils.getScreenHeight(mContext) / 2;
        //View可见区域的中心位置，距离屏幕中心位置的最短距离
        int minDistance = screenCenterY;
        int minDistanceViewPosition = 0;
        for (int i = 0; i < list.size(); i++) {
            View targetView = list.get(i);
            Rect rect = new Rect();
            targetView.getGlobalVisibleRect(rect);
            int viewCenterY = rect.bottom - targetView.getHeight() / 2;
            if (viewCenterY <= minDistance) {
                minDistance = viewCenterY;
                minDistanceViewPosition = i;
            }
        }
        return list.get(minDistanceViewPosition);
    }

    /**
     * 判断视频是否在屏幕之外
     *
     * @param targetView
     * @return
     */
    public boolean isViewOutOfScreen(View targetView) {
        Rect rect = new Rect();
        targetView.getLocalVisibleRect(rect);
        int outOfHeight = targetView.getHeight() / 2;
        if (rect.top == 0 && rect.bottom < outOfHeight) {
            //view滑出屏幕底部自己高度的一半大小
            return true;
        } else if (rect.top > outOfHeight && rect.bottom == targetView.getHeight()) {
            //view滑出屏幕顶部自己高度的一半大小
            return true;
        } else {
            return false;
        }
    }

    /**
     * 播放某个列表视频
     *
     * @param videoItemView
     */
    private void playVideo(final ListVideoItemView videoItemView) {
        if (!videoItemView.isPlaying()) {
            videoItemView.postDelayed(new Runnable() {
                @Override
                public void run() {
                    setDefaultVideoImageViewVisible(videoItemView, false);
                    videoItemView.setVisibility(View.VISIBLE);
                    long lastPlayPosition = 0;
                    if (videoItemView.getTag() != null) {
                        ArticleInfo articleInfo = (ArticleInfo) videoItemView.getTag();
                        if (videoPlayPositionCache.containsKey(articleInfo.id)) {
                            lastPlayPosition = videoPlayPositionCache.get(articleInfo.id);
                        }
                        videoItemView.setPlayItem(getPlayListItem(articleInfo));
                        if (TextUtils.isEmpty(videoItemView.getPlayListItem().getPlayVideoUrl())) {
                            YLog.error("ListVideo",
                                    "url is empty!! " + articleInfo.id + " / " + articleInfo.title);
                            return;
                        }
                        videoItemView.setVideoEventListener(mVideoEventListener);
                        videoItemView.playVideo(lastPlayPosition);
                    }
                }
            }, 300);
        }
    }

    private ListVideoPlayItem getPlayListItem(ArticleInfo articleInfo) {
        ListVideoPlayItem playListItem = new ListVideoPlayItem();
        if (!TextUtils.isEmpty(articleInfo.getHdUrl())) {
            VideoDefinition videoDefinition =
                    new VideoDefinition(VideoDefinition.ULTRA, articleInfo.getHdUrl());
            playListItem.addSupportVideoDefinition(videoDefinition);
        }
        if (!TextUtils.isEmpty(articleInfo.getSdUrl())) {
            VideoDefinition videoDefinition =
                    new VideoDefinition(VideoDefinition.STANDARTD, articleInfo.getSdUrl());
            playListItem.addSupportVideoDefinition(videoDefinition);
        }
        if (!TextUtils.isEmpty(articleInfo.getMdUrl())) {
            VideoDefinition videoDefinition =
                    new VideoDefinition(VideoDefinition.HD, articleInfo.getMdUrl());
            playListItem.addSupportVideoDefinition(videoDefinition);
        }
        playListItem.setName("" + articleInfo.id);
        return playListItem;
    }

    /**
     * 暂停播放某个列表视频
     *
     * @param videoItemView
     */
    private void pauseVideo(@NotNull ListVideoItemView videoItemView) {
        if (videoItemView.isPlaying()) {
            videoItemView.pauseVideo();
            onVideoPlayStop(videoItemView);
        }
    }

    /**
     * 停止播放某个列表视频
     *
     * @param videoItemView
     */
    private void stopPlayVideo(ListVideoItemView videoItemView) {
        if (videoItemView.isPlaying()) {
            onVideoPlayStop(videoItemView);
            videoItemView.stopVideo();
        }
    }

    private void onVideoPlayStop(ListVideoItemView videoItemView) {
        setDefaultVideoImageViewVisible(videoItemView, true);
        videoItemView.setVisibility(View.GONE);
        long playPosition = videoItemView.getPlayPosition();
        if (videoItemView.getTag() != null) {
            ArticleInfo articleInfo = (ArticleInfo) videoItemView.getTag();
            videoPlayPositionCache.put(articleInfo.id, playPosition);
        }
    }

    private void setDefaultVideoImageViewVisible(ListVideoItemView videoItemView, boolean visible) {
        if (videoItemView.getParent() != null) {
            View parentView = (View) videoItemView.getParent();
            Group defaultVideoView = parentView.findViewById(R.id.default_video_view);
            if (visible) {
                defaultVideoView.setVisibility(View.VISIBLE);
            } else {
                defaultVideoView.setVisibility(View.GONE);
            }
        }
    }

    private ListVideoItemView.VideoEventListener mVideoEventListener =
            new ListVideoItemView.VideoEventListener() {
                @Override
                public void onCompletion(ListVideoItemView listVideoItemView) {
                    setDefaultVideoImageViewVisible(listVideoItemView, true);
                    listVideoItemView.setVisibility(View.GONE);
                    if (listVideoItemView.getTag() != null) {
                        ArticleInfo articleInfo = (ArticleInfo) listVideoItemView.getTag();
                        if (videoPlayPositionCache.containsKey(articleInfo.id)) {
                            videoPlayPositionCache.remove(articleInfo.id);
                        }
                    }
                }

                @Override
                public void onSurfaceDestroy(ListVideoItemView listVideoItemView) {
                    stopPlayVideo(listVideoItemView);
                }

                @Override
                public void onError(ListVideoItemView listVideoItemView) {
                    stopPlayVideo(listVideoItemView);
                }
            };

    public long getPlayPositionByArticleId(long id) {
        if (videoPlayPositionCache.containsKey(id)) {
            return videoPlayPositionCache.get(id);
        }
        return 0;
    }

    public void pauseVideo() {
        ListVideoItemView lastVideoItemView = mLastVideoItemView;
        if (lastVideoItemView != null) {
            pauseVideo(lastVideoItemView);
        }
    }


}
