package com.edu24ol.newclass.discover.util;

import android.text.TextUtils;
import android.util.Pair;

import androidx.annotation.Nullable;

import com.edu24.data.DataApiFactory;
import com.edu24.data.server.upload.UploadPartRes;
import com.edu24.data.server.upload.entity.UploadPartBean;
import com.google.gson.Gson;
import com.hqwx.android.platform.exception.HqException;
import com.hqwx.android.platform.server.BaseRes;
import com.yy.android.educommon.log.YLog;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import okhttp3.MediaType;
import okhttp3.MultipartBody;
import okhttp3.RequestBody;
import okhttp3.internal.Util;
import okio.BufferedSink;
import okio.BufferedSource;
import okio.Okio;
import okio.Source;
import rx.Observable;
import rx.Subscriber;
import rx.Subscription;
import rx.android.schedulers.AndroidSchedulers;
import rx.functions.FuncN;
import rx.schedulers.Schedulers;

/**
 * Time:2022/12/5 11:06
 * Author:
 * Description:
 */
public class MultipartUploadUtil {
    public static final long perPartSize = 5 * 1024 * 1024L;
    public static final int MAX_THREAD =10;
    public static final long TIME_STAMP =System.currentTimeMillis();

    public static String getUploadFileName(String videoPath) {
        String fileExtension = getFileExtensionFromUrl(videoPath);
        //YLog.info("this", "keepon getUploadFileName " + fileExtension);
        return MD5Utils.getMD5(videoPath+TIME_STAMP) + fileExtension;
    }

    public static String getFileExtensionFromUrl(String fileName) {
        if (!TextUtils.isEmpty(fileName)) {
            if (!fileName.isEmpty()) {
                int dotPos = fileName.lastIndexOf('.');
                if (0 <= dotPos) {
                    return fileName.substring(dotPos);
                }
            }
        }

        return "";
    }

    public static boolean isNeedPart(long partSize) {
        return partSize > perPartSize;
    }

    //uploadFileName  uploadId:videoUrl
    private static HashMap<String, Pair<String, String>> mInitHashMap = new HashMap<>();

    private static HashMap<PartUploader, UploadPartBean> mPartUploadHashMap = new HashMap<>();
    //uploadFileName:uploadVideoUrl
    private static HashMap<String, String> mFinishMap = new HashMap<>();

    public static void handleUploadVideo(String passport, String videoFilePath, IHandleUploadVideo.UploadListener uploadListener) {

        String uploadFileName = MultipartUploadUtil.getUploadFileName(videoFilePath);
         YLog.info("this","keepon handleUploadVideo "+uploadFileName);
        String uploadVideoUrl = mFinishMap.get(uploadFileName);
        if (TextUtils.isEmpty(uploadVideoUrl)) {
            upLoadVideoInit(passport, uploadFileName, new MultipartUploadUtil.UploadInitListener() {

                @Override
                public void onInitSuccess(String uploadId, String preVideoUrl) {
                    Observable<String> stringObservable = uploadVideo(preVideoUrl, passport, videoFilePath, uploadFileName, uploadId);
                    uploadListener.onUploadSuccess(stringObservable);
                }

                @Override
                public void onCreateSubscription(Subscription subscription) {
                    uploadListener.onCreateSubscription(subscription);

                }

                @Override
                public void onInitFailed(Throwable throwable) {
                    YLog.error(this, "keepon onInitFailed ", throwable);
                    uploadListener.onFailed(throwable);
                }
            });
        } else {
            uploadListener.onUploadSuccess(Observable.create(new Observable.OnSubscribe<String>() {
                @Override
                public void call(Subscriber<? super String> subscriber) {
                    YLog.info(this, "keepon call 文件已经上传过");
                    subscriber.onNext(uploadVideoUrl);
                    subscriber.onCompleted();
                }
            }));
        }

    }

    public static void upLoadVideoInit(String passport, String uploadFileName, UploadInitListener uploadInitListener) {
        Pair<String, String> cachePair = mInitHashMap.get(uploadFileName);
        if (cachePair != null && cachePair.first != null && cachePair.second != null) {
            uploadInitListener.onInitSuccess(cachePair.first, cachePair.second);
            YLog.info("this", "keepon upLoadVideoInit 已经初始化过 " + uploadFileName);
        } else {
            Subscription subscribe = DataApiFactory.getInstance().getUploadJApi().multipartInit(passport, uploadFileName, 1)
                .subscribeOn(Schedulers.io())
                .observeOn(AndroidSchedulers.mainThread())
                .subscribe(uploadPartInitRes -> {
                    if (uploadPartInitRes.isSuccessful() && uploadPartInitRes.getData() != null && !TextUtils.isEmpty(uploadPartInitRes.getData().getUploadId())) {
                        String uploadId = uploadPartInitRes.getData().getUploadId();
                        String videoUrl = uploadPartInitRes.getData().getUrl();
                        mInitHashMap.put(uploadFileName, new Pair<String, String>(uploadId, videoUrl));
                        uploadInitListener.onInitSuccess(uploadId, videoUrl);
                    } else {
                        uploadInitListener.onInitFailed(uploadPartInitRes.getHqException());
                    }
                }, uploadInitListener::onInitFailed);
            uploadInitListener.onCreateSubscription(subscribe);
        }

    }

    private static Observable<String> uploadVideo(String preVideoUrl, String passport, String videoPath, String uploadFileName, String uploadId) {
        List<Observable<UploadPartBean>> videoObservables = MultipartUploadUtil.uploadMulti(passport, videoPath, uploadFileName, uploadId);
        return Observable.zip(videoObservables, new FuncN<String>() {
            @Override
            public String call(Object... args) {
                List<UploadPartBean> uploadPartBeans = new ArrayList<>();
                if (args != null && args.length > 0) {
                    {
                        for (Object arg : args) {
                            if (arg instanceof UploadPartBean) {
                                uploadPartBeans.add((UploadPartBean) arg);
                            }
                        }
                    }
                }
                Gson gson = new Gson();
                String partETags = gson.toJson(uploadPartBeans);
                String videoUrl = null;

                try {
                    BaseRes body = DataApiFactory.getInstance().getUploadJApi().multipartComplete(passport, uploadId, uploadFileName, partETags).execute().body();
                    if (body != null) {
                        if (body.isSuccessful()) {
                            videoUrl = preVideoUrl;
                            onUploadSuccess(uploadFileName, videoUrl);
                        } else {
                            YLog.error(this, "keepon call uploadVideo failed", body.getHqException());
                        }
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                }
                return videoUrl;
            }
        }).subscribeOn(Schedulers.io());
    }

    public static void onUploadSuccess(String uploadFileName, String videoPath) {
        mInitHashMap.remove(uploadFileName);
        mPartUploadHashMap.clear();
        mFinishMap.put(uploadFileName, videoPath);
    }

    public static List<Observable<UploadPartBean>> uploadMulti(String passport, String filePath, String uploadFileName, String uploadId) {
        File file = new File(filePath);
        long partSize = MultipartUploadUtil.perPartSize;
        long fileLength = file.length();
        int partCount = (int) (fileLength / partSize);
        //限制8个线程
        if (partCount > MultipartUploadUtil.MAX_THREAD) {
             YLog.info("this","keepon uploadMulti 超过8个线程了 ");
            partCount = MultipartUploadUtil.MAX_THREAD;
            partSize = fileLength / partCount;
        }

        if (fileLength % partSize != 0) {
            partCount++;
        }

        //if (partCount > 10000) {
        //    throw new RuntimeException("Total parts count should not exceed 10000");
        //} else {
        //    System.out.println("Total parts count " + partCount + "\n");
        //}
        List<Observable<UploadPartBean>> uploadObservableList = new ArrayList<>();
        for (int i = 0; i < partCount; i++) {
            long startPos = i * partSize;
            long curPartSize = (i + 1 == partCount) ? (fileLength - startPos) : partSize;
            //YLog.info("this", "keepon uploadMulti startPos " + startPos + " partSize=" + curPartSize);
            Observable<UploadPartBean> videoObservable = createVideoObservable(passport, new PartUploader(filePath, uploadFileName, startPos, curPartSize, i + 1, uploadId));
            uploadObservableList.add(videoObservable);
        }
        return uploadObservableList;
    }

    public static rx.Observable<UploadPartBean> createVideoObservable(String passport, PartUploader partUploader) {
        return Observable.create(new Observable.OnSubscribe<UploadPartBean>() {
            @Override
            public void call(Subscriber<? super UploadPartBean> subscriber) {
                try {
                    UploadPartBean uploadPartBean = mPartUploadHashMap.get(partUploader);
                    if (uploadPartBean == null) {
                        uploadPartBean =
                            MultipartUploadUtil.uploadMultiPartSyn(passport,
                                partUploader);
                        mPartUploadHashMap.put(partUploader, uploadPartBean);
                        YLog.info(this, "keepon call 分片上传成功 " + partUploader.getPartNumber());
                    } else {
                        YLog.info(this, "keepon call 从缓存获取 " + partUploader.getPartNumber());
                    }
                    subscriber.onNext(uploadPartBean);
                    subscriber.onCompleted();
                } catch (Exception e) {
                    subscriber.onError(
                        new HqException("call upload api error:" + e.getMessage()));
                }
            }
        }).subscribeOn(Schedulers.io());
    }


    public static UploadPartBean uploadMultiPartSyn(String passport, PartUploader partUploader) throws Exception {
        String filePath = partUploader.getFilePath();
        MultipartBody.Builder builder = new MultipartBody.Builder()
            .addFormDataPart("passport", passport);
        if (TextUtils.isEmpty(filePath)) {
            throw new RuntimeException("imagePath can not empty or null!");
        }
        File file = new File(filePath);
        if (!file.exists()) {
            throw new RuntimeException("file not exist!");
        }
        String fileName = partUploader.getUploadFileName();
        builder.addFormDataPart("file", fileName,
            create(MediaType.parse("application/octet-stream"), file, partUploader));
        UploadPartRes uploadFileRes =
            DataApiFactory.getInstance().getUploadJApi().multipartUpload(builder.build(), fileName, passport,
                    partUploader.getPartNumber(), partUploader.getPartSize(), partUploader.getUploadId())
                .execute().body();

        if (uploadFileRes == null) {
            throw new RuntimeException("upload failed");
        }
        if (uploadFileRes.isSuccessful() &&
            uploadFileRes.getData() != null) {
            return uploadFileRes.getData();
        } else {
            throw uploadFileRes.getHqException();
        }

    }

    public static RequestBody create(final MediaType contentType, final File file, PartUploader partUploader) {
        if (file == null) throw new NullPointerException("file == null");

        return new RequestBody() {

            @Override
            public @Nullable
            MediaType contentType() {
                return contentType;
            }

            @Override
            public long contentLength() {
                return partUploader.getPartSize();
            }

            @Override
            public void writeTo(BufferedSink sink) throws IOException {
                Source source = null;
                try {
                    source = Okio.source(file);
                    final BufferedSource bufferedSource = Okio.buffer(source);
                    long offset = partUploader.getStartPos();
                    if (offset > 0) {
                        bufferedSource.skip(offset);
                    }
                    long partSize = partUploader.getPartSize();
                    YLog.info(this, "keepon 开始读取 writeTo " + partUploader.getPartNumber());
                    sink.write(bufferedSource, partSize);
                    YLog.debug(this, "keepon 读取完毕  " + partUploader.getPartNumber());

                } finally {
                    Util.closeQuietly(source);
                }
            }
        };
    }

    interface UploadInitListener {
        void onInitSuccess(String uploadId, String preVideoUrl);

        void onCreateSubscription(Subscription subscription);

        void onInitFailed(Throwable throwable);
    }
}
