package com.edu24ol.newclass.discover.util;

import android.graphics.Point;
import android.os.Handler;
import android.os.HandlerThread;
import android.os.Looper;
import android.os.Message;
import android.util.SparseIntArray;

import java.lang.ref.WeakReference;

import androidx.recyclerview.widget.LinearLayoutManager;
import androidx.recyclerview.widget.RecyclerView;

public class ItemExposureReportHelper implements ItemExposureReportApi {

    private SparseIntArray mExposureData = new SparseIntArray();

    protected HandlerThread mHandlerThread;
    protected MyHandler mHandler;
    protected int mOldFirstCompleteVisiblePos = -1;
    protected int mOldLastCompleteVisiblePos = -1;
    protected RecyclerView mRecyclerView;

    public ItemExposureReportHelper(RecyclerView recyclerView) {
        mRecyclerView = recyclerView;
        mHandlerThread = new HandlerThread("ItemExposureReportHelper");
        mHandlerThread.start();
        mHandler = new MyHandler(mHandlerThread.getLooper(), this);
    }

    @Override
    public void reset() {
        mExposureData.clear();
        mOldFirstCompleteVisiblePos = -1;
        mOldLastCompleteVisiblePos = -1;
    }

    @Override
    public void release() {
        mExposureData.clear();
        mHandler.getLooper().quit();
        mHandlerThread.quit();
        mOldFirstCompleteVisiblePos = -1;
        mOldLastCompleteVisiblePos = -1;
    }

    @Override
    public SparseIntArray getData() {
        return mExposureData;
    }

    @Override
    public void onResume() {
        mHandler.sendEmptyMessage(WHAT_RESUME);
    }

    public void onScrollStateChanged(RecyclerView recyclerView, int scrollState) {
        switch (scrollState) {
            case RecyclerView.SCROLL_STATE_IDLE:
                statExposureViews();
                break;
            default:
                break;
        }
    }

    private void statExposureViews() {
        mHandler.sendEmptyMessage(WHAT_TOUCH);
    }

    private static final int WHAT_TOUCH = 0;
    protected static final int WHAT_RESUME = 1;

    protected static class MyHandler extends Handler {

        private WeakReference<ItemExposureReportHelper> mReference;

        private MyHandler(Looper looper, ItemExposureReportHelper refrence) {
            super(looper);
            mReference = new WeakReference(refrence);
        }

        @Override
        public void handleMessage(Message msg) {
            ItemExposureReportHelper itemExposureReportHelper = mReference.get();
            switch (msg.what) {
                case WHAT_TOUCH:
                    if (itemExposureReportHelper != null) {
                        itemExposureReportHelper.recordTouch();
                    }
                    break;
                case WHAT_RESUME:
                    if (itemExposureReportHelper != null) {
                        itemExposureReportHelper.recordResume();
                    }
                    break;
                default:
                    break;
            }
        }
    }

    protected void recordTouch() {
        Point visiblePosition = findRangePosition();
        if (visiblePosition == null) {
            return;
        }
        int firstCompleteVisiblePos = visiblePosition.x;
        int lastCompleteVisiblePos = visiblePosition.y;
        if (firstCompleteVisiblePos == mOldFirstCompleteVisiblePos &&
                lastCompleteVisiblePos == mOldLastCompleteVisiblePos) {
            return;
        }
        //首次&不包含相同项
        if (mOldFirstCompleteVisiblePos == -1 ||
                firstCompleteVisiblePos > mOldLastCompleteVisiblePos ||
                lastCompleteVisiblePos < mOldFirstCompleteVisiblePos) {
            for (int i = firstCompleteVisiblePos; i <= lastCompleteVisiblePos; i++) {
                addData(i);
            }
        } else {
            //排除相同项
            if (firstCompleteVisiblePos < mOldFirstCompleteVisiblePos) {
                for (int i = firstCompleteVisiblePos; i < mOldFirstCompleteVisiblePos; i++) {
                    addData(i);
                }
            }
            if (lastCompleteVisiblePos > mOldLastCompleteVisiblePos) {
                for (int i = mOldLastCompleteVisiblePos + 1; i <= lastCompleteVisiblePos; i++) {
                    addData(i);
                }
            }
        }
        mOldFirstCompleteVisiblePos = firstCompleteVisiblePos;
        mOldLastCompleteVisiblePos = lastCompleteVisiblePos;
    }

    protected void recordResume() {
        Point visiblePosition = findRangePosition();
        if (visiblePosition == null) {
            return;
        }
        int firstCompleteVisiblePos = visiblePosition.x;
        int lastCompleteVisiblePos = visiblePosition.y;
        for (int i = firstCompleteVisiblePos; i <= lastCompleteVisiblePos; i++) {
            addData(i);
        }
    }

    private void addData(int position) {
        int count = mExposureData.get(position);
        mExposureData.put(position, count + 1);
    }

    private Point findRangePosition() {
        LinearLayoutManager linearLayoutManager =
                (LinearLayoutManager) mRecyclerView.getLayoutManager();
        int firstCompleteVisiblePosition = -1;
        int lastCompleteVisiblePosition = -1;
        try {
            firstCompleteVisiblePosition =
                    linearLayoutManager.findFirstCompletelyVisibleItemPosition();
            lastCompleteVisiblePosition =
                    linearLayoutManager.findLastCompletelyVisibleItemPosition();
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (firstCompleteVisiblePosition == -1) {
            return null;
        } else {
            return new Point(firstCompleteVisiblePosition, lastCompleteVisiblePosition);
        }
    }
}
