package com.edu24ol.newclass.discover;

import android.os.Bundle;
import android.util.Log;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.CheckBox;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import androidx.recyclerview.widget.LinearLayoutManager;

import com.edu24.data.server.discover.entity.ArticleAuthor;
import com.edu24.data.server.discover.entity.HomeDiscoverArticleItemBean;
import com.edu24ol.newclass.discover.adapter.DiscoverBaseArticleListAdapter;
import com.edu24ol.newclass.discover.base.BaseDiscoverArticleListFragment;
import com.edu24ol.newclass.discover.home.IRefreshable;
import com.edu24ol.newclass.discover.home.recommend.HomeDiscoverRecommendListAdapter;
import com.edu24ol.newclass.discover.presenter.IAuthorFollowActionPresenter;
import com.edu24ol.newclass.discover.presenter.ui.IDiscoverAuthorFollowUI;
import com.edu24ol.newclass.discover.stat.DiscoverStat;
import com.hqwx.android.discover.common.R;
import com.hqwx.android.platform.AppMessage;
import com.hqwx.android.platform.exception.HqException;
import com.hqwx.android.platform.stat.StatAgent;
import com.hqwx.android.platform.utils.DoubleClickCheckUtils;
import com.hqwx.android.platform.utils.ToastUtil;
import com.hqwx.android.platform.widgets.pullrefresh.HqwxRefreshLayout;
import com.hqwx.android.service.AppRouter;
import com.hqwx.android.service.ServiceFactory;
import com.yy.android.educommon.log.YLog;
import com.yy.android.educommon.utils.NetworkUtil;

import java.util.List;

import de.greenrobot.event.EventBus;

/**
 * 带有关注作者功能的fragment
 *
 * @param <P>
 */
public abstract class AbsDiscoverFollowAuthorFragment<P extends IAuthorFollowActionPresenter>
        extends BaseDiscoverArticleListFragment implements IDiscoverAuthorFollowUI, IRefreshable {
    protected P mPresenter;
    protected boolean mIsFirstLoading = true;

    @Nullable
    @Override
    public View onCreateView(@NonNull LayoutInflater inflater, @Nullable ViewGroup container,
                             @Nullable Bundle savedInstanceState) {
        View view = inflater.inflate(R.layout.discover_common_frg_child_recommend_layout, null);
        getFragmentArguments(getArguments());
        mChildFragmentType = getChildFragmentType();
        initView(view);
        initListener();
        initAdapter();
        mPresenter = createPresenter();
        EventBus.getDefault().register(this);
        return view;
    }

    @Override
    public void onViewCreated(@NonNull View view, @Nullable Bundle savedInstanceState) {
        super.onViewCreated(view, savedInstanceState);
        loadFirstPageData();
    }

    private void loadFirstPageData() {
        if (isEnableSkeletonLoading()) {
            showSkeletonView();
            loadData();
        } else {
            refresh();
        }
    }

    public void refresh() {
        if (mSmartRefreshLayout != null) {
            mSmartRefreshLayout.getmSmartRefreshLayout().autoRefresh();
        }
    }

    protected void getFragmentArguments(Bundle bundle) {

    }

    abstract protected P createPresenter();

    @Override
    protected DiscoverBaseArticleListAdapter getAdapter() {
        return new HomeDiscoverRecommendListAdapter(getActivity());
    }

    private void initView(View view) {
        mRefreshNoticeView = view.findViewById(R.id.discover_recommend_notice_view);
        mSmartRefreshLayout = view.findViewById(R.id.smart_refresh_layout);
        mRecyclerView = view.findViewById(R.id.recycler_view);
        addScrollAutoPlayVideo();
        mLoadingDataStatusView = view.findViewById(R.id.recommend_loading_status_view);
    }

    private void initListener() {
        mLoadingDataStatusView.setOnClickListener(new View.OnClickListener() {
            @Override
            public void onClick(View v) {
                loadFirstPageData();
            }
        });

        setOnRefreshLoadMoreListener();

    }

    protected void setOnRefreshLoadMoreListener() {
        mSmartRefreshLayout.setOnRefreshLoadMoreListener(
                new com.hqwx.android.platform.widgets.pullrefresh.listener.OnRefreshLoadMoreListener() {
                    @Override
                    public void onLoadMore(HqwxRefreshLayout refreshLayout) {
                        if (mChildFragmentType == CHILD_RECOMMEND_FRG_TYPE) {
                            StatAgent.onEvent(getContext(), "Discovery_RecommendedList_slideLoading");
                        }
                        getNextArticleItemList();
                        StatAgent.onAppDiscoverRefresh(getActivity(), getBelongSeat(), "默认", "上滑加载");
                    }

                    @Override
                    public void onRefresh(HqwxRefreshLayout refreshLayout) {
                        loadData();
                    }
                });
    }

    private void loadData() {
        if (mIsFirstLoading) {
            mIsFirstLoading = false;
            getFirstArticleItemListAndShowDataView();
        } else {
            if (mChildFragmentType == CHILD_RECOMMEND_FRG_TYPE) {
                StatAgent
                        .onEvent(getContext(), "Discovery_RecommendedList_dropdownLoading");
            }
            getRefreshArticleItemList();
        }

        StatAgent.onAppDiscoverRefresh(getActivity(), getBelongSeat(), "默认", "下拉刷新");
    }

    @Override
    protected void refreshArticleList() {
        getFirstArticleItemListAndShowDataView();
    }

    private void getFirstArticleItemListAndShowDataView() {
        showDataView();
        getFirstArticleItemList();
    }

    /**
     * 进来时获取第一页的数据
     */
    protected abstract void getFirstArticleItemList();

    /**
     * 上拉加载更多数据
     */
    protected abstract void getNextArticleItemList();

    /**
     * 下拉刷新数据
     */
    protected abstract void getRefreshArticleItemList();

    @Override
    public void onDestroyView() {
        EventBus.getDefault().unregister(this);
        super.onDestroyView();
    }

    @Override
    public void onRefreshNoData() {
        mSmartRefreshLayout.finishRefresh();
    }

    @Override
    public void onNoData() {
        showEmptyView();
    }

    @Override
    public void onNoMoreData(boolean isRefresh) {
        if (isRefresh) {
            mSmartRefreshLayout.handleFinishRefreshData(true);
        } else {
            mSmartRefreshLayout.handleFinishLoadMoreData(true);
        }
    }

    protected void setGetFirstListFail(Throwable throwable) {
        //获取失败重新赋值为true，autoRefresh会走getFirstArticleItemListAndShowDataView()方法
        mIsFirstLoading = true;
        showErrorView(throwable);
    }

    protected void setGetMoreListSuccess(List<HomeDiscoverArticleItemBean> articleItemBeanList) {
        if (mAdapter != null) {
            if (articleItemBeanList != null) {
                mAdapter.addData(articleItemBeanList);
                mAdapter.notifyDataSetChanged();
            }
            mSmartRefreshLayout.handleFinishLoadMoreData(false);
        }
    }

    @Override
    public void onAttentionClick(CheckBox checkBox, ArticleAuthor author, boolean attention) {
        super.onAttentionClick(checkBox, author, attention);
        if (!DoubleClickCheckUtils.vertifyDurationByTime(500)) {
            checkBox.setChecked(!checkBox.isChecked());
            return;
        }
        if (!NetworkUtil.isNetworkAvailable(getActivity())) {
            checkBox.setChecked(!checkBox.isChecked());
            ToastUtil.showMessage(getActivity(), "当前网络不可用");
            return;
        }
        if (!ServiceFactory.getAccountService().isLogin()) {
            AppRouter.startAccountPhoneLogin(getContext());
            checkBox.setChecked(!attention);
            return;
        }
        if (attention) {
            if (mChildFragmentType == CHILD_SQUARE_FRG_TYPE) {
                DiscoverStat
                        .onAppFollow("发现广场页", "账号", author.id,
                                author.name);
            } else if (mChildFragmentType == CHILD_RECOMMEND_FRG_TYPE) {
                DiscoverStat
                        .onAppFollow("发现推荐页", "账号", author.id,
                                author.name);
            }
            mPresenter.followAuthor(ServiceFactory.getAccountService().getHqToken(), author.id);
        } else {
            mPresenter.unFollowAuthor(ServiceFactory.getAccountService().getHqToken(), author.id);
        }

    }

    private void updateAttention(long authorId, boolean attention) {
        if (mAdapter == null || mAdapter.getDatas() == null) {
            return;
        }
        List<HomeDiscoverArticleItemBean> beans = mAdapter.getDatas();
        if (attention) {
            for (HomeDiscoverArticleItemBean bean : beans) {
                if (bean.authorId == authorId) {
                    bean.author.setIsAttend(true);
                }
            }
            YLog.info(this, "onAttentionClick: follow " + authorId);
        } else {
            for (HomeDiscoverArticleItemBean bean : beans) {
                if (bean.authorId == authorId) {
                    bean.author.setIsAttend(false);
                }
            }
            YLog.info(this, "onAttentionClick: unFollow " + authorId);
        }
        LinearLayoutManager layoutManager = (LinearLayoutManager) mRecyclerView.getLayoutManager();
        int startPosition = layoutManager.findFirstVisibleItemPosition();
        int endPosition = layoutManager.findLastVisibleItemPosition();
        try {
            //有部分不会刷新，所以加5
            mAdapter.notifyItemRangeChanged(
                    startPosition, endPosition - startPosition + 5, "attention"
            );
        } catch (Exception e) {
            Log.e("TAG", "DiscoverRecommendFragment updateAttention Exception:", e);
            mAdapter.notifyItemRangeChanged(
                    startPosition, endPosition - startPosition + 1, "attention"
            );
        }

    }

    @Override
    public void onFollowAuthorSuccess(long authorId) {
        EventBus.getDefault().post(AppMessage.obtain(DiscoverEvent.ON_FOLLOW_AUTHOR, authorId));
        YLog.info(this, "onFollowSuccess: ");
    }

    @Override
    public void onFollowAuthorFail(Throwable throwable) {
        if (throwable instanceof HqException) {
            ToastUtil.showMessage(getActivity(), throwable.getMessage());
        } else {
            ToastUtil.showMessage(getActivity(), "关注失败，请重试");
        }
        YLog.error(this, "onFollowFailure: ", throwable);
    }

    @Override
    public void onUnFollowAuthorSuccess(long authorId) {
        EventBus.getDefault().post(AppMessage.obtain(DiscoverEvent.ON_UNFOLLOW_AUTHOR, authorId));
        YLog.info(this, "onUnFollowSuccess: ");
    }

    @Override
    public void onUnFollowAuthorFail(Throwable throwable) {
        if (throwable instanceof HqException) {
            ToastUtil.showMessage(getActivity(), throwable.getMessage());
        } else {
            ToastUtil.showMessage(getActivity(), "取消关注失败，请重试");
        }
        YLog.error(this, "onUnFollowFailure: ", throwable);
    }

    protected void setShowNoticeView(String showText) {
        mRefreshNoticeView.setText(showText);
        super.showNoticeView();
    }

    @Override
    public void onEvent(AppMessage appMessage) {
        super.onEvent(appMessage);
        if (DiscoverEvent.ON_FOLLOW_AUTHOR.equals(appMessage.getWhat())) {
            if (appMessage.getObj() != null && appMessage.getObj() instanceof Long) {
                long authorId = (long) appMessage.getObj();
                updateAttention(authorId, true);

                if (isShowOfficialAccountDialog()) {
                    //关注成功，显示公众号导流弹框
                    showFollowSuccessDialog(authorId);
                } else {
                    if (!mIsInteractiveState) {
                        return;
                    }
                    ToastUtil.showMessageWithSuccess(getActivity(), "关注成功");
                }
            }
        } else if (DiscoverEvent.ON_UNFOLLOW_AUTHOR.equals(appMessage.getWhat())) {
            if (appMessage.getObj() != null && appMessage.getObj() instanceof Long) {
                long authorId = (long) appMessage.getObj();
                updateAttention(authorId, false);
            }
        }
    }

    @Override
    public void showLoadingDialog() {

    }

    @Override
    public void dismissLoadingDialog() {

    }


}
