/*
 * Decompiled with CFR 0.152.
 */
package com.yy.base.cache;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;
import com.yy.base.cache.JCacheContainer;
import com.yy.base.cache.JCacheKey;
import com.yy.base.cache.JCacheValue;
import com.yy.base.cache.lru.HotEndLruCache;
import com.yy.base.logger.MLog;
import com.yy.base.taskexecutor.YYTaskExecutor;
import com.yy.base.thread.ThreadBus;
import java.lang.ref.WeakReference;
import java.util.concurrent.locks.ReentrantReadWriteLock;

public class JCache<T> {
    private static final float DEFAULT_HARD_HOT_PERCENT = 0.75f;
    private static final float DEFAULT_WEAK_HOT_PERCENT = 0.6f;
    private static final float DEFAULT_SIZE_INCREASE_STEP = 1.5f;
    private static final long TRIM_HARD_INTERVAL = 90000L;
    private static final long TRIM_WEAK_INTERVAL = 270000L;
    private static final long TRIM_WEAK_MAX_INTERVAL = 360000L;
    private static final int TRIM_HARD_MAX_COUNT = 1000;
    private static final int TRIM_WEAK_MAX_COUNT = 2000;
    private static final String TAG_PREFIX = "JCache_";
    private final HotEndLruCache<JCacheKey, JCacheValue<T>> mHardCache;
    private final HotEndLruCache<JCacheKey, JCacheValue<WeakReference<T>>> mWeakCache;
    private final String mCacheName;
    private final long mExpireTime;
    private final String mTag;
    private final ReentrantReadWriteLock mLock = new ReentrantReadWriteLock();
    private final CacheController<T> mCacheController;
    private final int mHardInitSize;
    private final int mWeakInitSize;
    private long mLastTrimWeakTime = System.currentTimeMillis();
    private final Runnable mTrimHardTask = new Runnable(){

        @Override
        public void run() {
            JCache.this.trimHard();
            ThreadBus.postDelayed(4, JCache.this.mTrimHardTask, 90000L);
        }
    };
    private final Runnable mTrimWeakTask = new Runnable(){

        @Override
        public void run() {
            JCache.this.trimWeak();
            ThreadBus.postDelayed(4, JCache.this.mTrimWeakTask, 270000L);
        }
    };

    public JCache(@NonNull JCacheContainer.JCacheBuilder<T> builder) {
        this.mCacheName = builder.cacheClazz.getName();
        this.mTag = TAG_PREFIX + this.mCacheName;
        this.mCacheController = builder.cacheController;
        this.mExpireTime = builder.expireTime;
        this.mHardInitSize = builder.minHardSize;
        this.mWeakInitSize = builder.minHardSize * 8;
        this.mHardCache = new HotEndLruCache(this.mHardInitSize, 0.75f);
        this.mWeakCache = new HotEndLruCache(this.mWeakInitSize, 0.6f);
        this.startTrimTask();
    }

    @Nullable
    public T putIfAbsent(@NonNull JCacheKey cacheKey, @NonNull T data) {
        this.mLock.readLock().lock();
        JCacheValue<T> cacheObject = this.mHardCache.get(cacheKey);
        if (cacheObject != null) {
            this.mLock.readLock().unlock();
            return cacheObject.value;
        }
        this.mLock.readLock().unlock();
        this.mLock.writeLock().lock();
        cacheObject = this.mHardCache.get(cacheKey);
        if (cacheObject != null) {
            this.mLock.writeLock().unlock();
            return cacheObject.value;
        }
        JCacheValue<WeakReference<T>> weakCacheObject = this.mWeakCache.remove(cacheKey);
        if (weakCacheObject == null) {
            this.putToHard(cacheKey, new JCacheValue<T>(cacheKey, data));
            this.mLock.writeLock().unlock();
            return null;
        }
        Object weakValue = ((WeakReference)weakCacheObject.value).get();
        if (weakValue == null) {
            this.putToHard(cacheKey, new JCacheValue<T>(cacheKey, data));
            this.mLock.writeLock().unlock();
            return null;
        }
        this.putToHard(cacheKey, new JCacheValue(cacheKey, weakValue));
        this.mLock.writeLock().unlock();
        return weakValue;
    }

    @NonNull
    public T get(@NonNull JCacheKey cacheKey) {
        return this.get(cacheKey, true);
    }

    @Nullable
    public T get(@NonNull JCacheKey cacheKey, boolean autoCreate) {
        JCacheValue<T> cacheObject = this.cacheObjectForKey(cacheKey, autoCreate);
        if (cacheObject != null) {
            long current = System.currentTimeMillis();
            long lastRefreshTime = cacheObject.lastRefreshTime;
            if (this.mExpireTime != -1L && current - lastRefreshTime >= this.mExpireTime) {
                cacheObject.lastRefreshTime = current;
                YYTaskExecutor.execute(() -> this.mCacheController.onNeedRefresh(cacheKey, cacheObject));
            }
            return cacheObject.value;
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    public JCacheValue<T> cacheObjectForKey(@NonNull JCacheKey cacheKey, boolean autoCreate) {
        this.mLock.readLock().lock();
        try {
            JCacheValue<T> cacheObject = this.mHardCache.get(cacheKey);
            if (cacheObject != null) {
                JCacheValue<T> jCacheValue = cacheObject;
                return jCacheValue;
            }
            this.mLock.readLock().unlock();
            this.mLock.writeLock().lock();
            cacheObject = this.mHardCache.get(cacheKey);
            if (cacheObject == null) {
                cacheObject = this.createNewValue(cacheKey, autoCreate);
            }
            this.mLock.readLock().lock();
            this.mLock.writeLock().unlock();
            JCacheValue<T> jCacheValue = cacheObject;
            return jCacheValue;
        }
        finally {
            this.mLock.readLock().unlock();
        }
    }

    @Nullable
    private JCacheValue<T> createNewValue(@NonNull JCacheKey cacheKey, boolean autoCreate) {
        JCacheValue<T> cacheObject = null;
        JCacheValue<WeakReference<T>> weakCacheObject = this.mWeakCache.remove(cacheKey);
        if (weakCacheObject != null) {
            Object weakValue = ((WeakReference)weakCacheObject.value).get();
            if (weakValue != null) {
                cacheObject = new JCacheValue(cacheKey, weakValue);
            } else if (autoCreate) {
                cacheObject = new JCacheValue<T>(cacheKey, this.mCacheController.createNewCacheObject(cacheKey));
            }
        } else if (autoCreate) {
            cacheObject = new JCacheValue<T>(cacheKey, this.mCacheController.createNewCacheObject(cacheKey));
        }
        if (cacheObject == null) {
            return null;
        }
        this.putToHard(cacheKey, cacheObject);
        return cacheObject;
    }

    private void putToHard(@NonNull JCacheKey cacheKey, @NonNull JCacheValue<T> value) {
        int hardMaxSize;
        int hardSize = this.mHardCache.size();
        if (hardSize + 1 > (hardMaxSize = this.mHardCache.maxSize())) {
            int newHardMaxSize = (int)((float)hardMaxSize * 1.5f);
            if (MLog.isLogLevelBlowInfo()) {
                MLog.info(this.mTag, "putToHard newHardMaxSize: " + newHardMaxSize, new Object[0]);
            }
            this.mHardCache.resize(newHardMaxSize, 0.75f);
        }
        this.mHardCache.put(cacheKey, value);
    }

    public void clear() {
        this.mLock.writeLock().lock();
        this.mHardCache.clear();
        this.mWeakCache.clear();
        this.mLock.writeLock().unlock();
    }

    public void release() {
        this.clear();
        this.stopTrimTask();
    }

    public boolean equals(Object o) {
        if (!(o instanceof JCache)) {
            return false;
        }
        JCache other = (JCache)o;
        return this.mCacheName.equals(other.mCacheName);
    }

    public int hashCode() {
        return this.mCacheName.hashCode();
    }

    private void startTrimTask() {
        ThreadBus.postDelayed(4, this.mTrimHardTask, 90000L);
        ThreadBus.postDelayed(4, this.mTrimWeakTask, 270000L);
    }

    private void stopTrimTask() {
        ThreadBus.removeCallbacks(4, this.mTrimHardTask, null);
        ThreadBus.removeCallbacks(4, this.mTrimWeakTask, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void trimHard() {
        this.mLock.writeLock().lock();
        try {
            int maxSize = this.mHardCache.maxSize();
            if (maxSize <= this.mHardInitSize) {
                return;
            }
            int currentSize = this.mHardCache.size();
            int maxHotSize = this.mHardCache.maxHotSize();
            int trimThresholdSize = (int)((float)maxHotSize * 0.75f);
            int maxTrimCount = Math.min(currentSize - trimThresholdSize, 1000);
            if (MLog.isLogLevelBlowInfo()) {
                MLog.info(this.mTag, "trimHard maxTrimCount: " + maxTrimCount + ", trimThresholdSize: " + trimThresholdSize + ", curSize: " + currentSize + ", maxSize: " + maxSize, new Object[0]);
            }
            if (maxTrimCount <= 0) {
                return;
            }
            long start = System.currentTimeMillis();
            IntCounter realTrimCount = new IntCounter();
            int traverseTrimCount = this.mHardCache.traverseTrim(maxTrimCount, (key, value) -> {
                if (!this.canValueBeTrimmed((JCacheKey)key, value.value)) {
                    return false;
                }
                this.mHardCache.remove((JCacheKey)key);
                JCacheValue weakValue = new JCacheValue((JCacheKey)key, new WeakReference(value.value));
                weakValue.lastRefreshTime = value.lastRefreshTime;
                int weakSize = this.mWeakCache.size();
                int weakMaxSize = this.mWeakCache.maxSize();
                if (weakSize + 1 > weakMaxSize) {
                    int newWeakMaxSize = (int)((float)weakMaxSize * 1.5f);
                    if (MLog.isLogLevelBlowInfo()) {
                        MLog.info(this.mTag, "trimHard weak resize: " + newWeakMaxSize, new Object[0]);
                    }
                    this.mWeakCache.resize(newWeakMaxSize, 0.6f);
                }
                this.mWeakCache.put((JCacheKey)key, weakValue);
                ++realTrimCount.count;
                return true;
            });
            currentSize = this.mHardCache.size();
            if (currentSize <= trimThresholdSize) {
                int newMaxSize = Math.max(maxHotSize, this.mHardInitSize);
                if (MLog.isLogLevelBlowInfo()) {
                    MLog.info(this.mTag, "trimHard resize: " + newMaxSize, new Object[0]);
                }
                this.mHardCache.resize(newMaxSize, 0.75f);
            }
            if (MLog.isLogLevelBlowInfo()) {
                MLog.info(this.mTag, "trimHard traverseTrimCount: " + traverseTrimCount + ", realTrimCount: " + realTrimCount.count + ", trimThresholdSize: " + trimThresholdSize + ", curSize: " + currentSize + ", maxSize: " + this.mHardCache.maxSize() + ", cost: " + (System.currentTimeMillis() - start), new Object[0]);
            }
        }
        finally {
            this.mLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void trimWeak() {
        this.mLock.writeLock().lock();
        try {
            int maxSize = this.mWeakCache.maxSize();
            if (maxSize <= this.mWeakInitSize) {
                return;
            }
            int currentSize = this.mWeakCache.size();
            int maxHotSize = this.mWeakCache.maxHotSize();
            int trimThresholdSize = (int)((float)maxHotSize * 0.75f);
            int maxTrimCount = Math.min(currentSize - trimThresholdSize, 2000);
            if (MLog.isLogLevelBlowInfo()) {
                MLog.info(this.mTag, "trimWeak maxTrimCount: " + maxTrimCount + ", trimThresholdSize: " + trimThresholdSize + ", curSize: " + currentSize + ", maxSize: " + maxSize, new Object[0]);
            }
            if (maxTrimCount <= 0) {
                if (this.mLastTrimWeakTime < 360000L || currentSize <= 0) {
                    return;
                }
                maxTrimCount = maxSize - maxHotSize;
            }
            this.mLastTrimWeakTime = System.currentTimeMillis();
            IntCounter realTrimCount = new IntCounter();
            int traverseTrimCount = this.mWeakCache.traverseTrim(maxTrimCount, (key, cacheValue) -> {
                Object value = ((WeakReference)cacheValue.value).get();
                if (value == null) {
                    this.mWeakCache.remove((JCacheKey)key);
                    ++realTrimCount.count;
                    return true;
                }
                return false;
            });
            currentSize = this.mWeakCache.size();
            if (currentSize <= trimThresholdSize) {
                int newMaxSize = Math.max(maxHotSize, this.mWeakInitSize);
                if (MLog.isLogLevelBlowInfo()) {
                    MLog.info(this.mTag, "trimWeak resize: " + newMaxSize, new Object[0]);
                }
                this.mWeakCache.resize(newMaxSize, 0.6f);
            }
            if (MLog.isLogLevelBlowInfo()) {
                MLog.info(this.mTag, "trimWeak traverseTrimCount: " + traverseTrimCount + ", realTrimCount: " + realTrimCount.count + ", trimThresholdSize: " + trimThresholdSize + ", curSize: " + currentSize + ", maxSize: " + this.mWeakCache.maxSize() + ", cost: " + (System.currentTimeMillis() - this.mLastTrimWeakTime), new Object[0]);
            }
        }
        finally {
            this.mLock.writeLock().unlock();
        }
    }

    protected boolean canValueBeTrimmed(JCacheKey cacheKey, T value) {
        return this.mCacheController.canValueBeTrimmed(cacheKey, value);
    }

    private static class IntCounter {
        int count = 0;

        private IntCounter() {
        }
    }

    public static abstract class CacheController<T> {
        public abstract T createNewCacheObject(@NonNull JCacheKey var1);

        public void onNeedRefresh(@NonNull JCacheKey cacheKey, @NonNull JCacheValue<T> cacheObject) {
        }

        public boolean canValueBeTrimmed(JCacheKey cacheKey, T value) {
            return true;
        }
    }
}

