package tv.athena.live.component.business.roominfo

import android.arch.lifecycle.Observer
import android.support.annotation.Nullable
import com.google.gson.Gson
import com.google.protobuf.nano.InvalidProtocolBufferNanoException
import com.thunder.livesdk.ThunderRtcConstant
import com.yy.liveplatform.proto.nano.LpfLiveinfo
import com.yy.liveplatform.proto.nano.LpfLiveinterconnect
import com.yy.liveplatform.proto.nano.LpfLiveroomtemplate
import com.yy.liveplatform.proto.nano.LpfMedia
import com.yy.liveplatform.proto.nano.LpfUser
import tv.athena.annotation.MessageBinding
import tv.athena.core.sly.Sly
import tv.athena.live.api.IDataCallback
import tv.athena.live.api.roominfo.MicInfo
import tv.athena.live.api.roominfo.RoomInfoApi
import tv.athena.live.api.roominfo.RoomInfoWrapper
import tv.athena.live.base.arch.IComponentViewModel
import tv.athena.live.base.manager.CommonViewModel
import tv.athena.live.basesdk.liveroom.ApplicationStatus
import tv.athena.live.basesdk.thunderblotwrapper.AbscThunderEventListener
import tv.athena.live.basesdk.thunderblotwrapper.ThunderHandleManager
import tv.athena.live.component.business.roominfo.repository.RoomInfoRepository
import tv.athena.live.utils.ALog
import tv.athena.live.utils.ThreadSafeMutableLiveData
import tv.athena.service.api.IMessageCallback
import tv.athena.service.api.MessageResponse
import tv.athena.service.api.ServiceFailResult
import tv.athena.service.api.event.ServiceBroadcastEvent
import java.util.concurrent.ConcurrentHashMap

/**
 * @author : chf
 * @e-mail : chenhaofeng@joyy.sg
 * @date : 2019/11/19 20:02
 * @desc :
 */
class RoomInfoViewModel() :
    AbscThunderEventListener(),
    IComponentViewModel {
    private val mRoomInfoListeners = ArrayList<RoomInfoApi.RoomInfoListener>()
    private val mRepository = RoomInfoRepository()

    //LiveInfo 信息
    var mLiveInfo = ThreadSafeMutableLiveData<LpfLiveinfo.LiveInfo>()

    //上一次麦位缓存信息
    private val mLastLinkMicInfos =
        ConcurrentHashMap<Int, LpfLiveinterconnect.LiveInterconnectInfo>()
    private val mJsonParser = Gson()

    //公共数据
    private var mCommonViewModel: CommonViewModel? = null

    //RoomInfo 房间信息包装类
    private val mRoomInfoWrapper = RoomInfoWrapper()

    //时间戳
    private var lastTimeStamp = -1L

    //网络连接类型
    private var lastNetworkType: Int? = null


    private var mSidObserver = Observer<Long> { sid ->
        if (sid != null) {
            getLiveRoomInfo(sid)
            //监听sid ，有变化更新到 mRoomWrapper
            updateRoomInfoWrapperSid(sid)
        }
    }

    /**
     * 获取房间包装信息
     * @return RoomInfoWrapper
     */
    val roomInfoWrapper: RoomInfoWrapper
        get() {
            ALog.i(TAG, "getRoomInfoWrapper $mRoomInfoWrapper")
            return mRoomInfoWrapper
        }

    val micInfos: Map<Int, LpfLiveinterconnect.LiveInterconnectInfo>
        get() {
            ALog.i(TAG, "getRoomInfoWrapper $mRoomInfoWrapper")
            return mLastLinkMicInfos
        }

    fun onCreate(component: RoomInfoComponent) {
        ALog.i(TAG, "onCreate($component)")
        Sly.subscribe(this)
        mCommonViewModel = component?.componentContext?.commonViewModel
        ThunderHandleManager.mAthLiveThunderEventCallback.registerThunderEventListener(this@RoomInfoViewModel)
        initRoomInfo()
    }

    fun destroy() {
        ALog.i(TAG, "destroy")
        Sly.unSubscribe(this)
        ThunderHandleManager.mAthLiveThunderEventCallback.unRegisterRtcEventListener(this@RoomInfoViewModel)
    }

    /**
     *  根据 sid 主动请求服务端获取当前房间信息
     *
     */
    private fun initRoomInfo() {
        mCommonViewModel?.observeSidForever(mSidObserver)
    }

    private fun updateRoomInfoWrapperSid(sid: Long) {
        ALog.i(TAG,
            "updateRoomInfoWrapperSid [commonSid :$sid] -- [roomInfoWrapper.sid : ${mRoomInfoWrapper.sid}]")
        if (sid != mRoomInfoWrapper.sid) {
            mRoomInfoWrapper.sid = sid
            for (listener in mRoomInfoListeners) {
                listener.onRoomInfoStatusChange(mRoomInfoWrapper)
            }
        }
    }

    /**
     * 添加 RoomInfoListener  用于获取麦位信息 以及 房间其他消息
     * @param listener
     */
    fun addRoomInfoListener(listener: RoomInfoApi.RoomInfoListener?) {
        listener?.let {
            if (!mRoomInfoListeners.contains(listener)) {
                ALog.i(TAG, "addRoomInfoListener $listener")
                mRoomInfoListeners.add(listener)
            }
        }
    }

    /**
     * 删除RoomInfoListener
     * @param listener
     */
    fun removeRoomInfoListener(listener: RoomInfoApi.RoomInfoListener?) {
        listener?.let {
            if (mRoomInfoListeners.contains(listener)) {
                ALog.i(TAG, "removeRoomInfoListener $listener")
                mRoomInfoListeners.remove(listener)
            }
        }
    }

    /**
     *  根据 position 获取麦位信息
     *  @param position 位置信息 从 1 开始
     */
    @Nullable
    fun getMicInfo(position: Int): LpfLiveinterconnect.LiveInterconnectInfo? {
        return mLastLinkMicInfos[position]
    }

    /**
     *  统一 处理服务端返回的 RoomInfo
     *  @param roomInfo 服务端返回房间信息
     */
    private fun handleRoomInfoUpdate(roomInfo: LpfLiveroomtemplate.LiveRoomInfo?) {
        try {
            if (roomInfo == null) {
                ALog.i(TAG, "handleRoomInfoUpdate roomInfo == null")
                return
            }
            if (mRoomInfoListeners.isEmpty()) {
                ALog.i(TAG, "handleRoomInfoUpdate mRoomInfoListeners isEmpty")
                return
            }
            //如果roomInfo 的信息不是当前频道的消息直接return
            if (roomInfo.sid != mCommonViewModel?.sid) {
                ALog.i(TAG, "handleRoomInfoUpdate is not the same sid " +
                    "[roomInfo.sid : ${roomInfo.sid}] " +
                    "[mCommonViewModel.sid : ${mCommonViewModel?.sid}]")
                return
            }
            ALog.i(TAG, "handleRoomInfoUpdate 当前房间房主的 uid [anchorUid : ${roomInfo.anchorUid}]")
            mCommonViewModel?.roomOwnerUid = roomInfo.anchorUid
            //处理 观看模块
            handleLiveViewModule(roomInfo)
            //处理房主直播状态变更(先处理连麦状态变更，再处理主播状态变化，是因为主播退频道，可能需要所有人退频道，导致时序问题)
            notifyRoomInfoStatusChange(roomInfo)
        } catch (e: java.lang.Exception) {
            e.printStackTrace()
            ALog.e(TAG, "handleRoomInfoUpdate error", e)
        }
    }

    /**
     * 处理观看模块
     */
    private fun handleLiveViewModule(roomInfo: LpfLiveroomtemplate.LiveRoomInfo) {
        //处理连麦信息变更
        if (roomInfo.liveViewModule == null) {
            ALog.i(TAG, "handleLiveViewModule 处理观看模块 roomInfo.liveViewModule == null")
            return
        }
        ALog.i(TAG, "handleLiveViewModule  观看模块 [ type : ${roomInfo.liveViewModule.type}")
        when (roomInfo.liveViewModule.type) {
            LpfLiveroomtemplate.LVT_SINGLE_LIVE -> { //单人直播
                val info = mJsonParser.fromJson(roomInfo.liveViewModule.info,
                    LpfLiveinterconnect.LiveInterconnectingInfo::class.java)
                if (info != null) {
                    notifyLinkMicChange(info.connectInfos.toList())
                }
            }
            LpfLiveroomtemplate.LVT_LIVE_INTERCONNECT -> { //多人连麦
                val info = mJsonParser
                    .fromJson(roomInfo.liveViewModule.info,
                        LpfLiveroomtemplate.LiveInterConnectingViewInfo::class.java)
                if (info?.connectingInfo != null) {
                    notifyLinkMicChange(info.connectingInfo.connectInfos.toList())
                }
            }
            LpfLiveroomtemplate.LVT_NONE -> {
                notifyLinkMicChange(ArrayList())
            }
        }
    }

    /**
     * 用一个map去存原来的房间连麦信息
     * position:info
     *
     *
     * 遍历后端下发的最新连麦列表
     * 拿postion去拿info
     * 1、如果info是空的，说明这个位置还没有人上麦，再判断当前这个人的livestatus是不是1
     * a:是1，说明当前这个人在这个位置上麦，通知外面；
     * b:是0，不管
     * 2、如果info不为空，说明之前有人上麦了，先判断当前这个人和之前那个人是不是同个人
     * a:是同个人，再看下两个人的livestatus是不是相同
     * a1:不是，通知外面改变
     * a2:是，不管
     * b:不是同个人，说明被抢占了,先通知外面取消前一个人的流，再订阅当前这个人的流
     *
     *
     * 遍历完，更新原来的map
     *
     * @param newLiveInterConnectInfo
     */
    private fun notifyLinkMicChange(
        newLiveInterConnectInfo: List<LpfLiveinterconnect.LiveInterconnectInfo>?
    ) {
        if (newLiveInterConnectInfo == null) {
            ALog.i(TAG, "notifyLinkMicChange 通知麦位列表数据更新 newLiveInterConnectInfo == null")
            return
        }
        //List 为空，证明该房间已经没有人在直播了
        if (newLiveInterConnectInfo.isEmpty()) {
            ALog.i(TAG, "notifyLinkMicChange, 麦位列表数据为空 streamer is over")
            //如果列表为空，则直接使用缓存数据，通知外面麦位当前处于关闭状态
            for (item in mLastLinkMicInfos.values) {
                //如果以前的缓存数据 item 麦位是打开那么这时候就需要把当前麦位变化为 CLOSE 通知出去
                if (item.liveStatus == LpfLiveroomtemplate.LiveRoomInfo.LIVING) {
                    if (item.user != null) {
                        notifyOutSide(MicInfo(item, MicInfo.MicStatus.VIDEO_CLOSE))
                    }
                }
            }
            mLastLinkMicInfos.clear()
            return
        }

        //List 长度比缓存的小，证明该房间改变了bzType，当麦位数从9->6时，需要删除多余的map长度，而且是从最后一个麦位删起
        if (mLastLinkMicInfos.size > newLiveInterConnectInfo.size) {
            val lastMaxPosition = mLastLinkMicInfos.size
            ALog.i(TAG, "Crop mLastLinkMicInfoList because callBack size changed" +
                "[${mLastLinkMicInfos.size} to ${newLiveInterConnectInfo.size}]")
            ALog.i(TAG, "**********************")
            for (position in lastMaxPosition downTo 1) {
                if (mLastLinkMicInfos.size > newLiveInterConnectInfo.size) {
                    mLastLinkMicInfos.remove(position)
                    ALog.i(TAG, "Del invalid position=$position")
                    continue
                }
                break
            }
            ALog.i(TAG, "**********************")
        }

        for (item in newLiveInterConnectInfo) {
            updateLinkMicInfo(item)
        }
    }

    /**
     * 更新麦位 流的信息 关闭、打开
     * @param item
     */
    private fun updateLinkMicInfo(item: LpfLiveinterconnect.LiveInterconnectInfo) {
        val position = item.positionInfo.position
        val lastItem = mLastLinkMicInfos[position]
        //val lastItem = findLastMicInfoWithUid(item.user?.uid)
        ALog.i(TAG, "\n\n*********************************************************************************")
        ALog.i(TAG, "updateLinkMicInfo,更新麦位各种状态 位置 [position: $position] [uid: ${item.user?.uid}]")
        ALog.i(TAG, "updateLinkMicInfo 麦位上一次存储的信息 ：[lastItem:\n $lastItem]")
        ALog.i(TAG, "updateLinkMicInfo 麦位最新存储的信息 ： [newItem:\n $item]")
        ALog.i(TAG, "*********************************************************************************")
        //处理位置的 媒体类型变化
        handlePositionMediaType(lastItem, item)
        //处理位置的 麦位视频状态
        handlePositionVideoStatus(lastItem, item)
        //处理位置的 麦位状态（静音、锁麦等）
        handlePositionStatus(lastItem, item)
        //处理位置的 流状态 （麦位的用户当前APP 是否是处于后台 或者前台）
        handlePositionStreamStatus(lastItem, item)
        //处理位置的用户信息拓展字段
        handlePositionMicUserInfoExtend(lastItem, item)
        mLastLinkMicInfos[position] = item
    }

    private fun handlePositionMediaType(
        lastItem: LpfLiveinterconnect.LiveInterconnectInfo?,
        item: LpfLiveinterconnect.LiveInterconnectInfo
    ) {
        ALog.i(TAG, "handlePositionMediaType uid=${item?.user?.uid} mediaType=${item?.mediaType}")
        if (lastItem != null && lastItem.mediaType != item.mediaType) {
            notifyOutSide(MicInfo(item, MicInfo.MicStatus.MEDIA_TYPE_CHANGE))
        }
    }

    /**
     * 处理视频模块
     */
    private fun handlePositionVideoStatus(
        lastItem: LpfLiveinterconnect.LiveInterconnectInfo?,
        item: LpfLiveinterconnect.LiveInterconnectInfo
    ) {
        ALog.i(TAG, "************ handlePositionVideoStatus 视频模块 pos=${item.positionInfo.position} " +
            "lastItem=${lastItem?.user?.uid} lastStatus=${lastItem?.liveStatus} ;" +
            " curItem=${item?.user?.uid} " + " curStatus=${item.liveStatus}")
        if (lastItem == null) {
            when (item.liveStatus) {
                LpfLiveroomtemplate.LiveRoomInfo.LIVE_END -> {
                    ALog.i(TAG, "handlePositionVideoStatus 位置 ${item.positionInfo?.position} not in live")
                }
                LpfLiveroomtemplate.LiveRoomInfo.LIVING -> {
                    if (item.user != null) {
                        notifyOutSide(MicInfo(item, MicInfo.MicStatus.VIDEO_OPEN))
                    }
                }
            }
            return
        }
        //同一个人
        if (isSamePerson(lastItem, item)) {
            //判断状态是否一致 ，一致不处理
            if (item.liveStatus == lastItem.liveStatus) {
                ALog.i(TAG, "handlePositionVideoStatus 视频模块 same person but status not change")
                return
            }
            if (item.liveStatus != lastItem.liveStatus) {
                when (item.liveStatus) {
                    LpfLiveroomtemplate.LiveRoomInfo.LIVE_END -> {
                        if (lastItem.user != null) {
                            notifyOutSide(MicInfo(lastItem, MicInfo.MicStatus.VIDEO_CLOSE))
                        }
                    }
                    LpfLiveroomtemplate.LiveRoomInfo.LIVING -> {
                        if (item.user != null) {
                            notifyOutSide(MicInfo(item, MicInfo.MicStatus.VIDEO_OPEN))
                        }
                    }
                }
            }
            return
        }
        // 不一样的上麦者，先通知上一个关闭
        if (lastItem.user != null) {
            ALog.i(TAG, "handlePositionVideoStatus 视频模块 通知关闭上一个连麦者视频模块 name=${lastItem.user.uid}")
            notifyOutSide(MicInfo(lastItem, MicInfo.MicStatus.VIDEO_CLOSE))
        }
        if (item.liveStatus == LpfLiveroomtemplate.LiveRoomInfo.LIVING) {
            //如果当前的人是打开状态
            if (item.user != null) {
                ALog.i(TAG, "handlePositionVideoStatus 视频模块 通知打开新的连麦者视频模块")
                notifyOutSide(MicInfo(item, MicInfo.MicStatus.VIDEO_OPEN))
            }
        }
    }

    /**
     * 目前 麦位状态 位置状态互斥 （静音、锁麦、空位置等）
     * 具体看 MicStatus
     * @param lastItem
     * @param item
     */
    private fun handlePositionStatus(
        lastItem: LpfLiveinterconnect.LiveInterconnectInfo?,
        item: LpfLiveinterconnect.LiveInterconnectInfo
    ) {
        val currentPositionStatus = item.positionInfo.positionStatus
        val lastPositionStatus = lastItem?.positionInfo?.positionStatus
        ALog.i(TAG, "handlePositionStatus 处理麦位情况 [ lastPositionStatus: $lastPositionStatus ]" +
            " [ currentPositionStatus : $currentPositionStatus]")
        if (lastPositionStatus == currentPositionStatus) {
            return
        }
        when (currentPositionStatus) {
            LpfLiveinterconnect.LivePositionInfo.EMPTY -> {
                notifyOutSide(MicInfo(item, MicInfo.MicStatus.MIC_EMPTY))
            }
            LpfLiveinterconnect.LivePositionInfo.AUDIO_CLOSE -> {
                notifyOutSide(MicInfo(item, MicInfo.MicStatus.AUDIO_CLOSE))
            }
            LpfLiveinterconnect.LivePositionInfo.LOCKED -> {
                notifyOutSide(MicInfo(item, MicInfo.MicStatus.MIC_LOCKED))
            }
            LpfLiveinterconnect.LivePositionInfo.NORMAL -> {
                notifyOutSide(MicInfo(item, MicInfo.MicStatus.AUDIO_OPEN))
                notifyOutSide(MicInfo(item, MicInfo.MicStatus.MIC_UNLOCK))
            }
        }
    }

    /**
     *  处理麦位处于后台还是前台情况
     */
    private fun handlePositionStreamStatus(
        lastItem: LpfLiveinterconnect.LiveInterconnectInfo?,
        item: LpfLiveinterconnect.LiveInterconnectInfo
    ) {
        if (item.liveStatus != LpfLiveroomtemplate.LiveRoomInfo.LIVING) {
            ALog.i(TAG,
                "handlePositionStreamStatus 处理麦位处于前后台情况 position " + item.positionInfo.position +
                    " the liveStatus is not living")
            return
        }
        if (lastItem != null && item.clientStreamStatus == lastItem.clientStreamStatus) {
            ALog.i(TAG,
                ("handlePositionStreamStatus 处理麦位处于前后台情况 position " + item.positionInfo.position +
                    " the same clientStream"))
            return
        }
        ALog.i(TAG,
            ("handlePositionStreamStatus 处理麦位处于前后台情况 " +
                "position " + item.positionInfo.position +
                " clientStream " + item.clientStreamStatus))
        if (item.clientStreamStatus == LpfMedia.CSS_LIVE_COMMON) {
            notifyOutSide(MicInfo(item, MicInfo.MicStatus.MIC_FOREGROUND))
        }
        if (item.clientStreamStatus == LpfMedia.CSS_RESTEAT_BG) {
            notifyOutSide(MicInfo(item, MicInfo.MicStatus.MIC_BACKGROUND))
        }
    }

    /**
     * 处理麦位用户信息拓展字段通知
     *
     */
    private fun handlePositionMicUserInfoExtend(
        lastItem: LpfLiveinterconnect.LiveInterconnectInfo?,
        item: LpfLiveinterconnect.LiveInterconnectInfo
    ) {
        if (lastItem?.user == null && item.user != null) {
            ALog.i(TAG, "handlePositionMicInfoExtend 处理麦位拓展字段 [position: ${item.positionInfo?.position}]")
            notifyOutSide(MicInfo(item, MicInfo.MicStatus.MIC_USER_INFO_CHANGE))
            return
        }

        item.user?.let {
            if (!lastItem?.user?.toString().equals(it.toString())) {
                ALog.i(TAG, "handlePositionMicInfoExtend 通知麦位用户 拓展字段变化 ")
                notifyOutSide(MicInfo(item, MicInfo.MicStatus.MIC_USER_INFO_CHANGE))
            }
        }
    }

    private fun findLastMicInfoWithUid(targetUid: Long?): LpfLiveinterconnect.LiveInterconnectInfo? {
        mLastLinkMicInfos.forEach {
            if (it.value.user != null && it.value.user.uid == targetUid) {
                return it.value
            }
        }
        return null
    }

    /**
     *  通知封装的 房间信息
     *  @param roomInfo
     */
    private fun notifyRoomInfoStatusChange(roomInfo: LpfLiveroomtemplate.LiveRoomInfo) {
        var hasNotify = false
        //当前房间的sid
        if (mRoomInfoWrapper.sid != roomInfo.sid) {
            ALog.i(TAG,
                "notifyRoomInfoBroadcastChange " +
                    "[lastSid : ${mRoomInfoWrapper.sid}]" +
                    " [ newSid : ${roomInfo.sid}")
            mRoomInfoWrapper.sid = roomInfo.sid
            hasNotify = true
        }
        mRoomInfoWrapper.coverUrl = roomInfo.channelInfo?.uploadCoverUrl
        //主播开播状态
        if (mRoomInfoWrapper.anchorLiveStatus != roomInfo.anchorLiveStatus) {
            ALog.i(TAG,
                "notifyRoomInfoBroadcastChange " +
                    "[lastAnchorLiveStatus : ${mRoomInfoWrapper.anchorLiveStatus}]" +
                    " [ newAnchorLiveStatus : ${roomInfo.anchorLiveStatus}")
            mRoomInfoWrapper.anchorLiveStatus = roomInfo.anchorLiveStatus
            hasNotify = true
        }
        //主播开播 businessType
        if (mRoomInfoWrapper.businessType != roomInfo.businessType) {
            ALog.i(TAG,
                "notifyRoomInfoBroadcastChange " +
                    "[lastBusinessType : ${mRoomInfoWrapper.businessType}]" +
                    " [ newBusinessType : ${roomInfo.businessType}]")
            mRoomInfoWrapper.businessType = roomInfo.businessType
            hasNotify = true
        }
        //频道当前状态
        roomInfo.channelInfo?.let {
            if (mRoomInfoWrapper.channelClose != it.channelClose) {
                ALog.i(TAG,
                    "notifyRoomInfoBroadcastChange " +
                        "[lastChannelClose : ${mRoomInfoWrapper.channelClose}]" +
                        " [ newChannelClose : ${it.channelClose}]")
                mRoomInfoWrapper.channelClose = roomInfo.channelInfo.channelClose
            }
            if (!mRoomInfoWrapper.title.equals(it.title)) {
                //通知标题信息
                ALog.i(TAG,
                    "notifyRoomInfoBroadcastChange channelTitle : ${roomInfo.channelInfo.title}")
                mRoomInfoWrapper.title = roomInfo.channelInfo.title
                hasNotify = true
            }
        }
        //当前房间主播信息
        roomInfo.anchorUserInfo?.let {
            if (!isSameAnchorInfo(mRoomInfoWrapper.anchorInfo, it)) {
                ALog.i(TAG,
                    "notifyRoomInfoBroadcastChange " +
                        "[lastAnchorInfo : ${mRoomInfoWrapper.anchorInfo} ]" +
                        " [ newAnchorInfo : $it]")
                hasNotify = true
            }
            mRoomInfoWrapper.anchorInfo = roomInfo.anchorUserInfo
        }
        if (hasNotify) {
            for (listener in mRoomInfoListeners) {
                listener.onRoomInfoStatusChange(mRoomInfoWrapper)
            }
        }
    }

    private fun isSameAnchorInfo(
        lastAnchorInfo: LpfUser.UserInfo?, newAnchorInfo: LpfUser.UserInfo
    ): Boolean {
        if (lastAnchorInfo != null && lastAnchorInfo.uid == newAnchorInfo.uid) {
            return true
        }
        return false
    }

    private fun notifyOutSide(micInfo: MicInfo) {
        ALog.i(TAG, "notifyOutSide micInfo pos: ${micInfo.info.positionInfo.position} ; " +
            "uid: ${micInfo.info.user?.uid} ; status: ${micInfo.status} ; sid: ${micInfo.info.sid}")
        for (listener in mRoomInfoListeners) {
            listener.onRoomInfoChange(micInfo)
        }
    }

    /**
     * 获取 LiveRoomInfo By Sid
     * @param sid
     */
    fun getLiveRoomInfo(sid: Long?) {
        if (sid == null) {
            ALog.i(TAG, "getLiveRoomInfo sid == null")
            return
        }
        if (sid <= 0L) {
            ALog.i(TAG, "getLiveRoomInfo [sid : $sid] is invalied")
            return
        }
        ALog.i(TAG, "getLiveRoomInfo [sid : $sid]")
        mRepository.getLiveRoomInfo(sid,
            object : IMessageCallback<LpfLiveroomtemplate.GetLiveRoomInfoResp> {
                override fun onMessageSuccess(
                    response: MessageResponse<LpfLiveroomtemplate.GetLiveRoomInfoResp>
                ) {
                    val resp = response.message
                    ALog.i(TAG, "getLiveRoomInfo onMessageSuccess : [code : ${resp.code}  " +
                        " [ timeStamp : ${resp.timestamp}  [ lastTimeStamp : $lastTimeStamp] ")
                    if (lastTimeStamp > resp.timestamp) {
                        return
                    }
                    handleRoomInfoUpdate(resp.liveRoomInfo)
                    lastTimeStamp = resp.timestamp
                }

                override fun onMessageFail(
                    errorCode: ServiceFailResult,
                    ex: Exception?
                ) {
                    ALog.i(TAG,
                        "getLiveRoomInfo error reason = $errorCode")
                }

                override fun get(): LpfLiveroomtemplate.GetLiveRoomInfoResp {
                    return LpfLiveroomtemplate.GetLiveRoomInfoResp()
                }
            })
    }

    fun getLiveRoomInfoByUid(
        uid: Long, liveBzType: Int,
        callback: IDataCallback<LpfLiveinfo.LiveInfo>
    ) {
        mRepository.getLiveInfoByUidReq(uid, liveBzType,
            object : IMessageCallback<LpfLiveinfo.GetLiveInfoByUidResp> {
                override fun onMessageSuccess(
                    response: MessageResponse<LpfLiveinfo.GetLiveInfoByUidResp>
                ) {
                    val rsp = response.message
                    if (rsp.code == 0 && rsp.liveInfo != null) {
                        val info = rsp.liveInfo
                        ALog.i(TAG, "getLiveRoomInfoByUid success,liveInfo:$info")
                        callback.onDataLoaded(info)
                    } else {
                        ALog.i(TAG,
                            "getLiveRoomInfoByUid fail [ code : ${rsp.code} [message: ${rsp.message}]")
                        callback.onDataNotAvailable(rsp.code, rsp.message)
                    }
                }

                override fun onMessageFail(
                    errorCode: ServiceFailResult,
                    ex: Exception?
                ) {
                    ALog.e(TAG, "getLiveRoomInfoByUid fail", ex)
                }

                override fun get(): LpfLiveinfo.GetLiveInfoByUidResp {
                    return LpfLiveinfo.GetLiveInfoByUidResp()
                }
            })
    }

    fun getLiveStatusByUid(uid: Long): Boolean {
        for (item in mLastLinkMicInfos.values) {
            //如果以前的缓存数据 item 麦位是 有当前uid 则麦位状态是流打开
            if (item.user?.uid == uid) {
                ALog.i(TAG, "getLiveStatusByUid ${item.liveStatus}")
                when (item.liveStatus) {
                    LpfLiveroomtemplate.LiveRoomInfo.LIVE_END -> return false
                    LpfLiveroomtemplate.LiveRoomInfo.LIVING -> return true
                }
            }
        }
        ALog.i(TAG, "getLiveStatusByUid does not have")
        return false
    }

    @MessageBinding
    fun onBroadcastGroupEvent(event: ServiceBroadcastEvent?) { //组播
        if (event == null) {
            return
        }
        try {
            if (FUNC_UPDATE_LIVE_ROOM_INFO_BROADCAST == event.funcName) {
                val broadcast = LpfLiveroomtemplate.UpdateLiveRoomInfoBroadcast
                    .parseFrom(event.message)
                if (broadcast != null) {
                    ALog.i(TAG, "onBroadcastGroupEvent [ timeStamp : ${broadcast.timestamp}]  " +
                        "[lastTimeStamp : $lastTimeStamp] ")
                    if (lastTimeStamp > broadcast.timestamp) {
                        return
                    }
                    handleRoomInfoUpdate(broadcast.liveRoomInfo)
                    lastTimeStamp = broadcast.timestamp
                }
            }
        } catch (e: InvalidProtocolBufferNanoException) {
            e.printStackTrace()
        }
    }

    @MessageBinding
    fun appStatus(applicationStatus: ApplicationStatus) {
        ALog.i(TAG, "appStatus [status : ${applicationStatus.status}]")
        when (applicationStatus.status) {
            ApplicationStatus.Status.ONBACKGROUND -> {
                ALog.i(TAG, "appOnBackground")
            }
            ApplicationStatus.Status.ONFOREGROUND -> {
                ALog.i(TAG, "appOnForeground")
                mCommonViewModel.let {
                    getLiveRoomInfo(it?.sid)
                }
            }
        }
    }

    /**
     * 网络连接类型发生改变，当网络再次可用时，更新RoomInfo
     */
    override fun onNetworkTypeChanged(type: Int) {
        ALog.i(TAG, "onNetType: ${getNetTypeString(type)}")
        var lastInvalid =
            lastNetworkType == ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_DISCONNECTED ||
                lastNetworkType == ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_MOBILE_2G
        var curValid = type == ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_WIFI
            || type == ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_CABLE
            || type == ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_MOBILE_3G
            || type == ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_MOBILE_4G

        //判断上一次为无可用网络，且当前为可用网络状态时，重新请求一次room info数据
        if (lastInvalid && curValid) {
            ALog.i(TAG, "Update roomInfo with network valid")
            getLiveRoomInfo(mRoomInfoWrapper.sid)
        }
        lastNetworkType = type
    }

    private fun getNetTypeString(status: Int): String {
        return when (status) {
            ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_DISCONNECTED -> "DISCONNECTED"
            ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_CABLE -> "CABLE(有线网络)"
            ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_WIFI -> "WIFI"
            ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_MOBILE -> "MOBILE(未识别的移动网络)"
            ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_MOBILE_2G -> "2G"
            ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_MOBILE_3G -> "3G"
            ThunderRtcConstant.ThunderNetworkType.THUNDER_NETWORK_TYPE_MOBILE_4G -> "4G"
            else -> {
                "UNKNOWN($status)"
            }
        }
    }

    companion object {
        private val TAG = RoomInfoViewModel::class.java.simpleName
        private const val FUNC_UPDATE_LIVE_ROOM_INFO_BROADCAST = "updateLiveRoomInfoBroadcast"
        private fun isSamePerson(
            lastItem: LpfLiveinterconnect.LiveInterconnectInfo?,
            item: LpfLiveinterconnect.LiveInterconnectInfo?
        ): Boolean {
            return (lastItem != null && item != null
                && lastItem.sid != 0L
                && item.sid != 0L
                && lastItem.sid == item.sid
                && lastItem.user != null
                && item.user != null
                && item.user.uid != 0L
                && lastItem.user.uid != 0L
                && lastItem.user.uid == item.user.uid)
        }
    }
}
