package tv.athena.live.component.business.wath

import com.yy.liveplatform.proto.nano.LpfConfig
import com.yy.liveplatform.proto.nano.LpfLiveroomtemplateV2
import com.yy.liveplatform.proto.nano.LpfMedia
import tv.athena.live.api.wath.AudienceCDNStatus
import tv.athena.live.api.wath.AudienceLineStreamInfoListener
import tv.athena.live.api.wath.MediaProtocol
import tv.athena.live.api.wath.MediaType
import tv.athena.live.api.wath.bean.LineStreamInfo
import tv.athena.live.basesdk.config.BaseDataConfig
import tv.athena.live.component.business.activitybar.webview.util.JsonParser
import tv.athena.live.utils.ALog
import tv.athena.live.utils.TimeConsumingUtil
import tv.athena.live.utils.TimeConsumingUtil.END_BY_CALLBACK_AUDIENCE_MSG
import tv.athena.live.utils.TimeConsumingUtil.START_FROM_REGISTER_BROADCAST_BY_STREAM_ROOM_ID
import java.util.LinkedList
import java.util.concurrent.ConcurrentLinkedQueue
import kotlin.collections.sortedBy as sortedBy1

/**
 *  @author chenhaofeng  on 2020/3/2
 *  @description 观众系统（cdn播放器） 处理类
 */
class AudienceLineStreamHandle {
    private val TAG = "AudienceLineStreamHandle"
    private var mLineStreamInfoListener: ListenerBuilder? = null

    //观众系统流信息缓存
    private val mCacheAudienceLineStreamInfos = ConcurrentLinkedQueue<LineStreamInfo>()

    //推荐档位缓存
    private val mRecommendAudienceLineStreamInfos = ConcurrentLinkedQueue<LineStreamInfo>()

    private var mCacheAudienceCDNStatus: AudienceCDNStatus? = null

    init {
        ALog.i(TAG, "init")
    }

    fun onDestroy() {
        mLineStreamInfoListener = null
        mCacheAudienceLineStreamInfos.clear()
        ALog.i(TAG, "AudienceLineStreamHandle onDestroy")
    }

    fun setLineStreamInfoListener(listener: AudienceLineStreamInfoListener.() -> Unit) {
        this.mLineStreamInfoListener = ListenerBuilder().also(listener)
        ALog.i(TAG, "AudienceLineStreamHandle setLineStreamInfoListener")
    }

    fun handleRoomInfoV2(roomInfoV2: LpfLiveroomtemplateV2.LiveRoomInfoV2) {
        ALog.i(TAG, "handleRoomInfoV2 [ streamRoomId : ${roomInfoV2.channelInfo?.streamRoomId}  ; " +
            " audienceCdnStatus ${roomInfoV2.audienceCdnStatus} ]")
        when (roomInfoV2.audienceCdnStatus) {
            0 -> { // 代表不用等待了
                notifyAudienceCDNStatus(AudienceCDNStatus.END)
            }
            1 -> {
                notifyAudienceCDNStatus(AudienceCDNStatus.WAIT)
            }
            2 -> {
                notifyAudienceCDNStatus(AudienceCDNStatus.READY)
            }
        }

        val audienceStreamInfos = roomInfoV2.audienceLineStreamInfos
        val mNewAudienceStreamInfos = LinkedList<LineStreamInfo>()
        val selectionStrategies = BaseDataConfig.getStrategy()?.selectionStrategies?.associateBy(
            LpfConfig.AudienceStreamStrategy.SelectionStrategy::serviceProvider,
            LpfConfig.AudienceStreamStrategy.SelectionStrategy::mediaProtocols
        )?.apply {
            keys.forEach {
                ALog.i(TAG, "selectionStrategies [key: $it ] - [value : ${getValue(it)} ]")
            }
        }

        val selectionQuicStrategies = BaseDataConfig.getStrategy()?.selectionStrategies?.associateBy(
            LpfConfig.AudienceStreamStrategy.SelectionStrategy::serviceProvider,
            LpfConfig.AudienceStreamStrategy.SelectionStrategy::isSupportQuic
        )?.apply {
            keys.forEach {
                ALog.i(TAG, "selectionQuicStrategies [key: $it ] - [value : ${getValue(it)}")
            }
        }

        if (selectionStrategies == null) {
            //如果拿不到该数据，直接通知cdn 挂载走源流播放
            ALog.i(TAG, "handleRoomInfoV2 selectionStrategies == null")
            notifyAudienceCDNStatus(AudienceCDNStatus.END)
            return
        }
        audienceStreamInfos.filter {
            val serviceProvider = it.serviceProvider
            selectionStrategies.containsKey(serviceProvider)
        }.sortedBy1 {
            selectionStrategies.keys?.indexOf(it.serviceProvider) ?: 0
        }.apply {
            //如果过滤的数据是为空，并且cdn 返回状态是READY 状态也回调挂了
            if (isEmpty() && roomInfoV2.audienceCdnStatus == LpfLiveroomtemplateV2.ACS_READY) {
                ALog.i(TAG, "handleRoomInfoV2 filter size == 0, " +
                    "and audienceCdnStatus : ${roomInfoV2.audienceCdnStatus}")
                notifyAudienceCDNStatus(AudienceCDNStatus.END)
            }
        }.forEach { audienceLineStreamInfo ->
            val provider = audienceLineStreamInfo.serviceProvider
            val mediaProtocols = selectionStrategies!![provider]?.toList()
            var isSupportQuic = false
            selectionQuicStrategies?.let {
                isSupportQuic = it[provider] ?: false
            }
            audienceLineStreamInfo.audienceStreamInfos.forEach { audienceStreamInfo ->
                val joinStreamUids = audienceStreamInfo.joinStreamUids.toList()

                val streamRoomId = roomInfoV2.channelInfo?.streamRoomId
                audienceStreamInfo.streamInfos.filter { commonStreamInfo ->
                    val mediaProtocol = commonStreamInfo.mediaProtocol
                    mediaProtocols?.contains(mediaProtocol) == true
                }.forEach {
                    val streamUrls = it.streamUrls.toList()
                    val streamDefinition = it.streamDefinition
                    val mediaProtocol = when (it.mediaProtocol) {
                        LpfLiveroomtemplateV2.MP_THUNDER_BOLOT -> MediaProtocol.THUNDER_BOLOT
                        LpfLiveroomtemplateV2.MP_FLV -> MediaProtocol.FLV
                        LpfLiveroomtemplateV2.MP_RTMP -> MediaProtocol.RTMP
                        LpfLiveroomtemplateV2.MP_HLS -> MediaProtocol.HLS
                        else -> MediaProtocol.THUNDER_BOLOT
                    }

                    val mediaType = when (it.mediaType) {
                        LpfMedia.MT_VIDEO -> MediaType.MT_VIDEO
                        LpfMedia.MT_AUD -> MediaType.MT_AUDIO
                        LpfMedia.MT_AV -> MediaType.MT_AV
                        LpfMedia.MT_NONE -> MediaType.NONE
                        else -> MediaType.MT_AV
                    }
                    val lineStreamInfo = LineStreamInfo(provider,
                        streamRoomId,
                        joinStreamUids,
                        mediaType,
                        mediaProtocol,
                        streamUrls,
                        streamDefinition,
                        it.videoWidth,
                        it.videoHeight,
                        it.videoBitrate,
                        it.videoFrameRate, isSupportQuic = isSupportQuic, configResolution = it.configResolution)
                    mNewAudienceStreamInfos.add(lineStreamInfo)
                }
            }
        }

        //1. 先处理流离开
        val leaveLineStreamInfos = handleLeave(mNewAudienceStreamInfos, mCacheAudienceLineStreamInfos)
        //2. 在处理流到达
        val arriveLineStreamInfos = handleArrive(mNewAudienceStreamInfos, mCacheAudienceLineStreamInfos)

        val reCommendLineStreamInfos = handleReCommendLineStreams(mNewAudienceStreamInfos)
        //先重置缓存流信息情况，避免在回调函数中取到旧的缓存数据
        mCacheAudienceLineStreamInfos.clear()
        mCacheAudienceLineStreamInfos.addAll(mNewAudienceStreamInfos)
        val roomInfoJson = JsonParser.toJson(mCacheAudienceLineStreamInfos)
        val isSame = JsonParser.toJson(mLastCacheAudienceLineStreamInfo) == roomInfoJson   //耗时在2ms
        if (isSame) {
            ALog.i(TAG, "handleRoomInfoV2 isSame")
        } else {
            mLastCacheAudienceLineStreamInfo = mCacheAudienceLineStreamInfos
            ALog.i(TAG, "handleRoomInfoV2 newCaches $mCacheAudienceLineStreamInfos")
        }
        //先回调离开
        if (leaveLineStreamInfos.isNotEmpty()) {
            mLineStreamInfoListener?.mLineStreamInfoLeaveAction?.invoke(leaveLineStreamInfos)
        }
        //再回调新增加
        if (arriveLineStreamInfos.isNotEmpty()) {
            mLineStreamInfoListener?.mLineStreamInfoArriveAction?.invoke(arriveLineStreamInfos)
        }
        //回调档位推荐
        reCommendLineStreamInfos?.let {
            mLineStreamInfoListener?.mRecommendLineStreamInfoAction?.invoke(it)
        }
    }

    private var mLastCacheAudienceLineStreamInfo: ConcurrentLinkedQueue<LineStreamInfo>? = null

    private fun notifyAudienceCDNStatus(status: AudienceCDNStatus) {
        ALog.i(TAG, "notifyAudienceCDNStatus " +
            "[listener : ${mLineStreamInfoListener?.mAudienceCDNStatus}] " +
            "[status :$status]")
        mCacheAudienceCDNStatus = status
        mLineStreamInfoListener?.mAudienceCDNStatus?.invoke(status)
    }

    /**
     * 处理推荐档位逻辑
     */
    private fun handleReCommendLineStreams(mNewAudienceStreamInfos: LinkedList<LineStreamInfo>): List<LineStreamInfo>? {
        if (mNewAudienceStreamInfos.isEmpty()) {
            ALog.i(TAG, "handleReCommendLineStreams mNewAudienceStreamInfos isEmpty")
            return null
        }
        val serviceProvider = mNewAudienceStreamInfos.first.serviceProvider
        val mediaProtocol = mNewAudienceStreamInfos.first.mediaProtocol
        val reCommendLineStreamInfos = mNewAudienceStreamInfos.filter {
            it.serviceProvider == serviceProvider && it.mediaProtocol == mediaProtocol
        }
        ALog.i(TAG, "handleReCommendLineStreams " +
            "${mLineStreamInfoListener?.mRecommendLineStreamInfoAction} -- size: " +
            "${reCommendLineStreamInfos.size}")
        mRecommendAudienceLineStreamInfos.clear()
        mRecommendAudienceLineStreamInfos.addAll(reCommendLineStreamInfos)
        TimeConsumingUtil.end(START_FROM_REGISTER_BROADCAST_BY_STREAM_ROOM_ID,
            END_BY_CALLBACK_AUDIENCE_MSG)
        return reCommendLineStreamInfos
    }

    /**
     * 处理流到达
     */
    private fun handleArrive(
        newLineStreamInfos: List<LineStreamInfo>,
        cacheLineStreamInfos: ConcurrentLinkedQueue<LineStreamInfo>
    ): List<LineStreamInfo> {
        val arriveLineStreamInfos = newLineStreamInfos.filter {
            !cacheLineStreamInfos.contains(it)
        }
        ALog.i(TAG, "handleArrive ${mLineStreamInfoListener?.mLineStreamInfoLeaveAction} -- size: " +
            "${arriveLineStreamInfos.size}")
        return arriveLineStreamInfos
    }

    /**
     *  处理流离开
     */
    private fun handleLeave(
        newLineStreamInfos: List<LineStreamInfo>,
        cacheLineStreamInfos: ConcurrentLinkedQueue<LineStreamInfo>
    ): List<LineStreamInfo> {
        val leaveLineStreamInfos = cacheLineStreamInfos.filter {
            !newLineStreamInfos.contains(it)
        }
        ALog.i(TAG, "handleLeave ${mLineStreamInfoListener?.mLineStreamInfoLeaveAction} -- size: " +
            "${leaveLineStreamInfos.size}")
        return leaveLineStreamInfos
    }

    inner class ListenerBuilder : AudienceLineStreamInfoListener {
        //流到达 action
        internal var mLineStreamInfoArriveAction: ((lineStreamInfos: List<LineStreamInfo>) -> Unit)? = null

        //流离开 action
        internal var mLineStreamInfoLeaveAction: ((lineStreamInfos: List<LineStreamInfo>) -> Unit)? = null

        //推荐档位 action
        internal var mRecommendLineStreamInfoAction: ((lineStreamInfos: List<LineStreamInfo>) -> Unit)? = null

        //观众系统 cdn 系统状态
        internal var mAudienceCDNStatus: ((status: AudienceCDNStatus) -> Unit)? = null

        override fun onAudienceLineStreamInfoArrive(action: (lineStreamInfos: List<LineStreamInfo>) -> Unit) {
            this.mLineStreamInfoArriveAction = action
        }

        override fun onAudienceLineStreamLeave(action: (lineStreamInfos: List<LineStreamInfo>) -> Unit) {
            this.mLineStreamInfoLeaveAction = action
        }

        override fun onRecommendLineStream(action: (lineStreamInfos: List<LineStreamInfo>) -> Unit) {
            this.mRecommendLineStreamInfoAction = action
        }

        override fun onAudienceCDNStatus(action: (status: AudienceCDNStatus) -> Unit) {
            this.mAudienceCDNStatus = action
        }
    }

    fun getCdnLineStreamInfo(): List<LineStreamInfo> {
        ALog.i(TAG, "getCdnLineStreamInfo size ${mCacheAudienceLineStreamInfos.size}")
        return mCacheAudienceLineStreamInfos.toList()
    }

    fun getAudienceCDNStatus(): AudienceCDNStatus? {
        ALog.i(TAG, "getAudienceCDNStatus $mCacheAudienceCDNStatus")
        return mCacheAudienceCDNStatus
    }
}