package tv.athena.live.component.business.report;

import android.Manifest;
import android.content.Context;
import android.content.pm.PackageManager;
import android.os.Environment;
import android.support.v4.content.ContextCompat;
import android.text.TextUtils;

import com.yy.liveplatform.proto.nano.LpfAudit;

import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;

import tv.athena.live.api.report.ReportApi;
import tv.athena.live.base.arch.IComponentViewModel;
import tv.athena.live.base.manager.ComponentContext;
import tv.athena.live.basesdk.thunderblotwrapper.AbscThunderEventListener;
import tv.athena.live.basesdk.liveroom.LivePlatformSdk;
import tv.athena.live.basesdk.thunderblotwrapper.ThunderHandle;
import tv.athena.live.component.business.report.cache.AudioCacheService;
import tv.athena.live.component.business.report.repository.ReportRepository;
import tv.athena.live.utils.ALog;
import tv.athena.live.utils.UploadUtil;
import tv.athena.service.api.IMessageCallback;
import tv.athena.service.api.MessageResponse;
import tv.athena.service.api.ServiceFailResult;
import tv.athena.util.common.FileUtils;

/**
 * 负责举报服务
 *
 * @author : liaojidong
 * @e-mail : liaojidong@joyy.sg
 * @date : 2019/10/21 14:02
 * @desc :
 */
public class ReportService extends AbscThunderEventListener implements IComponentViewModel {
    private static final String TAG = ReportService.class.getSimpleName();
    private static final String DIR_NAME = ".audioOutput";
    private ReportRepository mRepository = new ReportRepository();
    private ComponentContext mComponentContext;
    private ThunderHandle mThunderHandle;

    public ReportService(ComponentContext context) {
        this.mComponentContext = context;
        this.mThunderHandle = context.getThunderHandle();
    }

    private String getRoomId() {
        if (mComponentContext != null && mComponentContext.getCommonViewModel() != null) {
            return mComponentContext.getCommonViewModel().getSid() + "";
        }
        return "";
    }

    public void init() {
        ALog.i(TAG, "init");
        /*ThunderHandle.Companion.getInstance().registerThunderEventListener(this);
        ThunderHandle.Companion.getInstance().enableAudioDataIndication(true);*/
        mThunderHandle.registerThunderEventListener(this);
        mThunderHandle.enableAudioDataIndication(true);
    }

    @Override
    public void onJoinRoomSuccess(String room, String uid, int elapsed) {
        ALog.i(TAG, "onJoinRoomSuccess");
        super.onJoinRoomSuccess(room, uid, elapsed);
    }

    @Override
    public void onAudioPlayData(byte[] data, long cpt, long pts, String uid, long duration) {
        //打印太频繁
        /*ALog.i(TAG, "onAudioPlayData cpt = "
                + cpt + " pts = " + pts + " uid = " + uid + " duration = " + duration);*/
        super.onAudioPlayData(data, cpt, pts, uid, duration);
        if (TextUtils.isEmpty(getRoomId())) {
            ALog.i(TAG, "onAudioPlayData roomId is empty! ");
            return;
        }
        AudioCacheService.INSTANCE.cacheAudioData(getRoomId(), Long.parseLong(uid), data, duration);
    }

    public void report(Context context,
                       final long reportedUid,
                       final int illegalType,
                       final int auditType,
                       final ReportApi.OnReportListener listener,
                       final String ext) {
        if (LpfAudit.LIVE_REPORT == auditType) {
            //直接发送举报通知
            sendReport("", reportedUid, illegalType, auditType, listener, ext);
        } else {
            //1.获取到被举报人到音频数据
            byte[] audioData = AudioCacheService.INSTANCE.getCachedUserAudio(getRoomId(),
                reportedUid, System.currentTimeMillis() - (20 * 1000));
            if (audioData != null && audioData.length > 0) {
                //2.写入到磁盘
                final String savePath = writeAudioToFile(audioData);
                if (savePath == null) {
                    ALog.i(TAG, "failed to save audio file");
                    if (listener != null) {
                        listener.onFailed(-1, "failed to save audio file");
                    }
                    return;
                }

                //3.把音频上传到OSS服务器
                UploadUtil.getInstance().upload(context, savePath, new UploadUtil.UploadCallBack() {
                    @Override
                    public void onSuccess(String url) {
                        ALog.i(TAG, "upload success url = " + url);
                        //4.把上传后得到到URL发送到中台服务器
                        sendReport(url, reportedUid, illegalType, auditType, listener, ext);
                        //删除临时文件
                        FileUtils.delete(savePath);
                    }

                    @Override
                    public void onFail() {
                        if (listener != null) {
                            listener.onFailed(-1, "failed to upload");
                        }
                        ALog.i(TAG, "upload failed url");
                        //删除临时文件
                        FileUtils.delete(savePath);
                    }
                });
            } else {
                ALog.w(TAG, "audioData is empty"); //如果音频为空，
                sendReport("", reportedUid, illegalType, LpfAudit.LIVE_REPORT, listener, ext);
            }
        }
    }

    private String writeAudioToFile(byte[] audioData) {
        if (audioData == null
            || audioData.length == 0
            || !hasWriteExternalStoragePermission()) {
            return null;
        }
        final String saveDir = Environment.getExternalStorageDirectory().getAbsolutePath()
            + File.separator + DIR_NAME;
        FileUtils.createOrExistsDir(saveDir);
        FileOutputStream fos = null;
        try {
            final String savePath = saveDir + File.separator + "report_" + System.currentTimeMillis();
            fos = new FileOutputStream(savePath);
            fos.write(audioData);
            fos.flush();
            return savePath;
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (fos != null) {
                try {
                    fos.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return null;
    }

    private boolean hasWriteExternalStoragePermission() {
        return ContextCompat.checkSelfPermission(
            LivePlatformSdk.getInstance().getApplication(),
            Manifest.permission.WRITE_EXTERNAL_STORAGE) == PackageManager.PERMISSION_GRANTED;
    }

    private void sendReport(String content, long reportedUid,
                            int illegalType, int auditType,
                            ReportApi.OnReportListener listener,
                            String ext) {
        mRepository.reportContent(
            Long.parseLong(getRoomId()),
            reportedUid,
            mComponentContext.getCommonViewModel().getMyUid(),
            LpfAudit.AUDIO_LIVE_REPORT,
            auditType,
            content,
            illegalType,
            auditType == LpfAudit.LIVE_REPORT ? false : true,
            ext,
            new IMessageCallback<LpfAudit.AuditReportResp>() {
                @Override
                public void onMessageSuccess(MessageResponse<LpfAudit.AuditReportResp> messageResponse) {
                    int code = messageResponse != null && messageResponse.getMessage() != null
                        ? messageResponse.getMessage().code : -1;
                    ALog.i(TAG, "senReport success reason code = " + code);
                    if (code != 0) {
                        if (listener != null) {
                            listener.onFailed(code, messageResponse.getMessage().message);
                        }
                    } else {
                        if (listener != null) {
                            listener.onSuccess(
                                messageResponse != null ? messageResponse.getMessage().data : "");
                        }
                    }
                }

                @Override
                public void onMessageFail(ServiceFailResult serviceFailResult, Exception e) {
                    ALog.i(TAG, "senReport failed reason = "
                        + (serviceFailResult != null ? serviceFailResult.toString() : ""));
                    if (listener != null) {
                        listener.onFailed(
                            serviceFailResult != null ? serviceFailResult.getResultCode() : -1,
                            serviceFailResult != null ? serviceFailResult.getDescription() : "");
                    }
                }

                @NotNull
                @Override
                public LpfAudit.AuditReportResp get() {
                    return new LpfAudit.AuditReportResp();
                }
            });
    }

    private synchronized byte[] getCachedUserAudio(long uid, long expiredTime) {
        return AudioCacheService.INSTANCE.getCachedUserAudio(getRoomId(), uid, expiredTime);
    }

    public void close() {
        ALog.i(TAG, "close");
        //ThunderHandle.Companion.getInstance().unRegisterRtcEventListener(this);
        mThunderHandle.unRegisterRtcEventListener(this);
        AudioCacheService.INSTANCE.clearRoomCache(getRoomId());
    }
}
