package tv.athena.live.component.business.broadcasting

import android.text.TextUtils
import android.view.WindowManager
import com.yy.liveplatform.proto.nano.LpfMedia
import tv.athena.live.api.IDataCallback
import tv.athena.live.api.broadcast.bean.StartLiveParams
import tv.athena.live.api.broadcast.bean.StartLiveStatus
import tv.athena.live.api.roominfo.RoomInfoApi
import tv.athena.live.base.manager.ComponentContext
import tv.athena.live.basesdk.thunderblotwrapper.AbscThunderEventListener
import tv.athena.live.basesdk.thunderblotwrapper.ThunderHandle
import tv.athena.live.basesdk.thunderblotwrapper.ThunderHandleManager
import tv.athena.live.channel.ChannelRepositiy
import tv.athena.live.component.business.broadcasting.accessibility.time.ITimerApi
import tv.athena.live.component.business.broadcasting.accessibility.time.ObserTimeListener
import tv.athena.live.component.business.broadcasting.accessibility.time.TimerApiImpl
import tv.athena.live.component.business.broadcasting.repository.BroadcastRepository
import tv.athena.live.utils.ALog
import tv.athena.live.utils.LiveRoomBeatHeartUtils
import tv.athena.live.utils.ServiceUtils
import tv.athena.live.utils.ThreadSafeMutableLiveData
import tv.athena.live.utils.UploadUtil
import tv.athena.service.api.MessageResponse
import tv.athena.service.api.ServiceFailResult

/**
 *    create by chenhaofeng 2019-08-13
 *    具体SDK 接口的调用方法，参考 ：
 *    https://www.sunclouds.com/cloud/v2/developer/doc.htm?serviceId=102&typeCode=API_DOC&title=Android&version=2.1.5&parentId=
 */
class BroadcastNormalImpl() :
    AbscThunderEventListener(),
    IBroadcast {

    companion object {
        const val TAG = "BroadcastViewModel"
        const val MIN_PREVIEWWIDTH = 480
        const val MIN_PREVIEWHEIGHT = 640
        const val MAX_PREVIEWWIDTH = 720
        const val MAX_PREVIEWHEIGHT = 1280
    }

    private var mComponentContext: ComponentContext? = null
    private var mBroadcastComponent: BroadcastComponent? = null
    private var mToken: String? = null
    private var mUploadCoverUrl: String? = null
    private var mThunderHandle: ThunderHandle? = null

    private val iTimerApi: ITimerApi = TimerApiImpl()

    private val mLiveTime = ThreadSafeMutableLiveData<Long>()
    private val mLiveTimeStr = ThreadSafeMutableLiveData<String>()

    //记录是否调用 startLive 接口
    private var hasStartLive = false

    // 上麦推流标志
    private var hasTakeMic = false

    // 记录rtmp 推流地址
    private val mRtmpUrlList = ArrayList<String>()
    private var mStartLiveParams: StartLiveParams? = null

    init {
        //ProcessLifecycleOwner.get().lifecycle.addObserver(this)
    }

    override fun onDestroy() {
        ALog.i(TAG, "onDestroy")
        mThunderHandle?.unRegisterRtcEventListener(this)
        this.iTimerApi.stopTimer()
        this.hasStartLive = false
        this.hasTakeMic = false
    }

    override fun onLeave() {
        mThunderHandle?.unRegisterRtcEventListener(this)
    }

    override fun setComponent(component: BroadcastComponent) {
        ALog.i(TAG, "setComponent($component)")
        this.mBroadcastComponent = component
        this.mComponentContext = mBroadcastComponent!!.componentContext
        mThunderHandle = mComponentContext?.thunderHandle
        mThunderHandle?.registerThunderEventListener(this@BroadcastNormalImpl)
    }

    /**
     *  开播并且加入指定房间
     */
    @Deprecated(message = "请使用StartLiveParams参数的")
    override fun startLive(
        title: String,
        coverUrl: String,
        extend: String?,
        liveBzType: Int,
        audioSourceType: Int,
        callBack: IDataCallback<Int>?
    ) {
        val startLiveParams = StartLiveParams(title = title,
            coverUrl = coverUrl,
            extend = extend,
            liveBzType = liveBzType,
            audioSourceType = audioSourceType,
            callback = callBack)
        startLive(startLiveParams)
    }

    override fun startLive(startLiveParams: StartLiveParams) {
        val startLiveReq = LpfMedia.StartLiveReq()
        startLiveReq.title = startLiveParams.title
        startLiveReq.uploadCoverUrl = if (startLiveParams.coverUrl.isNotEmpty())
            startLiveParams.coverUrl else mUploadCoverUrl ?: ""
        startLiveReq.liveBzType = startLiveParams.liveBzType
        startLiveReq.extend = startLiveParams.extend
        startLiveParams.targetSid?.let {
            startLiveReq.sid = it.toLong()
        }
        startLiveReq.interconnectPosition = startLiveParams.interconnectPosition
        ALog.i(TAG, "startLive ($startLiveReq)")

        mThunderHandle?.setAudioSourceType(startLiveParams.audioSourceType)
        mStartLiveParams = startLiveParams
        BroadcastRepository.startLive(startLiveReq,
            object : ServiceUtils.SvcResultCusRetryCallBack<LpfMedia.StartLiveResp>() {
                override fun get(): LpfMedia.StartLiveResp {
                    return LpfMedia.StartLiveResp()
                }

                override fun onMessageFail(errorCode: ServiceFailResult, ex: Exception?) {
                    ALog.i(TAG, "startLive onMessageFail errorCode = $errorCode , $ex")
                    changeStartLiveStatus(errorCode.getResultCode())
                }

                override fun onMessageSuccess(response: MessageResponse<LpfMedia.StartLiveResp>) {
                    ALog.i(TAG,
                        "startLive onMessageSuccess ${response.message.code} , ${response.message}")
                    if (response.message.code == 0) {
                        mToken = response.message.token
                        //设置ComponentContext Ext 中 sid
                        mComponentContext?.commonViewModel?.sid = response.message.sid
                        val mUid = mComponentContext?.commonViewModel?.myUid.toString()
                        if (response.message.sid <= 0) {
                            ALog.i(TAG, "startLive [sid : ${response.message.sid}]")
                            return
                        }
                        if (TextUtils.isEmpty(mToken)) {
                            ALog.i(TAG, "startLive mToken isEmpty")
                            return
                        }
                        if (TextUtils.isEmpty(mUid)) {
                            ALog.i(TAG, "startLive mUid isEmpty")
                            return
                        }
                        hasStartLive = true
                        if (response.message.pushCDNUrl.isNotEmpty()) {
                            response.message.pushCDNUrl.forEach {
                                mRtmpUrlList.add(it)
                            }
                        }
                        if (!ThunderHandleManager.hasJoinRoom) {
                            ALog.i(TAG, "startLive 主播 joinRoom 房间")
                            mThunderHandle?.joinRoom(mToken!!.toByteArray(),
                                response.message.sid.toString(),
                                mUid)
                        } else {
                            //进行推流
                            startPublishStream()
                            //进行rtmp 推流
                            startPublishRtmpUrl()
                            changeStartLiveStatus(0, response.message.extend)
                        }
                        mThunderHandle?.registerThunderEventListener(this@BroadcastNormalImpl)
                        mThunderHandle?.updateToken(mToken!!.toByteArray())
                        mBroadcastComponent?.componentContext?.activity?.window?.setFlags(
                            WindowManager.LayoutParams
                                .FLAG_KEEP_SCREEN_ON,
                            WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON)
                        ChannelRepositiy.registerChannelBroadcast(response.message.sid)
                    } else {
                        changeStartLiveStatus(response.message.code, response.message.extend)
                    }
                }
            })
    }

    private fun startPublishRtmpUrl() {
        mRtmpUrlList.forEach {
            val value = mThunderHandle?.addPublishOriginStreamUrl(it)
            try {
                val playUrl = it.replace("rtmp", "http")
                    .replace("push", "pull")
                    .plus(".m3u8")
                ALog.i(TAG, "startPublishRtmpUrl [value : $value ] ; [$it ; 播放地址：$playUrl]")
            } catch (e: java.lang.Exception) {
                e.printStackTrace()
            }
        }
    }

    private fun removePublishRtmpUrl() {
        mRtmpUrlList.forEach {
            val value = mThunderHandle?.removePublishOriginStreamUrl(it)
            ALog.i(TAG, "removePublishRtmpUrl $it ; value = $value")
        }
        mRtmpUrlList.clear()
    }

    /**
     *  上传封面
     *  @param imagePath 封面地址
     */
    override fun uploadCoverUrl(imagePath: String, callback: IDataCallback<String>?) {
        val context = mComponentContext?.activity?.applicationContext
        context?.let {
            ALog.i(TAG, "uploadCoverUrl [$imagePath]")
            UploadUtil.getInstance().upload(it, imagePath, object : UploadUtil.UploadCallBack {
                override fun onSuccess(url: String?) {
                    mUploadCoverUrl = url
                    if (url != null) {
                        callback?.onDataLoaded(url)
                    } else {
                        callback?.onDataNotAvailable(-1, "uploadCoverUrl fail")
                    }
                }

                override fun onFail() {
                    ALog.i(TAG, "uploadCoverUrl onFail()")
                    callback?.onDataNotAvailable(-1, "uploadCoverUrl fail")
                }
            })
        }
    }

    /**
     * 停止开播
     * 1.请求服务端， 结束开播
     * 2.断开直播心跳
     * 3.调用SDK leaveRoom
     */
    override fun stopLive(extend: String?, callback: IDataCallback<LpfMedia.EndLiveResp>?) {
        ALog.i(TAG, "endLiveReq ")
        val endLiveReq = LpfMedia.EndLiveReq()
        BroadcastRepository.endLiveReq(endLiveReq, object :
            ServiceUtils.SvcResultCusRetryCallBack<LpfMedia.EndLiveResp>() {
            override fun get(): LpfMedia.EndLiveResp {
                return LpfMedia.EndLiveResp()
            }

            override fun onMessageFail(errorCode: ServiceFailResult, ex: Exception?) {
                ALog.i(TAG, "endLiveReq onMessageFail " +
                    "errorCode = $errorCode, $ex")
                callback?.onDataNotAvailable(errorCode.getResultCode(), errorCode.description)
            }

            override fun onMessageSuccess(response: MessageResponse<LpfMedia.EndLiveResp>) {
                ALog.i(TAG,
                    "endLiveReq onMessageSuccess " +
                        "${response.message.code} , ${response.message}")
                when (response.message.code) {
                    0 -> {
                        callback?.onDataLoaded(response.message)
                    }
                    else -> {
                        callback?.onDataNotAvailable(response.message.code, response.descption)
                    }
                }
            }
        })
        mComponentContext?.commonViewModel?.sid?.let {
            ChannelRepositiy.unRegisterChannelBroadcast(it)
        }
        iTimerApi.stopTimer()
        mThunderHandle?.unRegisterRtcEventListener(this)
        removePublishRtmpUrl()
        mThunderHandle?.leaveRoom()
        //iAudioFilePlayer.destroyAudioFilePlayer()
        //重置改标纪
        //hasJoinRoom = false
        //结束推流心跳，并尝试恢复频道心跳
        LiveRoomBeatHeartUtils.tryResetChannelHeartbeat()
        hasStartLive = false
    }

    /**
     * 以下 code 具体可参考 lpf.media  StartLiveResp 定义的返回code
     */
    private fun changeStartLiveStatus(code: Int, extend: String? = "") {
        ALog.i(TAG, "changeStartLiveStatus ($code, $extend)")
        when (code) {
            0 -> mStartLiveParams?.callback?.onDataLoaded(code)
            else -> mStartLiveParams?.callback?.onDataNotAvailable(code, extend ?: "")
        }
        mStartLiveParams = null
    }

    /**
     * @param liveBzType 开播类型，中台业务服务端配置
     * @param callBack 回调
     */
    override fun checkLivePermission(
        liveBzType: Int,
        callBack: IDataCallback<LpfMedia.CheckLivePermissionResp>
    ) {
        val req = LpfMedia.CheckLivePermissionReq()
        req.liveBzType = liveBzType
        ALog.i(TAG, "checkLivePermission $req")
        BroadcastRepository.checkLivePermission(req,
            object : ServiceUtils.SvcResultCusRetryCallBack<LpfMedia.CheckLivePermissionResp>() {
                override fun get(): LpfMedia.CheckLivePermissionResp {
                    return LpfMedia.CheckLivePermissionResp()
                }

                override fun onMessageFail(errorCode: ServiceFailResult, ex: java.lang.Exception?) {
                    ALog.i(TAG,
                        "checkLivePermission onMessageFail " +
                            "${errorCode.getResultCode()} - ${errorCode.description}")
                    callBack.onDataNotAvailable(errorCode.getResultCode(), errorCode.description)
                }

                override fun onMessageSuccess(
                    response: MessageResponse<LpfMedia.CheckLivePermissionResp>
                ) {
                    if (response.resultCode == 0) {
                        callBack.onDataLoaded(response.message)
                    } else {
                        ALog.i(TAG, "checkLivePermission code = ${response.resultCode}")
                        callBack.onDataNotAvailable(response.resultCode, response.descption)
                    }
                }
            })
    }


    /**
     * ==============sdk 回调 ============
     */

    /**
     * 开启推流， 只有在 joinRoom 成功后才能调用
     */
    override fun startPublishStream() {
        mThunderHandle?.let {
            ALog.i(TAG, "startPublishStream()")
            //设置 模式 文件 + 麦克风
            //mThunderHandle?.setAudioSourceType(THUNDER_PUBLISH_MODE_MIX)
            it.stopLocalAudioStream(false)
        }

        //开始推流心跳
        val sid = mBroadcastComponent?.componentContext?.commonViewModel?.sid
        val uid = mBroadcastComponent?.componentContext?.commonViewModel?.myUid
        if (sid == null) {
            ALog.e(TAG, "startLiveHeartbeat error, current sid is invalid")
            return
        }
        if (uid == null) {
            ALog.e(TAG, "startLiveHeartbeat error, current uid is invalid")
            return
        }
        LiveRoomBeatHeartUtils.startLiveHeartbeat(
            topSid = sid,
            uid = uid)
    }

    override fun stopPublishStream() {
        mThunderHandle?.let {
            ALog.i(TAG, "stopPublishStream()")
            it.stopLocalAudioStream(true)
        }

        //结束推流心跳
        LiveRoomBeatHeartUtils.tryResetChannelHeartbeat()
    }

    /**
     * 进入房间成功回调
     * @param room 频道信息
     * @param uid
     * @param elapsed 花费时间
     *
     */
    override fun onJoinRoomSuccess(room: String, uid: String, elapsed: Int) {
        super.onJoinRoomSuccess(room, uid, elapsed)
        ALog.i(TAG, "onJoinRoomSuccess [room : $room]," +
            " [uid: $uid ], " +
            "[elapsed : $elapsed] , " +
            "[hasStartLive : $hasStartLive]")
        if (hasStartLive) {
            changeStartLiveStatus(StartLiveStatus.START_LIVE_SUCCESS)
            startPublishStream()
            startPublishRtmpUrl()
            startTimeAndLiveHeart()
        }
        room.let {
            ALog.i(TAG, "joinRoom success post RoomStatusEvent ")
            mComponentContext?.commonViewModel?.streamRoomId = room
        }
    }

    /**
     * 开启倒计时与开播心跳
     */
    private fun startTimeAndLiveHeart() {
        iTimerApi.startTimer()
        iTimerApi.addObserTimeListener(object :
            ObserTimeListener {
            override fun changeLiveTime(second: Long) {
                mLiveTime.value = second
            }

            override fun changeLiveTimeStr(timeStr: String) {
                mLiveTimeStr.value = timeStr
            }
        })
    }

    /**
     * ------------------同频道
     */

    override fun takeMic(isEnable: Boolean) {
        ALog.i(TAG, "takeMic ($isEnable) ")
        if (isEnable) {
            hasTakeMic = isEnable
            //开始推流
            startPublishStream()
            //开启心跳
            startTimeAndLiveHeart()
        } else {
            hasTakeMic = isEnable
            //停止推流
            mThunderHandle?.stopLocalAudioStream(true)
            //停止心跳
            iTimerApi.stopTimer()
        }
    }

    override fun changeLiveMediaType(mediaType: Int, callBack: IDataCallback<LpfMedia.UpdateMediaTypeResp>) {
        val sid = mComponentContext?.commonViewModel?.sid
        if (sid == null || sid <= 0) {
            ALog.i(BroadcastViewModel.TAG, "changeLiveMediaType() sid is invalid")
            callBack.onDataNotAvailable(-1, "sid is invalid")
            return
        }
        val req = LpfMedia.UpdateMediaTypeReq()
        req.sid = sid
        req.mediaType = mediaType
        val lastMediaType = getCurrentMediaType()
        BroadcastRepository.changeLiveMediaType(req,
            object : ServiceUtils.SvcResultCusRetryCallBack<LpfMedia.UpdateMediaTypeResp>() {
                override fun get(): LpfMedia.UpdateMediaTypeResp {
                    return LpfMedia.UpdateMediaTypeResp()
                }

                override fun onMessageFail(errorCode: ServiceFailResult, ex: java.lang.Exception?) {
                    ALog.i(BroadcastViewModel.TAG,
                        "changeLiveMediaType onMessageFail " +
                            "${errorCode.getResultCode()} - ${errorCode.description} ${ex?.message}")
                    callBack.onDataNotAvailable(errorCode.getResultCode(), errorCode.description)
                }

                override fun onMessageSuccess(response: MessageResponse<LpfMedia.UpdateMediaTypeResp>) {
                    ALog.i(BroadcastViewModel.TAG, "changeLiveMediaType.onMessageSuccess ${response.message}")
                    if (response.message.code == 0) {
                        dispatchMediaTypeChanged(lastMediaType, mediaType)
                        callBack.onDataLoaded(response.message)
                    } else {
                        callBack.onDataNotAvailable(response.message.code, response.message.toString())
                    }
                }
            })
        ALog.i(BroadcastViewModel.TAG, "changeLiveMediaType $req")
    }

    private fun getCurrentMediaType(): Int {
        mComponentContext?.componentManager?.getOtherComponentApi(RoomInfoApi::class.java)?.micInfos?.let {
            it.entries.forEach { info ->
                if (mComponentContext?.commonViewModel?.myUid == info.value?.user?.uid) {
                    return info.value.mediaType
                }
            }
        }
        return LpfMedia.MT_NONE
    }

    private fun dispatchMediaTypeChanged(oldMediaType: Int, newMediaType: Int) {
        ALog.i(BroadcastViewModel.TAG, "dispatchMediaTypeChanged  old=${mediaTypeString(oldMediaType)} " +
            " new=${mediaTypeString(newMediaType)}")
        if (newMediaType == LpfMedia.MT_NONE) {
            ALog.w(BroadcastViewModel.TAG, "None mediaType is invalid")
            return
        }

        //纯音频
        if (newMediaType == LpfMedia.MT_AUD) {
            when (oldMediaType) {
                LpfMedia.MT_NONE -> { //之前是音视频都关闭，现在要打开音频
                    mThunderHandle?.stopLocalAudioStream(false)
                }
                LpfMedia.MT_AUD -> { //之前是纯音频，现在不需要操作
                    //Ignore
                }
                LpfMedia.MT_VIDEO -> { //之前是纯视频，现在要关闭视频，打开音频
                    mThunderHandle?.stopLocalAudioStream(false)
                }
                LpfMedia.MT_AV -> { //之前是音视频，现在需要关闭视频
                }
            }
            return
        }

        //纯视频
        if (newMediaType == LpfMedia.MT_VIDEO) {
            when (oldMediaType) {
                LpfMedia.MT_NONE -> { //之前是音视频都关闭，现在需要打开视频
                }
                LpfMedia.MT_AUD -> { //之前是纯音频，现在需要关闭音频，打开视频
                    mThunderHandle?.stopLocalAudioStream(true)
                }
                LpfMedia.MT_VIDEO -> { //之前是纯视频，现在不需要操作
                    //Ignore
                }
                LpfMedia.MT_AV -> { //之前是音视频，现在要关闭音频
                    mThunderHandle?.stopLocalAudioStream(true)
                }
            }
            return
        }

        //音视频
        if (newMediaType == LpfMedia.MT_AV) {
            when (oldMediaType) {
                LpfMedia.MT_NONE -> { //之前是音视频都关闭，现在要打开音视频
                    mThunderHandle?.stopLocalAudioStream(false)
                }
                LpfMedia.MT_AUD -> { //之前是纯音频，现在要打开视频
                }
                LpfMedia.MT_VIDEO -> { //之前是纯视频，现在要打开音频
                    mThunderHandle?.stopLocalAudioStream(false)
                }
                LpfMedia.MT_AV -> { //之前是音视频，现在不需要操作
                    //Ignore
                }
            }
            return
        }
    }

    private fun mediaTypeString(mediaType: Int): String {
        return when (mediaType) {
            LpfMedia.MT_NONE -> "MT_NONE"
            LpfMedia.MT_AUD -> "MT_AUD"
            LpfMedia.MT_VIDEO -> "MT_VIDEO"
            LpfMedia.MT_AV -> "MT_AV"
            else -> mediaType.toString()
        }
    }

    override fun changeLiveRoomType(liveBzType: Int, callBack: IDataCallback<LpfMedia.ChangeLiveRoomTypeResp>) {
        val sid = mComponentContext?.commonViewModel?.sid
        if (sid == null || sid <= 0) {
            ALog.i(BroadcastViewModel.TAG, "changeLiveRoomType() sid is invalid")
            callBack.onDataNotAvailable(-1, "sid is invalid")

            return
        }
        val req = LpfMedia.ChangeLiveRoomTypeReq()
        req.sid = sid
        req.targetLiveBzType = liveBzType
        req.positionStretchType = 2
        BroadcastRepository.changeLiveRoomType(req,
            object : ServiceUtils.SvcResultCusRetryCallBack<LpfMedia.ChangeLiveRoomTypeResp>() {
                override fun get(): LpfMedia.ChangeLiveRoomTypeResp {
                    return LpfMedia.ChangeLiveRoomTypeResp()
                }

                override fun onMessageFail(errorCode: ServiceFailResult, ex: java.lang.Exception?) {
                    ALog.i(BroadcastViewModel.TAG,
                        "changeLiveRoomType onMessageFail " +
                            "${errorCode.getResultCode()} - ${errorCode.description} ${ex?.message}")
                    callBack.onDataNotAvailable(errorCode.getResultCode(), errorCode.description)
                }

                override fun onMessageSuccess(response: MessageResponse<LpfMedia.ChangeLiveRoomTypeResp>) {
                    ALog.i(BroadcastViewModel.TAG, "changeLiveRoomType.onMessageSuccess ${response.message}")
                    if (response.message.code == 0) {
                        callBack.onDataLoaded(response.message)
                    } else {
                        callBack.onDataNotAvailable(response.message.code, response.message.toString())
                    }
                }
            })
        ALog.i(BroadcastViewModel.TAG, "changeLiveRoomType $req")
    }
}