package tv.athena.live.component.business.activitybar;

import android.arch.lifecycle.LifecycleOwner;
import android.arch.lifecycle.MutableLiveData;
import android.arch.lifecycle.Observer;
import android.text.TextUtils;

import com.yy.liveplatform.proto.nano.LpfActivity;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.ArrayList;
import java.util.List;

import tv.athena.core.sly.Sly;
import tv.athena.live.api.activitybar.ActivityBarApi;
import tv.athena.live.api.activitybar.bean.ActivityBarConfig;
import tv.athena.live.api.activitybar.bean.BroadcastData;
import tv.athena.live.api.activitybar.bean.EntryData;
import tv.athena.live.base.arch.IComponentApi;
import tv.athena.live.base.manager.CommonViewModel;
import tv.athena.live.base.manager.ComponentContext;
import tv.athena.live.component.business.activitybar.repository.ActivityBarServiceImpl;
import tv.athena.live.component.business.activitybar.repository.IActivityBarService;
import tv.athena.live.utils.ALog;
import tv.athena.live.utils.ThreadSafeMutableLiveData;
import tv.athena.service.api.IMessageCallback;
import tv.athena.service.api.MessageResponse;
import tv.athena.service.api.ServiceFailResult;

/**
 * @author baiting
 * https://git.duowan.com/server/hamo/liveplatform-base/blob/master/
 * 后端接口概览与说明.md#客户端模块-后端接口调用说明
 */
class ActivityBarApiImpl implements ActivityBarApi {

    private static final String TAG = "ActivityBarApiImpl";
    private ActivityBarView activityBarView;
    private IActivityBarService repository = new ActivityBarServiceImpl();
    private String mEntryUrl;
    private ComponentContext context;
    private MutableLiveData<Boolean> mEntryShow = new ThreadSafeMutableLiveData<Boolean>();
    private MutableLiveData<BroadcastData> mBroadcastData = new ThreadSafeMutableLiveData<>();
    private EntryData data;
    private BroadcastData broadcastData;
    private List<Observer<BroadcastData>> observers = new ArrayList<>();

    @Override
    public Class<? extends IComponentApi> getApiKey() {
        return ActivityBarApi.class;
    }

    private void observeId(ActivityBarConfig config) {
        if (config != null && activityBarView != null) {
            activityBarView.setId(config.getId());
        }
    }

    @Override
    public void show(ActivityBarConfig config) {
        observeId(config);
        if (activityBarView != null && activityBarView.isReady()) {
            repository.queryActWindow(context.getCommonViewModel().getSid(), false,
                new IMessageCallback<LpfActivity.IfActivityWindowPresentResp>() {
                    @Override
                    public void onMessageSuccess(@NotNull MessageResponse
                        <LpfActivity.IfActivityWindowPresentResp> messageResponse) {
                        try {
                            LpfActivity.IfActivityWindowPresentResp message = messageResponse.getMessage();
                            ALog.e(TAG, message.toString());
                            mEntryUrl = message.showUrl;
                            if (data == null) {
                                data = new EntryData();
                            }
                            data.setShowUrl(mEntryUrl);
                            data.setActivityInfos(message.activityInfos);
                            data.setIfPresent(message.ifPresent);

                            mEntryShow.setValue(message.ifPresent);
                            String url = getEntryUrl();
                            if (message.code == 0 && url != null) {
                                activityBarView.showActivity(config, url, data);
                            }
                        } catch (Exception e) {
                            e.printStackTrace();
                        }
                    }

                    @Override
                    public void onMessageFail(@NotNull ServiceFailResult serviceFailResult, @Nullable Exception e) {
                        data = null;
                    }

                    @NotNull
                    @Override
                    public LpfActivity.IfActivityWindowPresentResp get() {
                        return new LpfActivity.IfActivityWindowPresentResp();
                    }
                });

        }
    }

    @Override
    public EntryData getEntryData() {
        return data;
    }

    public String getEntryUrl() {
        try {
            String url = mEntryUrl;
            CommonViewModel commonData = context.getCommonViewModel();
            if (!TextUtils.isEmpty(url)) {
                // {房间sid}&ownerId={主播/房主id}&isLivingAnchor={是否是主播端(0 不是，1 是)}&userId={用户id}
                String tempUrl = "sid=" + commonData.getSid() + "&ownerId=" + commonData.getRoomOwnerUid() +
                    "&isLivingAnchor=0&userId=" + commonData.getMyUid();
                if (url.contains("?")) {
                    return url + "&" + tempUrl;
                } else {
                    return url + "?" + tempUrl;
                }
            }
            return url;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }

    @Override
    public void showPopView(String url, String configUIJson) {
        if (activityBarView != null) {
            activityBarView.showPopView(url, configUIJson);
        }
    }

    @Override
    public void closePopView() {
        if (activityBarView != null) {
            activityBarView.closePopView();
        }
    }

    @Override
    public void observeBroadcastData(LifecycleOwner owner, Observer<BroadcastData> observer) {
        observers.add(observer);
        mBroadcastData.observe(owner, observer);
    }

    @Override
    public void removeObserver(Observer<BroadcastData> observer) {
        observers.remove(observer);
        mBroadcastData.removeObserver(observer);
    }

    public void clearObserver() {
        if (observers != null && observers.size() > 0) {
            for (Observer<BroadcastData> obs : observers) {
                mBroadcastData.removeObserver(obs);
            }
            observers.clear();
        }

    }

    @Override
    public void loadJavascript(String js) {
        if (activityBarView != null) {
            activityBarView.loadJavaScript(js);
        }
    }

    public void bindActivityBarView(ActivityBarView activityBarView) {
        this.activityBarView = activityBarView;
    }

    public void saveBroadcastData(String url, String data) {
        if (TextUtils.isEmpty(data)) {
            return;
        }
        if (broadcastData == null) {
            broadcastData = new BroadcastData();
        }
        broadcastData.setUrl(url);
        broadcastData.setData(data);
        Sly.Companion.postMessage(broadcastData);
        mBroadcastData.setValue(broadcastData);
    }

    public void setContext(ComponentContext mComponentContext) {
        this.context = mComponentContext;
    }
}
