package tv.athena.live.component.business.link_mic

import com.yy.liveplatform.proto.nano.LpfLiveinterconnect
import tv.athena.annotation.MessageBinding
import tv.athena.annotation.ServiceRegister
import tv.athena.core.sly.Sly
import tv.athena.live.api.IDataCallback
import tv.athena.live.api.link_mic.ILinkMicService
import tv.athena.live.utils.ServiceUtils
import tv.athena.live.component.business.link_mic.repository.LinkMicRepository
import tv.athena.live.utils.ALog
import tv.athena.service.api.MessageResponse
import tv.athena.service.api.ServiceFailResult
import tv.athena.service.api.event.ServiceUnicastEvent

@ServiceRegister(serviceInterface = ILinkMicService::class)
class LinkMicServiceImpl : ILinkMicService {
    private val inviteLiveInterconnectUnicast = "inviteLiveInterconnectUnicast"
    private val inviteLiveInterconnectResultUnicast = "inviteLiveInterconnectResultUnicast"
    private val liveInterconnectEndUnicast = "liveInterconnectEndUnicast"
    private val liveInterconnectUpdateUnicast = "applyConnectUpdateUnicast"
    private val mLinkMicUnicastListeners = ArrayList<ILinkMicService.LinkMicUnicastListener>()
    private val mLinkMicRepository = LinkMicRepository()

    companion object {
        private const val TAG = "LinkMicServiceImpl"
    }

    init {
        Sly.subscribe(this)
    }

    override fun addLinkMicUnicastListener(listener: ILinkMicService.LinkMicUnicastListener) {
        synchronized(mLinkMicUnicastListeners) {
            if (!mLinkMicUnicastListeners.contains(listener)) {
                mLinkMicUnicastListeners.add(listener)
            }
        }
    }

    override fun removeLinkMicUnicastListener(listener: ILinkMicService.LinkMicUnicastListener) {
        synchronized(mLinkMicUnicastListeners) {
            if (mLinkMicUnicastListeners.contains(listener)) {
                mLinkMicUnicastListeners.remove(listener)
            }
        }
    }

    @MessageBinding
    fun registerUnicast(event: ServiceUnicastEvent) {
        //业务自己处理
        when (event.funcName) {
            inviteLiveInterconnectUnicast -> {
                // 处理被邀请单播
                handleInviteLiveInterconnectUnicast(event.message)
            }
            inviteLiveInterconnectResultUnicast -> {
                // 处理回复单播
                handInviteLiveInterconnectResultUnicast(event.message)
            }
            liveInterconnectEndUnicast -> {
                // 处理断开单播
                handleLiveInterconnectEndUnicast(event.message)
            }
            liveInterconnectUpdateUnicast -> {
                // 处理申请连麦信息变更单播
                handleLiveInterconnectUpdateUnicast(event.message)
            }
            else -> {
                // todo
            }
        }
    }

    /**
     * 被邀请方 ， 处理邀请单播下发
     */
    private fun handleInviteLiveInterconnectUnicast(message: ByteArray) {
        val inviteLiveInterconnectUnicast =
            LpfLiveinterconnect.InviteLiveInterconnectUnicast.parseFrom(message)
        //todo 修改连麦状态 被邀请状态
        ALog.i(TAG,
            "handleInviteLiveInterconnectUnicast [inviteLiveInterconnectUnicast : $inviteLiveInterconnectUnicast]")
        for (listener in mLinkMicUnicastListeners) {
            listener.onInviteLiveInterconnect(inviteLiveInterconnectUnicast)
        }
    }

    /**
     *  处理回复单播
     */
    private fun handInviteLiveInterconnectResultUnicast(message: ByteArray) {
        val inviteconnectResultUnicast =
            LpfLiveinterconnect.InviteLiveInterconnectResultUnicast.parseFrom(message)
        ALog.i(TAG,
            "handInviteLiveInterconnectResultUnicast [inviteconnectResultUnicast : $inviteconnectResultUnicast]")
        for (listener in mLinkMicUnicastListeners) {
            listener.onInviteLiveInterconnectResult(inviteconnectResultUnicast)
        }
    }

    /**
     * 处理连麦断开单播
     */
    private fun handleLiveInterconnectEndUnicast(message: ByteArray) {
        val liveInterconnectEndUnicast =
            LpfLiveinterconnect.LiveInterconnectEndUnicast.parseFrom(message)
        //todo 修改连麦状态 断开连麦
        ALog.i(TAG,
            "handleLiveInterconnectEndUnicast [liveInterconnectEndUnicast : $liveInterconnectEndUnicast]")
        for (listener in mLinkMicUnicastListeners) {
            listener.onLiveInterconnectEnd(liveInterconnectEndUnicast)
        }
    }

    /**
     * 处理申请连麦列表信息变更单播
     */
    private fun handleLiveInterconnectUpdateUnicast(message: ByteArray) {
        val unicastEvent =
            LpfLiveinterconnect.ApplyConnectUpdateUnicast.parseFrom(message)
        ALog.i(TAG, "handleLiveInterconnectUpdateUnicast [$unicastEvent]")
        for (listener in mLinkMicUnicastListeners) {
            listener.onApplyConnectListUpdated(unicastEvent)
        }
    }

    override fun refuseInviteLiveInterconnectReq(
        sid: Long,
        replyInviteUserInfo: LpfLiveinterconnect.LiveInterconnectInfo,
        businessType: Int,
        position: Int, isAutoReject: Boolean,
        callBack: IDataCallback<LpfLiveinterconnect.ReplyInviteLiveInterconnectResp>?
    ) {
        if (sid <= 0) {
            ALog.i(LinkMicViewModel.TAG, "refuseInviteLiveInterconnectReq [sid : $sid]")
            callBack?.onDataNotAvailable(LinkMicViewModel.IN_VALIED_SID, "当前频道号值 异常")
            return
        }
        val req = LpfLiveinterconnect.ReplyInviteLiveInterconnectReq()
        req.sid = sid
        req.isAccept = false
        req.replyInviteUserInfo = replyInviteUserInfo
        req.businessType = businessType
        req.position = position
        req.isAutoReject = isAutoReject //是否时自动取消， 倒计时结束调用
        ALog.i(LinkMicViewModel.TAG, "refuseInviteLiveInterconnectReq [req : $req]")
        mLinkMicRepository.replyInviteLiveInterConnectReq(req,
            object : ServiceUtils.SvcResultCusRetryCallBack<LpfLiveinterconnect
            .ReplyInviteLiveInterconnectResp>() {
                override fun get(): LpfLiveinterconnect.ReplyInviteLiveInterconnectResp {
                    return LpfLiveinterconnect.ReplyInviteLiveInterconnectResp()
                }

                override fun onMessageFail(errorCode: ServiceFailResult, ex: java.lang.Exception?) {
                    //todo 回复连麦失败处理
                    ALog.i(LinkMicViewModel.TAG,
                        "refuseInviteLiveInterconnectReq onMessageFail [errorCode : ${errorCode.getResultCode()}]")
                    callBack?.onDataNotAvailable(errorCode.getResultCode(), errorCode.description)
                }

                override fun onMessageSuccess(
                    response: MessageResponse<LpfLiveinterconnect.ReplyInviteLiveInterconnectResp>
                ) {
                    ALog.i(LinkMicViewModel.TAG,
                        "refuseInviteLiveInterconnectReq onMessageSuccess [code : ${response.message.code}]")
                    when (response.message.code) {
                        0 -> {
                            callBack?.onDataLoaded(response.message)
                        }
                        //被邀请方 接受邀请连麦 失败各种问题
                        else -> {
                            callBack?.onDataNotAvailable(response.message.code, response.descption)
                        }
                    }
                }
            })
    }
}