package tv.athena.live.component.business.audience;

import android.arch.lifecycle.LifecycleOwner;
import android.arch.lifecycle.MutableLiveData;
import android.arch.lifecycle.Observer;

import com.yy.liveplatform.proto.nano.LpfChannel;
import com.yy.liveplatform.proto.nano.LpfUser;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Arrays;
import java.util.List;

import tv.athena.annotation.MessageBinding;
import tv.athena.core.sly.Sly;
import tv.athena.live.utils.ThreadSafeMutableLiveData;
import tv.athena.live.api.audience.AudienceApi;
import tv.athena.live.base.arch.IComponentViewModel;
import tv.athena.live.base.manager.ComponentContext;
import tv.athena.live.component.business.audience.repository.AudienceRepository;
import tv.athena.live.utils.ALog;
import tv.athena.service.api.IMessageCallback;
import tv.athena.service.api.MessageResponse;
import tv.athena.service.api.ServiceFailResult;
import tv.athena.service.api.event.ServiceBroadcastEvent;

/**
 * liaojidong
 *
 * @Data 2019/8/21
 */
public class AudienceViewModel implements IComponentViewModel {
    private static final String TAG = AudienceViewModel.class.getSimpleName();
    private static final int DEFAULT_PAGE_SIZE = 50;
    private ComponentContext mContext;
    private AudienceRepository mRepository;
    private MutableLiveData<List<LpfUser.UserInfo>> mUsers = new ThreadSafeMutableLiveData<>();
    private MutableLiveData<String> mUserCount = new ThreadSafeMutableLiveData<>();
    private Observer<Long> mSidObserver = new Observer<Long>() {
        @Override
        public void onChanged(@Nullable Long sid) {
            getChannelUsers(sid, DEFAULT_PAGE_SIZE, 0);
        }
    };

    public AudienceViewModel(ComponentContext context) {
        this.mContext = context;
        mRepository = new AudienceRepository();
        Sly.Companion.subscribe(this);
        mContext.getCommonViewModel().observeSidForever(mSidObserver);
    }

    private void getChannelUsers(long sid, int pageSize, int offset) {
        mRepository.getChannelUsers(sid, pageSize, offset, new IMessageCallback<LpfChannel.GetChannelUsersResp>() {
            @Override
            public void onMessageSuccess(@NotNull MessageResponse<LpfChannel.GetChannelUsersResp> messageResponse) {
                if (messageResponse == null || messageResponse.getMessage() == null) {
                    return;
                }
                mUsers.setValue(Arrays.asList(messageResponse.getMessage().users));
            }

            @Override
            public void onMessageFail(@NotNull ServiceFailResult serviceFailResult, @Nullable Exception e) {
                ALog.w(TAG, serviceFailResult.toString());
            }

            @NotNull
            @Override
            public LpfChannel.GetChannelUsersResp get() {
                return new LpfChannel.GetChannelUsersResp();
            }
        });
    }

    public void getChannelUsers(int pageSize,
                                int offset,
                                AudienceApi.OnChannelQueryListener listener) {
        final long sid = mContext.getCommonViewModel().getSid();
        mRepository.getChannelUsers(sid, pageSize, offset, new IMessageCallback<LpfChannel.GetChannelUsersResp>() {
            @Override
            public void onMessageSuccess(@NotNull MessageResponse<LpfChannel.GetChannelUsersResp> messageResponse) {
                if (messageResponse == null || messageResponse.getMessage() == null) {
                    return;
                }
                if (listener != null) {
                    listener.onSuccess(Arrays.asList(messageResponse.getMessage().users));
                }
            }

            @Override
            public void onMessageFail(@NotNull ServiceFailResult serviceFailResult, @Nullable Exception e) {
                ALog.w(TAG, serviceFailResult.toString());
                if (listener != null) {
                    listener.onFail(serviceFailResult.getResultCode(), serviceFailResult.toString());
                }
            }

            @NotNull
            @Override
            public LpfChannel.GetChannelUsersResp get() {
                return new LpfChannel.GetChannelUsersResp();
            }
        });
    }

    public void observeChannelUsers(LifecycleOwner owner, Observer<List<LpfUser.UserInfo>> observer) {
        mUsers.observe(owner, observer);
    }

    public void observeUserCount(LifecycleOwner owner, Observer<String> observer) {
        mUserCount.observe(owner, observer);
    }

    public void onDestroy() {
        Sly.Companion.unSubscribe(this);
        mContext.getCommonViewModel().removeSidObserver(mSidObserver);
    }

    @MessageBinding
    public void onBroadcastGroupEvent(ServiceBroadcastEvent event) {
        if (event == null) {
            return;
        }
        if (mRepository.handleUserNoticeBroadcast(mUsers, mUserCount, event)) {
            return;
        }
    }

}
